/*	exp2f.c
 *
 *	Base 2 exponential function
 *
 *
 *
 * SYNOPSIS:
 *
 * float x, y, exp2f();
 *
 * y = exp2f( x );
 *
 *
 *
 * DESCRIPTION:
 *
 * Returns 2 raised to the x power.
 *
 * Range reduction is accomplished by separating the argument
 * into an integer k and fraction f such that
 *     x    k  f
 *    2  = 2  2.
 *
 * A polynomial approximates 2**x in the basic range [-0.5, 0.5].
 *
 *
 * ACCURACY:
 *
 *			Relative error:
 * arithmetic	domain	   # trials	 peak	      rms
 *    IEEE     -127,+127    100000	1.7e-7	    2.8e-8
 *
 *
 * See exp.c for comments on error amplification.
 *
 *
 * ERROR MESSAGES:
 *
 *   message	     condition	    value returned
 * exp underflow    x < -MAXL2	      0.0
 * exp overflow     x > MAXL2	      MAXNUMF
 *
 * For IEEE arithmetic, MAXL2 = 127.
 */

// Cephes Math Library Release 2.2:  June, 1992
// Copyright 1984, 1987, 1988, 1992 by Stephen L. Moshier
// Direct inquiries to 30 Frost Street, Cambridge, MA 02140
//
// Modified for DJGPP/GCC by KB Williams,
// kbwms@aol.com, November 2003
//

#include <errno.h>
#include <fdlibml.h>
#include <fenv.h>
#include <float.h>

static float P[] = {
    1.535336188319500E-004,
    1.339887440266574E-003,
    9.618437357674640E-003,
    5.550332471162809E-002,
    2.402264791363012E-001,
    6.931472028550421E-001
};

static float polevlf(float, float *, int);

static float
polevlf(float xx, float *coef, int N)
{
    float   ans, x;
    float  *p;
    int     i;

    x = xx;
    p = coef;
    ans = *p++;

/*
for( i=0; i<N; i++ )
	ans = ans * x  +  *p++;
*/

    i = N;
    do
	ans = ans * x + *p++;
    while (--i);

    return (ans);
}

// -----------------------------------------
// exp2f - float Base 2 exponential function
// -----------------------------------------
float
exp2f(float xx)
{
    float   x, px;
    int     i0;

    x = xx;
    if (isnanf(x))
    {
	return x;
    }
    if (x == 0.0f)
    {
	return (1.0f);
    }
    if (x >= FLT_MAX_EXP)
    {
	if (!isinff(x))
	{
	    x = HUGE_VALF;
	    __math_set_errno(ERANGE);
	    __fp_raise_except(FE_OVERFLOW);
	}
    	return x;
    }
    if (x < (FLT_MIN_EXP))
    {
    	if (isinff(x))
	{
	    return 0.0f;
	}
	else
	{
    	    __math_set_errno(ERANGE);
	    __fp_raise_except(FE_UNDERFLOW);
	}
    }

/* separate into integer and fractional parts */
    px = floorf(x);
    i0 = px;
    x = x - px;

    if (x > 0.5)
    {
	i0 += 1;
	x -= 1.0f;
    }

/* rational approximation
 * exp2(x) = 1.0 +  xP(x)
 */
    px = 1.0 + x * polevlf(x, P, 5);

/* scale by power of 2 */
    {
    	int errno_saved = errno;
    	px = ldexpf(px, i0);
	__math_set_errno(errno_saved);
    }
    return (px);
}
