/*
 * dllink - dynamic linking system
 * Version:  0.1
 *
 * Copyright (C) 2005  Daniel Borca   All Rights Reserved.
 *
 * dllink is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * dllink is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNU Make; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
 * This code is loosely based on Stonewheel 0.0.8 by Burton Radons
 * (loth@pacificcoast.net), repackaged by Peter Wang (tjaden@psynet.net)
 */


#include <elf.h>
#include <crt0/list.h>
#include <crt0/internal.h>


#ifndef NULL
#define NULL		((void *)0)
#endif
#define SNULL		((void *)-1)


#define alloca		__builtin_alloca


#define RTLD_DEFAULT	((void *) 0)
#define RTLD_LAZY	0x001
#define RTLD_NOW	0x002
#define RTLD_GLOBAL	0x100
#define RTLD_LOCAL	0


#define SUB_VADDR(x)	if (x >= (int)vaddr) x = x - vaddr
#define ROUND_UP(x)	((x + 15) & ~15)
#define ELF_ERROR()	goto fail
#define CHECK_STACK()


#define LOADABLE(psh)	(psh.sh_flags & SHF_ALLOC)
#define TYP_RESIDENT	(1 << 0)
#define TYP_STRINGS	(1 << 1)


#ifdef DEBUG
void __attribute__((format(printf, 1, 2))) dprintf (const char *fmt, ...);
#define DLOG(x) dprintf x
#else
#define DLOG(x)
#endif


int dos_open (const char *path);
int dos_seek (int f, int offset);
int dos_read (int handle, void *buffer, unsigned int count);
int dos_close (int f);
void *mysbrk (int delta);
void elf_load_error (const char *objname);
void elf_link_error (const char *objname, const char *symname);


static stone_object *do_load (const char *filename, stone_object *objects);


unsigned __elf_vaddr (int f, int infile, Elf32_Ehdr *hdr);
stone_object *__elf_scan_symbol (const char *name, stone_sym **ret, stone_object *obj, stone_object *objects);
int __elf_resolve (int report, stone_object *obj, stone_object *objects);
int __elf_link (stone_object *objects);
void *__elf_load (int file, int offs, void *code, const char *src, stone_object *objects);


/**
 * Compare two strings.
 *
 * \param s1 first string
 * \param s2 second string
 *
 * \return same as strcmp
 *
 * \note this code runs unrelocated
 */
static int
str_cmp (const char *s1, const char *s2)
{
    while (*s1 == *s2) {
	if (*s1 == 0)
	    return 0;
	s1++;
	s2++;
    }
    return *(unsigned const char *)s1 - *(unsigned const char *)(s2);
}


/**
 * Compare two strings (except at-at suffix).
 *
 * \param s1 first string
 * \param s2 second string
 *
 * \return same as strcmp
 *
 * \note this code runs unrelocated
 */
static int
str_cmp_at (const char *s1, const char *s2)
{
    while (*s1 == *s2) {
	if (*s1 == 0)
	    return 0;
	s1++;
	s2++;
	if (*s1 == 0 && *s2 == '@' && *(s2 + 1) == '@')
	    return 0;
    }
    return *(unsigned const char *)s1 - *(unsigned const char *)(s2);
}


/**
 * Clear memory buffer.
 *
 * \param p buffer to be zeroed
 * \param l length of buffer
 *
 * \note this code runs unrelocated
 */
static void
b_zero (void *p, int l)
{
    int i;
    for (i = 0; i < l; i++) {
	((char *)p)[i] = 0;
    }
}


/**
 * Read data from file.
 *
 * \param ptr buffer to read data to
 * \param size number of bytes to be read
 * \param f DOS file handle
 * \param offset offset in file to read data from
 *
 * \return nonzero if success, else 0
 *
 * \note this code runs unrelocated
 */
static int
f_read (void *ptr, int size, int f, long offset)
{
    if (dos_seek(f, offset) != 0) {
	return 0;
    }
    return (dos_read(f, ptr, size) == size);
}


/**
 * Read ELF section.
 *
 * \param file DOS file handle
 * \param dat array of pointers for all sections
 * \param idx ELF section number
 * \param sh ELF section header
 * \param infile ELF offset in executable (equivalent to stub size)
 *
 * \return address of section data
 *
 * \note this code runs unrelocated
 */
static char *
readsection (int file, char **dat, int idx, Elf32_Shdr *sh, int infile)
{
    if (dat[idx] == SNULL) {
	dat[idx] = mysbrk(sh[idx].sh_size);
    }
    if (dat[idx] != SNULL) {
	if (!f_read(dat[idx], sh[idx].sh_size, file, infile + sh[idx].sh_offset)) {
	    return SNULL;
	}
    }
    return dat[idx];
}


/**
 * Calculate VADDR for ELF executables.
 *
 * \param f DOS file handle
 * \param infile ELF offset in executable (equivalent to stub size)
 * \param hdr ELF header
 *
 * \return VADDR, or zero if ELF is not EXEC
 *
 * \note this code runs unrelocated
 */
unsigned
__elf_vaddr (int f, int infile, Elf32_Ehdr *hdr)
{
    int c;
    Elf32_Phdr *ph;
    unsigned vaddr = 0;

    if (hdr->e_type == ET_EXEC) {
	ph = alloca(sizeof(Elf32_Phdr) * hdr->e_phnum);
	CHECK_STACK();
	if (f_read(ph, sizeof(Elf32_Phdr) * hdr->e_phnum, f, infile + hdr->e_phoff)) {
	    vaddr = -1U;
	    for (c = 0; c < hdr->e_phnum; c++) {
		if (ph[c].p_type == PT_LOAD) {
		    if (vaddr > ph[c].p_vaddr) {
			vaddr = ph[c].p_vaddr;
		    }
		}
	    }
	    vaddr &= ~0xFFFU;	/* XXX page size assumed 4kB */
	}
    }

    return vaddr;
}


/**
 * Load ELF.
 *
 * \param f DOS file handle
 * \param infile ELF offset in executable (equivalent to stub size)
 * \param code address at which the stub organized *BITS sections
 * \param src name of the executable
 * \param objects linked list of all objects
 *
 * \return loaded object 
 *
 * \note this code runs unrelocated
 */
static stone_object *
elf_load (int f, int infile, char *code, const char *src, stone_object *objects)
{
    stone_object *obj;
    Elf32_Ehdr hdr;
    Elf32_Shdr *sh;
    char **dat, *typ;
    int i, c, old;
    int nsym, nrel, nsyn, ndep, len;
    char *base, *strings;
    char *old_break, *new_break;
    unsigned vaddr;

    if (!f_read(&hdr, sizeof(hdr), f, infile + 0)) {
	return NULL;
    }
    if (hdr.e_ident[EI_MAG0] != ELFMAG0 ||
	hdr.e_ident[EI_MAG1] != ELFMAG1 ||
	hdr.e_ident[EI_MAG2] != ELFMAG2 ||
	hdr.e_ident[EI_MAG3] != ELFMAG3) {
	return NULL;
    }
    if (hdr.e_ident[EI_CLASS] != ELFCLASS32 ||
	hdr.e_ident[EI_DATA] != ELFDATA2LSB ||
	hdr.e_ident[EI_VERSION] != EV_CURRENT ||
	hdr.e_machine != EM_386 ||
	hdr.e_shentsize != sizeof(Elf32_Shdr)) {
	return NULL;
    }
    if (hdr.e_type != ET_EXEC &&
	hdr.e_type != ET_DYN) {
	return NULL;
    }
    sh = alloca(hdr.e_shnum * sizeof(Elf32_Shdr));
    CHECK_STACK();
    if (!f_read(sh, sizeof(Elf32_Shdr) * hdr.e_shnum, f, infile + hdr.e_shoff)) {
	return NULL;
    }
    dat = alloca(hdr.e_shnum * sizeof(char *));
    CHECK_STACK();
    for (c = 0; c < hdr.e_shnum; c++) {
	dat[c] = SNULL;
    }
    typ = alloca(hdr.e_shnum * sizeof(char));
    CHECK_STACK();
    for (c = 0; c < hdr.e_shnum; c++) {
	typ[c] = 0;
    }

    vaddr = __elf_vaddr(f, infile, &hdr);
    DLOG(("VADDR %s: %x\n", src, vaddr));

    /* calculate sizes */
    nsym = nrel = nsyn = ndep = len = 0;
    strings = 0x0;
    for (c = 0; c < hdr.e_shnum; c++) {
	if (sh[c].sh_type == SHT_REL) {
	    if (sh[c].sh_entsize == sizeof(Elf32_Rel) &&
		/* XXX sh_info weirdness */
		(sh[c].sh_info == 0 || LOADABLE(sh[sh[c].sh_info]))) {
		nrel += sh[c].sh_size / sizeof(Elf32_Rel);
	    } else {
		sh[c].sh_type = SHT_NULL;
	    }
	}
	if (LOADABLE(sh[c])) {
	    sh[c].sh_addr -= vaddr;
	    if (len < (int)sh[c].sh_addr + (int)sh[c].sh_size) {
		len = sh[c].sh_addr + sh[c].sh_size;
		dat[c] = (char *)sh[c].sh_addr;
		typ[c] = TYP_RESIDENT;
	    }
	}
    }
    for (c = 0; c < hdr.e_shnum; c++) {
	if (sh[c].sh_type == SHT_SYMTAB) {
	    /* SHT_SYMTAB is only needed by ET_EXEC */
	    if (sh[c].sh_entsize == sizeof(Elf32_Sym) && hdr.e_type == ET_EXEC) {
		/* XXX Attention: can't have multiple SYMTAB/DYNSYM */
		nsym += sh[c].sh_size / sizeof(Elf32_Sym);
		if (!typ[sh[c].sh_link]) {
		    dat[sh[c].sh_link] = strings;
		    strings += sh[sh[c].sh_link].sh_size;
		    typ[sh[c].sh_link] = TYP_STRINGS;
		}
	    } else {
		sh[c].sh_type = SHT_NULL;
	    }
	} else if (sh[c].sh_type == SHT_DYNSYM) {
	    if (sh[c].sh_entsize == sizeof(Elf32_Sym)) {
		/* XXX Attention: can't have multiple SYMTAB/DYNSYM */
		nsyn += sh[c].sh_size / sizeof(Elf32_Sym);
		if (!typ[sh[c].sh_link]) {
		    dat[sh[c].sh_link] = strings;
		    strings += sh[sh[c].sh_link].sh_size;
		    typ[sh[c].sh_link] = TYP_STRINGS;
		}
	    } else {
		sh[c].sh_type = SHT_NULL;
	    }
	} else if (sh[c].sh_type == SHT_DYNAMIC) {
	    if (sh[c].sh_entsize == sizeof(Elf32_Dyn)) {
		ndep += sh[c].sh_size / sizeof(Elf32_Dyn);
		if (!typ[sh[c].sh_link]) {
		    dat[sh[c].sh_link] = strings;
		    strings += sh[sh[c].sh_link].sh_size;
		    typ[sh[c].sh_link] = TYP_STRINGS;
		}
	    } else {
		sh[c].sh_type = SHT_NULL;
	    }
	}
    }

    /* allocate data:
     * +------------------+
     * | object structure |
     * +------------------+
     * |      strings     |
     * +------------------+
     * |      symtab      |
     * +------------------+
     * |      dynsym      |
     * +------------------+
     * |      dynamic     |
     * +------------------+
     */
    i = ROUND_UP(sizeof(stone_object) + (long)strings);
    i += ROUND_UP(nsym * sizeof(stone_sym) +
		  nsyn * sizeof(stone_sym) +
		  nrel * sizeof(stone_rel) +
		  ndep * sizeof(char *));
    if (code == NULL) {
	i += ROUND_UP(len);
    }

    old_break = mysbrk(i);
    if (old_break == SNULL) {
	return NULL;
    }
    new_break = mysbrk(0);
    DLOG(("old_break: %x, new_break: %x, blocksize: %x\n", (int)old_break, (int)new_break, i));

    /* fill up object */
    obj = (stone_object *)old_break;
    b_zero(obj, sizeof(stone_object));
    obj->src = src;
    obj->len = len;
    obj->nsym = nsym;
    obj->nsyn = nsyn;
    obj->nrel = nrel;
    obj->ndep = ndep;
    obj->mode = RTLD_GLOBAL;
    obj->type = hdr.e_type;
    list_append(objects, obj);

    /* set up pointers (ET_DYN will have obj->sym == obj->dyn) */
    base = (char *)obj + sizeof(stone_object);
    obj->sym = (stone_sym *)ROUND_UP(((long)base + (long)strings));
    obj->syn = (stone_sym *)((char *)obj->sym + nsym * sizeof(stone_sym));
    obj->rel = (stone_rel *)((char *)obj->syn + nsyn * sizeof(stone_sym));
    obj->dep = (char **)((char *)obj->rel + nrel * sizeof(stone_rel));
    obj->code = (char *)ROUND_UP((long)obj->dep + ndep * sizeof(char *));
    strings = base;

    if (code != NULL) {
	obj->code = code;
    }

    /* fixup pointers */
    for (c = 0; c < hdr.e_shnum; c++) {
	if (typ[c] == TYP_RESIDENT) {
	    dat[c] += (long)obj->code;
	} else if (typ[c] == TYP_STRINGS) {
	    dat[c] += (long)strings;
	}
    }

    /* load the bulk */
    if (code == NULL) {
	for (c = 0; c < hdr.e_shnum; c++) {
	    if (typ[c]) {
		if (sh[c].sh_type == SHT_NOBITS) {
		    b_zero(dat[c], sh[c].sh_size);
		} else {
		    if (!f_read(dat[c], sh[c].sh_size, f, infile + sh[c].sh_offset)) {
			ELF_ERROR();
		    }
		}
	    }
	}
    }

    /* now read in the symbols (ET_DYN won't load SHT_SYMTAB) */
    obj->nsym = 0;
    for (c = 0; c < hdr.e_shnum; c++) {
	if (sh[c].sh_type == SHT_SYMTAB) {
	    Elf32_Sym *sym;
	    stone_sym *sobj;
	    char *str;

	    sym = (Elf32_Sym *)readsection(f, dat, c, sh, infile);
	    str = readsection(f, dat, sh[c].sh_link, sh, infile);
	    if (sym == SNULL || str == SNULL) {
		ELF_ERROR();
	    }

	    old = obj->nsym;
	    obj->nsym += sh[c].sh_size / sizeof(Elf32_Sym);

	    for (i = old, sobj = obj->sym + old; i < obj->nsym; sym++, i++, sobj++) {
		sobj->name = &str[sym->st_name];
		sobj->obj = NULL;
		sobj->sym = NULL;
		sobj->type = csst_unknown;

		if (ELF32_ST_BIND(sym->st_info) == STB_GLOBAL ||
		    ELF32_ST_BIND(sym->st_info) == STB_WEAK) {
		    sobj->type = csst_export;
		    if (sym->st_shndx == SHN_UNDEF) {
			sobj->type = csst_import;
		    }
		    if (ELF32_ST_BIND(sym->st_info) == STB_WEAK) {
			sobj->type |= csst_weak;
		    }
		}
		if (sym->st_shndx < hdr.e_shnum || sym->st_shndx == SHN_ABS) {
		    sobj->value = sym->st_value;
		} else {
		    ELF_ERROR();
		}
		SUB_VADDR(sobj->value);
		DLOG(("    symbol %x offset %x %s\n", i, sobj->value, sobj->name));
	    }
	}
    }

    /* now read in the symbols */
    obj->nsyn = 0;
    for (c = 0; c < hdr.e_shnum; c++) {
	if (sh[c].sh_type == SHT_DYNSYM) {
	    Elf32_Sym *sym;
	    stone_sym *sobj;
	    char *str;

	    sym = (Elf32_Sym *)readsection(f, dat, c, sh, infile);
	    str = readsection(f, dat, sh[c].sh_link, sh, infile);
	    if (sym == SNULL || str == SNULL) {
		ELF_ERROR();
	    }

	    old = obj->nsyn;
	    obj->nsyn += sh[c].sh_size / sizeof(Elf32_Sym);

	    for (i = old, sobj = obj->syn + old; i < obj->nsyn; sym++, i++, sobj++) {
		sobj->name = &str[sym->st_name];
		sobj->obj = NULL;
		sobj->sym = NULL;
		sobj->type = csst_unknown;

		if (ELF32_ST_BIND(sym->st_info) == STB_GLOBAL ||
		    ELF32_ST_BIND(sym->st_info) == STB_WEAK) {
		    sobj->type = csst_export;
		    if (sym->st_shndx == SHN_UNDEF) {
			sobj->type = csst_import;
		    }
		    if (ELF32_ST_BIND(sym->st_info) == STB_WEAK) {
			sobj->type |= csst_weak;
		    }
		}
		if (sym->st_shndx < hdr.e_shnum || sym->st_shndx == SHN_ABS) {
		    sobj->value = sym->st_value;
		} else {
		    ELF_ERROR();
		}
		SUB_VADDR(sobj->value);
		DLOG(("    dynsym %x offset %x %s\n", i, sobj->value, sobj->name));
	    }
	}
    }

    if (obj->sym == obj->syn) {
	obj->nsym = obj->nsyn;
    }

    /* and the links */
    obj->nrel = 0;
    for (c = 0; c < hdr.e_shnum; c++) {
	if (sh[c].sh_type == SHT_REL) {
	    Elf32_Rel *rel = (Elf32_Rel *)readsection(f, dat, c, sh, infile);
	    stone_rel *robj;
	    stone_sym *tab = (sh[sh[c].sh_link].sh_type == SHT_SYMTAB) ? obj->sym : obj->syn;

	    if (rel == SNULL) {
		ELF_ERROR();
	    }

	    old = obj->nrel;
	    obj->nrel += sh[c].sh_size / sizeof(Elf32_Rel);

	    for (i = old, robj = obj->rel + old; i < obj->nrel; rel++, i++, robj++) {
		int offset = rel->r_offset;
		SUB_VADDR(offset);

		robj->off = &obj->code[offset];
		robj->sym = &tab[ELF32_R_SYM(rel->r_info)];
		robj->type = csr_unknown;
		robj->addend = 0;

		if (ELF32_R_TYPE(rel->r_info) == R_386_JMP_SLOT ||	/* S */
		    ELF32_R_TYPE(rel->r_info) == R_386_GLOB_DAT) {	/* S */
		    robj->type = csr_type1;
		    DLOG(("    rel1 %x off = %x sym = %x %s\n",
			i, offset, (int)ELF32_R_SYM(rel->r_info), robj->sym->name));
		    /*plt = *(int *)&obj->code[*(int *)robj->off + 6 - vaddr] + *(int *)robj->off + 0xA - vaddr;*/
		} else if (ELF32_R_TYPE(rel->r_info) == R_386_RELATIVE) {/* B + A */
		    robj->type = csr_type2;
		    robj->addend = *(int *)robj->off;
		    SUB_VADDR(robj->addend);
		    DLOG(("    rel2 %x off = %x add = %x sym = %x %s\n",
			i, offset, robj->addend,
			(int)ELF32_R_SYM(rel->r_info), robj->sym->name));
		} else if (ELF32_R_TYPE(rel->r_info) == R_386_32) {	/* S + A */
		    robj->type = csr_type3;
		    robj->addend = *(int *)robj->off;
		    SUB_VADDR(robj->addend);
		    DLOG(("    rel3 %x off = %x add = %x sym = %x %s\n",
			i, offset, robj->addend,
			(int)ELF32_R_SYM(rel->r_info), robj->sym->name));
		} else if (ELF32_R_TYPE(rel->r_info) == R_386_PC32) {	/* S + A - P */
		    robj->type = csr_type4;
		    robj->addend = *(int *)robj->off;
		    SUB_VADDR(robj->addend);
		    DLOG(("    rel4 %x off = %x add = %x sym = %x %s\n",
			i, offset, robj->addend,
			(int)ELF32_R_SYM(rel->r_info), robj->sym->name));
		} else if (ELF32_R_TYPE(rel->r_info) == R_386_GOTOFF) {	/* S + A - GOT */
		    /* XXX ET_EXEC:
		     * robj->addend + &_GLOBAL_OFFSET_TABLE_ -> symbol address
		     * no further relocation is required
		     */
		    robj->type = csr_fixed;
		    robj->addend = *(int *)robj->off;
		    SUB_VADDR(robj->addend);
		    DLOG(("    rel5 %x off = %x add = %x sym = %x %s\n",
			i, offset, robj->addend,
			(int)ELF32_R_SYM(rel->r_info), robj->sym->name));
		} else if (ELF32_R_TYPE(rel->r_info) == R_386_GOTPC) {	/* GOT + A - P */
		    /* XXX ET_EXEC:
		     * robj->addend + EBX -> _GLOBAL_OFFSET_TABLE_
		     * no further relocation is required
		     */
		    robj->type = csr_fixed;
		    robj->addend = *(int *)robj->off;
		    SUB_VADDR(robj->addend);
		    DLOG(("    rel6 %x off = %x add = %x sym = %x %s\n",
			i, offset, robj->addend,
			(int)ELF32_R_SYM(rel->r_info), robj->sym->name));
		} else if (ELF32_R_TYPE(rel->r_info) == R_386_GOT32) {	/* G + A - P */
		    /* XXX ET_EXEC:
		     * _GLOBAL_OFFSET_TABLE_[robj->addend] = &symbol
		     * no further relocation is required
		     */
		    robj->type = csr_fixed;
		    robj->addend = *(int *)robj->off;
		    SUB_VADDR(robj->addend);
		    DLOG(("    rel7 %x off = %x add = %x sym = %x %s\n",
			i, offset, robj->addend,
			(int)ELF32_R_SYM(rel->r_info), robj->sym->name));
		} else if (ELF32_R_TYPE(rel->r_info) == R_386_PLT32) {	/* L + A - P */
		    /* XXX FIXME: this call will never go through PLT */
		    robj->type = csr_type4;
		    robj->addend = *(int *)robj->off;
		    SUB_VADDR(robj->addend);
		    DLOG(("    rel8 %x off = %x add = %x sym = %x %s\n",
			i, offset, robj->addend,
			(int)ELF32_R_SYM(rel->r_info), robj->sym->name));
		} else if (ELF32_R_TYPE(rel->r_info) == R_386_NONE) {
		    robj->type = csr_fixed;
		}
	    }
	}
    }

    /* collect the dependencies */
    obj->ndep = 0;
    for (c = 0; c < hdr.e_shnum; c++) {
	if (sh[c].sh_type == SHT_DYNAMIC) {
	    Elf32_Dyn *dyn;
	    char **dobj;
	    char *str;

	    dyn = (Elf32_Dyn *)readsection(f, dat, c, sh, infile);
	    str = readsection(f, dat, sh[c].sh_link, sh, infile);

	    if (dyn == SNULL || str == SNULL) {
		ELF_ERROR();
	    }

	    old = obj->ndep;
	    ndep = old + sh[c].sh_size / sizeof(Elf32_Dyn);

	    for (i = old, dobj = obj->dep + old; i < ndep; i++) {
		if (dyn[i].d_tag == DT_NEEDED) {
		    dobj[0] = &str[dyn[i].d_un.d_val];
		    DLOG(("    need %s\n", dobj[0]));
		    dobj++;
		    obj->ndep++;
		} else if (dyn[i].d_tag == DT_INIT) {
		    obj->init = (void (*) (void))(dyn[i].d_un.d_ptr - vaddr + obj->code);
		} else if (dyn[i].d_tag == DT_FINI) {
		    obj->fini = (void (*) (void))(dyn[i].d_un.d_ptr - vaddr + obj->code);
		/*} else if (dyn[i].d_tag == DT_PLTGOT) {
		    got = dyn[i].d_un.d_ptr - vaddr;*/
		}
	    }
	}
    }

    /* finish allocation now */
    len = 0;
    len = new_break - (char *)mysbrk(0) + len; /* XXX ok in MULTIBLOCK?!? */
    DLOG(("Freeing %x bytes\n", -len));
    mysbrk(len);

    /* external modules */
    for (i = 0; i < obj->ndep; i++) {
	stone_object *dep = do_load(obj->dep[i], objects);
	if (dep == NULL) {
	    elf_load_error(obj->dep[i]);
	    ELF_ERROR();
	}
    }

    DLOG(("OK\n"));
    return obj;

  fail:
    list_remove(obj);
    len = old_break - (char *)mysbrk(0);
    mysbrk(len);
    return NULL;
}


/**
 * Scan symbol.
 *
 * \param name symbol name
 * \param ret used to return the symbol
 * \param obj object to scan symbol for (RTLD_DEFAULT to scan all objects)
 * \param objects linked list of all objects
 *
 * \return object in which the exported symbol is found
 *
 * \note this code runs both relocated and unrelocated
 */
stone_object *
__elf_scan_symbol (const char *name, stone_sym **ret, stone_object *obj, stone_object *objects)
{
    if (name != NULL) {
	int i;
	stone_sym *sym;

	if (obj == RTLD_DEFAULT) {
	    list_foreach (obj, objects) {
		if (obj->mode & RTLD_GLOBAL) {
		    for (sym = obj->sym, i = 0; i < obj->nsym; i++, sym++) {
			if (sym->type == csst_export &&
			    sym->name != NULL && !str_cmp_at(sym->name, name)) {
			    *ret = sym;
			    return obj;
			}
		    }
		}
	    }
	    list_foreach (obj, objects) {
		if (obj->mode & RTLD_GLOBAL) {
		    for (sym = obj->sym, i = 0; i < obj->nsym; i++, sym++) {
			if (sym->type == (csst_export | csst_weak) &&
			    sym->name != NULL && !str_cmp_at(sym->name, name)) {
			    *ret = sym;
			    return obj;
			}
		    }
		}
	    }
	} else {
	    for (sym = obj->sym, i = 0; i < obj->nsym; i++, sym++) {
		if (sym->type == csst_export &&
		    sym->name != NULL && !str_cmp_at(sym->name, name)) {
		    *ret = sym;
		    return obj;
		}
	    }
	    for (sym = obj->sym, i = 0; i < obj->nsym; i++, sym++) {
		if (sym->type == (csst_export | csst_weak) &&
		    sym->name != NULL && !str_cmp_at(sym->name, name)) {
		    *ret = sym;
		    return obj;
		}
	    }
	}
    }

    return NULL;
}


/**
 * Link symbol.
 *
 * \param obj object in which the undefined symbol resides
 * \param sym symbol to be linked
 * \param objects linked list of all objects
 *
 * \return 1 for success, 0 if the symbol could not be linked
 *
 * \note this code runs both relocated and unrelocated
 */
static int
link_symbol (stone_object *obj, stone_sym *sym, stone_object *objects)
{
    if (sym->obj != NULL) {
	return 1;
    }

    if ((sym->type & 0xFF) == csst_export || sym->type == csst_unknown) {
	sym->obj = obj;
	sym->sym = sym;
	return 1;
    }

    sym->obj = __elf_scan_symbol(sym->name, &sym->sym, RTLD_DEFAULT, objects);
    if (sym->obj != NULL) {
	return 1;
    }

    return 0;
}


/**
 * Link one object.
 *
 * \param report report unsolved links
 * \param obj object to be linked
 * \param objects linked list of all objects
 *
 * \return 1 for success, 0 if the object could not be linked
 *
 * \note this code runs both relocated and unrelocated
 */
int
__elf_resolve (int report, stone_object *obj, stone_object *objects)
{
    if (!obj->link) {
	int i, n = 0;

	DLOG(("%s %x\n", obj->src, obj->nrel));
	/* link relocations */
	for (i = 0; i < obj->nrel; i++) {
	    char *rel_off = obj->rel[i].off;
	    int rel_type = obj->rel[i].type;
	    int rel_addend = obj->rel[i].addend;

	    stone_sym *isym = obj->rel[i].sym;
	    stone_sym *esym = NULL;
	    stone_object *eobj = NULL;

	    if (rel_type & csr_fixed) {
		/* do not redo fixups */
		n++;
		continue;
	    }

	    if (rel_type == csr_type1) {
		DLOG(("  fix1 %x", i));
		if (link_symbol(obj, isym, objects)) {
		    eobj = isym->obj;
		    esym = isym->sym;
		    DLOG((" [%x] = %x                       = %x %s", rel_off - obj->code,
			(int)&eobj->code[esym->value],
			(int)&eobj->code[esym->value], isym->name));
		    *(int *)rel_off = (int)&eobj->code[esym->value];
		    rel_type |= csr_fixed;
		}
#ifdef DEBUG
		else {
		    DLOG((" %s", isym->name));
		}
#endif
		DLOG(("\n"));
	    } else if (rel_type == csr_type2) {
		DLOG(("  fix2 %x [%x] = %x + %x            = %x %s\n", i, rel_off - obj->code, (int)&obj->code[0], rel_addend, (int)&obj->code[0] + rel_addend, isym->name));
		*(int *)rel_off = (int)&obj->code[0] + rel_addend;
		rel_type |= csr_fixed;
	    } else if (rel_type == csr_type3) {
		DLOG(("  fix3 %x", i));
		if (link_symbol(obj, isym, objects)) {
		    eobj = isym->obj;
		    esym = isym->sym;
		    /* bypasses PLT and solves the problem of external vars */
		    if (obj->type == ET_EXEC && obj != eobj) {
			DLOG((" [%x] = %x                       : %x %s", rel_off - obj->code,
			  (int)&eobj->code[esym->value],
			  (int)&eobj->code[esym->value],
			  isym->name));
			*(int *)rel_off = (int)&eobj->code[esym->value];
		    } else if (obj->type == ET_EXEC) {
			DLOG((" [%x] = %x + %x            ~ %x %s", rel_off - obj->code,
			  (int)&obj->code[0], rel_addend,
			  (int)&obj->code[0] + rel_addend,
			  isym->name));
			*(int *)rel_off = (int)&obj->code[0] + rel_addend;
		    } else {
			DLOG((" [%x] = %x + %x            = %x %s", rel_off - obj->code,
			  (int)&eobj->code[esym->value], rel_addend,
			  (int)&eobj->code[esym->value] + rel_addend,
			  isym->name));
			*(int *)rel_off = (int)&eobj->code[esym->value] + rel_addend;
		    }
		    rel_type |= csr_fixed;
		}
#ifdef DEBUG
		else {
		    DLOG((" %s", isym->name));
		}
#endif
		DLOG(("\n"));
	    } else if (rel_type == csr_type4) {
		DLOG(("  fix4 %x", i));
		if (link_symbol(obj, isym, objects)) {
		    eobj = isym->obj;
		    esym = isym->sym;
		    if (obj->type == ET_EXEC && obj == eobj) {
			DLOG((" [%x] =            %x            = %x %s", rel_off - obj->code, rel_addend, rel_addend, isym->name));
		    } else {
			rel_addend = -4;
			DLOG((" [%x] = %x + %x - %x = %x %s", rel_off - obj->code,
			  (int)&eobj->code[esym->value], rel_addend, (int)rel_off,
			  (int)&eobj->code[esym->value] + rel_addend - (int)rel_off,
			  isym->name));
			*(int *)rel_off = (int)&eobj->code[esym->value] + rel_addend - (int)rel_off;
		    }
		    rel_type |= csr_fixed;
		}
#ifdef DEBUG
		else {
		    DLOG((" %s", isym->name));
		}
#endif
		DLOG(("\n"));
	    }

	    if ((rel_type & csr_fixed) || (isym->type & csst_weak)) {
		n++;
		obj->rel[i].type = rel_type;
	    } else if (report) {
		elf_link_error(obj->src, isym->name);
	    }
	    if ((eobj != NULL) && (eobj != obj)) {
		eobj->user++;
	    }
	}

	obj->link = (n == obj->nrel);
    }

    DLOG(("linked %x\n", obj->link));
    return obj->link;
}


/**
 * Link all objects.
 *
 * \param objects linked list of all objects
 *
 * \return 0 for success, -1 if any of the objects could not be linked
 *
 * \note this code runs both relocated and unrelocated
 */
int
__elf_link (stone_object *objects)
{
    stone_object *obj;
    list_foreach (obj, objects) {
	__elf_resolve(!0, obj, objects);
	if (!obj->link) {
	    return -1;
	}
    }
    return 0;
}


/**
 * Load any image.
 *
 * \param filename filename of the image
 * \param objects linked list of all objects
 *
 * \return loaded object
 *
 * \note this code runs unrelocated
 */
static stone_object *
do_load (const char *filename, stone_object *objects)
{
    int fd;
    stone_object *obj;

    list_foreach (obj, objects) {
	if (!str_cmp(obj->src, filename)) {
	    return obj;
	}
    }

    fd = dos_open(filename);
    if (fd < 0) {
	return NULL;
    }

    obj = elf_load(fd, 0, NULL, filename, objects);
    dos_close(fd);

    return obj;
}


/**
 * Load self image.
 *
 * \param file DOS file handle
 * \param offs ELF offset in executable (equivalent to stub size)
 * \param code address at which the stub organized *BITS sections
 * \param src name of the executable
 * \param objects linked list of all objects
 *
 * \return self object
 *
 * \note this code runs unrelocated
 */
void *
__elf_load (int file, int offs, void *code, const char *src, stone_object *objects)
{
    stone_object *obj;
    list_create(objects);
    obj = elf_load(file, offs, code, src, objects);
    dos_close(file);
    return obj;
}
