/* more 3d functions Jan 30th 2001 */
#include <stdio.h>
#include <float.h>
#include "m3d.h"

/* some coo macros */
#define MIN3(x,y,z) (MIN(MIN(x,y),z))
#define MAX3(x,y,z) (MAX(MAX(x,y),z))

/* render state */
static face faces[TOTAL_FACES];          /* transformed faces */
static int  nofaces;                     /* the number in the buf */
static light *lights[TOTAL_LIGHTS];      /* lights used */
static int  nolights;                    /* number of lights */
static MATRIX_f *cam;                    /* the viewing camera */
static float cx,cy,cz;
static BITMAP *scr;
static V3D_f *ctmp[6], _ctmp[6], *cout[6], _cout[6], *temp[3];
static int out[6];
static int inited=0;

float sframes=0,sfaces=0,clipdist=1000.0;

/* shade a RGB based on a 0-256 level */
static unsigned long pixelshade(unsigned long col, unsigned long level)
{
    return
        ((((col&255)*level)&0xFF00)>>8)|
        ((((col>>8)&255)*level)&0xFF00)|
        (((((col>>16)&255)*level)&0xFF00)<<8);
}

/* Begin rendering objects, dest is the output BITMAP, camera is your viewing
 * camera.  You cannot change either before you call m3d_draw */
void m3d_Begin(BITMAP *dest, MATRIX_f *camera)
{
    int x;

    if (!inited) {
        for (x = 0; x < 6; x++) {
            ctmp[x] = &_ctmp[x];
            cout[x] = &_cout[x]; }

    #ifdef DJGPP
        /* Set FPU into a low-precision mode without exceptions */
        _control87(MCW_EM|PC_24,MCW_EM|MCW_PC);
    #endif

        inited = 1;
    }
    /* setup pointers */
    cam = camera;
    scr = dest;
    nolights = nofaces = 0;
    apply_matrix_f(cam,0,0,1,&cx,&cy,&cz);
}

/* Adds a light to the mix */
void m3d_RenderLight(light *l)
{
    if (nolights < (sizeof(lights) / sizeof(light)))
        lights[nolights++] = l;
}

/* does the lighting of a face */
static void do_lights(face *f)
{
    float power, sum, xv,yv,zv;
    int x, y;

    if (nolights==0) return;

    /* for each vertex */
    for (y = 0; y < 3; y++) {
        sum = 0.0;
        /* for each light */
        for (x = 0; x < nolights; x++) {
            power = 256.0 * lights[x]->intensity;

            /* make a directed power */
            if (lights[x]->flags & LIGHT_DIRECTIONAL) {
                xv = f->v[y].x - lights[x]->Xp;
                yv = f->v[y].y - lights[x]->Yp;
                zv = f->v[y].z - lights[x]->Zp;
                normalize_vector_f(&xv, &yv, &zv);
                xv = dot_product_f( xv,yv,zv,lights[x]->Xa,
                                    lights[x]->Ya, lights[x]->Za);
                if (xv > 0)
                    power *= xv;
                else
                    power = 0.0;    /* object is not facing light */
            }

            /* distance modified */
            if (lights[x]->flags & LIGHT_FADE) {
                xv = f->v[y].x - lights[x]->Xp;
                yv = f->v[y].y - lights[x]->Yp;
                zv = f->v[y].z - lights[x]->Zp;
                xv = vector_length_f(xv,yv,zv) / lights[x]->fadedist;
                xv = pow(xv, lights[x]->falloff);
                if (xv > 0)
                   power /= xv;
            }

            /* dark light? */
            if (lights[x]->flags & LIGHT_INVERSE)
                power = 256.0 - power;
            sum += power;
        }
        if (sum > 256.0)
           sum = 256.0;
        else if (sum < 0.0)
           sum = 0.0;

        /* apply light */
        f->v[y].c = pixelshade(f->v[y].c, (unsigned long)sum);
    }
}

/* Adds the faces of an object and it's childs to the mix */
static void _m3d_RenderObj(obj *o, float Xa, float Ya, float Za, float Xp, float Yp, float Zp)
{
    MATRIX_f omat, mat;
    float xt,yt,zt;
    int x;

    /* make the object matrix */
    get_rotation_matrix_f(&omat, Xa=o->Xa+Xa, Ya=o->Ya+Ya, Za=o->Za+Za);
    omat.t[0] = Xp=o->Xp+Xp;
    omat.t[1] = Yp=o->Yp+Yp;
    omat.t[2] = Zp=o->Zp+Zp;

    /* Compute CamMat * ObjMat */
    matrix_mul_f(&omat, cam, &mat);

    /* for all faces */
    for (x = 0; (x < o->nfaces) && (nofaces < (sizeof(faces) / sizeof(face))); x++) {
        /* rotate normal */
        apply_matrix_f( &omat, o->faces[x].norm[0], o->faces[x].norm[1],
                        o->faces[x].norm[2], &xt, &yt, &zt );
        xt -= Xp;
        yt -= Yp;
        zt -= Zp;

        if ((o->faces[x].backface) || (dot_product_f(xt,yt,zt,cx,cy,cz)<0)) {
            /* copy misc data */
            memcpy(&(faces[nofaces].v), &(o->faces[x].v), sizeof(faces[0].v));

            /* add the face */
            apply_matrix_f( &mat, o->faces[x].v[0].x, o->faces[x].v[0].y,
                            o->faces[x].v[0].z, &faces[nofaces].v[0].x,
                            &faces[nofaces].v[0].y, &faces[nofaces].v[0].z);
            apply_matrix_f( &mat, o->faces[x].v[1].x, o->faces[x].v[1].y,
                            o->faces[x].v[1].z, &faces[nofaces].v[1].x,
                            &faces[nofaces].v[1].y, &faces[nofaces].v[1].z);
            apply_matrix_f( &mat, o->faces[x].v[2].x, o->faces[x].v[2].y,
                            o->faces[x].v[2].z, &faces[nofaces].v[2].x,
                            &faces[nofaces].v[2].y, &faces[nofaces].v[2].z);

            if ((faces[nofaces].v[0].z > 0) ||
                (faces[nofaces].v[1].z > 0) ||
                (faces[nofaces].v[2].z > 0)) {
                /* perform lighting */
                do_lights(&faces[nofaces]);
            
                /* setup textures */
                faces[nofaces].texture = o->faces[x].texture;
                faces[nofaces].shadetype = o->faces[x].shadetype;
                ++nofaces;
            }
        }
    }
    for (x = 0; x < o->nchild; x++)
        _m3d_RenderObj(&o->child[x], Xa,Ya,Za,Xp,Yp,Zp);
}

void m3d_RenderObj(obj *o)
{
    _m3d_RenderObj(o,0,0,0,0,0,0);
}

static int m3d_sort(const void *a, const void *b)
{
    face *e1 = (face *)a, *e2 = (face *)b;
    float s1, s2;

    s1 = (e1->v[0].z+e1->v[1].z+e1->v[2].z);
    s2 = (e2->v[0].z+e2->v[1].z+e2->v[2].z);

    if (s1 > s2)
        return -1;
    else if (s2 > s1)
        return 1;
    else
        return 0;
}
        
void m3d_Draw(int zbuffered)
{
    int x, y, z;

    /* stats */
    ++sframes;
    sfaces += nofaces;

    if (!zbuffered)
        /* sort faces back to front */
        qsort((void *)faces, nofaces, sizeof(face), &m3d_sort);

    /* draw faces */
    for (x = 0; x < nofaces; x++) {
        temp[0] = &faces[x].v[0];
        temp[1] = &faces[x].v[1];
        temp[2] = &faces[x].v[2];
        y = clip3d_f(faces[x].shadetype, 0.1, clipdist, 3,
                     temp, cout, ctmp, out);

        for (z=0; z<y; z++)
            persp_project_f(cout[z]->x, cout[z]->y, cout[z]->z, &cout[z]->x, &cout[z]->y);

        if (faces[x].shadetype == POLYTYPE_WIRE) {
            for (z = 1; z < y; z++)
                line(scr, cout[z-1]->x, cout[z-1]->y, cout[z]->x, cout[z]->y, makecol(faces[x].v[0].c, faces[x].v[0].c, faces[x].v[0].c));
        } else 
            polygon3d_f(scr, faces[x].shadetype, faces[x].texture,
                        y, cout);
    }
}

obj *m3d_CreateObj(int Nofaces, int Nochild)
{
    obj *o;

    o = calloc(1, sizeof(obj));
    if (!o) return NULL;

    o->nfaces = Nofaces;
    if (Nofaces)
        o->faces = calloc(Nofaces, sizeof(face));
    else
        o->faces = NULL;
    o->nchild = Nochild;
    if (Nochild)
        o->child = calloc(Nochild, sizeof(obj));
    else
        o->child = NULL;

    return o;
}

static void _m3d_CopyObj(obj *dest, obj *src)
{
    int x;

    /* allocate space for children and faces */
    dest->nfaces = src->nfaces;
    dest->nchild = src->nchild;
    if (src->nfaces)
        dest->faces = calloc(src->nfaces, sizeof(face));
    else
        dest->faces = NULL;

    if (src->nchild)
        dest->child = calloc(src->nchild, sizeof(obj));
    else
        dest->child = NULL;

    /* copy faces+child */
    memcpy(dest->faces, src->faces, src->nfaces * sizeof(face));
    for (x = 0; x < src->nchild; x++)
        _m3d_CopyObj(&dest->child[x], &src->child[x]);
}

obj *m3d_CopyObj(obj *src)
{
    obj *o;

    o = calloc(1, sizeof(obj));
    _m3d_CopyObj(o, src);
    return o;
}

void m3d_CalcNormals(obj *o)
{
    int x;
    float x1,y1,z1,x2,y2,z2;
    for (x = 0; x < o->nfaces; x++) {
        /* get two vectors */
        x1 = o->faces[x].v[0].x - o->faces[x].v[1].x;
        y1 = o->faces[x].v[0].y - o->faces[x].v[1].y;
        z1 = o->faces[x].v[0].z - o->faces[x].v[1].z;
        x2 = o->faces[x].v[0].x - o->faces[x].v[2].x;
        y2 = o->faces[x].v[0].y - o->faces[x].v[2].y;
        z2 = o->faces[x].v[0].z - o->faces[x].v[2].z;
        cross_product_f(x1,y1,z1,x2,y2,z2,&(o->faces[x].norm[0]),&(o->faces[x].norm[1]),&(o->faces[x].norm[2]));
        normalize_vector_f(&(o->faces[x].norm[0]),&(o->faces[x].norm[1]),&(o->faces[x].norm[2]));
    }
    for (x = 0; x < o->nchild; x++)
        m3d_CalcNormals(&o->child[x]);
}

void m3d_FlipNormals(obj *o)
{
    int x;
    for (x = 0; x < o->nfaces; x++) {
        o->faces[x].norm[0] = -o->faces[x].norm[0];
        o->faces[x].norm[1] = -o->faces[x].norm[1];
        o->faces[x].norm[2] = -o->faces[x].norm[2]; }
    for (x = 0; x < o->nchild; x++)
        m3d_FlipNormals(&o->child[x]);
}

static void _m3d_DeleteObj(obj *o)
{
    int x;
    for (x = 0; x < o->nchild; x++)
        _m3d_DeleteObj(&o->child[x]);
    if (o->faces != NULL)
        free(o->faces);
    if (o->child != NULL)
        free(o->child);
}

void m3d_DeleteObj(obj *o)
{
    _m3d_DeleteObj(o);
    free(o);
}

void m3d_ApplyTextureFace(face *f, BITMAP *txt)
{
    int x;
    float w, h;

    w = txt->w;
    h = txt->h;
    f->texture = txt;
    for (x = 0; x < 3; x++) {
        f->v[x].u = (int)(w * ((float)f->v[x].u / (float)(1<<16)));
        f->v[x].v = (int)(h * ((float)f->v[x].v / (float)(1<<16)));
        f->v[x].c = 255; }
}

void m3d_ApplyTexture(obj *o, BITMAP *txt)
{
    int x, y;
    float w, h;

    w = txt->w;
    h = txt->h;
    for (x = 0; x < o->nfaces; x++) {
        o->faces[x].texture = txt;
        for (y = 0; y < 3; y++) {
            o->faces[x].v[y].u = (int)(w * ((float)o->faces[x].v[y].u / (float)(1<<16)));
            o->faces[x].v[y].v = (int)(h * ((float)o->faces[x].v[y].v / (float)(1<<16)));
            o->faces[x].v[y].c = 255;
        }
    }
    for (x = 0; x < o->nchild; x++)
        m3d_ApplyTexture(&o->child[x], txt);
}

void m3d_ChangeShading(obj *o, int newtype)
{
    int x;
    for (x = 0; x < o->nfaces; x++)
        o->faces[x].shadetype = newtype;
    for (x = 0; x < o->nchild; x++)
        m3d_ChangeShading(&o->child[x], newtype);
}

void m3d_SetBackface(obj *o, int newsetting)
{
    int x;
    for (x = 0; x < o->nfaces; x++)
        o->faces[x].backface = newsetting;
    for (x = 0; x < o->nchild; x++)
        m3d_SetBackface(&o->child[x], newsetting);
}


void m3d_StretchObj(obj *o, float x, float y, float z)
{
    int i, ii;
    for (i = 0; i < o->nfaces; i++)
        for (ii = 0; ii < 3; ii++) {
            o->faces[i].v[ii].x *= x;
            o->faces[i].v[ii].y *= y;
            o->faces[i].v[ii].z *= z;
        }
    for (i = 0; i < o->nchild; i++)
        m3d_StretchObj(&o->child[i], x, y, z);
    m3d_CalcNormals(o);
}

void m3d_ApplyColor(obj *o, unsigned long color)
{
    int i, ii;
    for (i = 0; i < o->nfaces; i++)
        for (ii = 0; ii < 3; ii++)
            o->faces[i].v[ii].c = color;
    for (i = 0; i < o->nchild; i++)
        m3d_ApplyColor(&o->child[i], color);
}
