/* pc_inode.c - assign the same unique st_ino to a file name
   and its file descriptor.
   Copyright (C) 2010 Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

/* Written by Juan M. Guerrero <juan.guerrero@gmx.de>,  2010-05-08.  */


#if HAVE_CONFIG_H
# include <config.h>
#endif

#if HAVE_STDLIB_H
# include <stdlib.h>
#endif

#if HAVE_STRING_H
# include <string.h>
#endif

#include <ctype.h>
#include <stdarg.h>
#include <stdbool.h>


void *xmalloc(size_t size);

typedef struct inode_table_entry_tag *entry_ptr;
typedef struct inode_table_entry_tag {
  ino_t      inode_number;
  int        fd;
  char      *file_name;
  entry_ptr  next_inode;
} inode_table_entry;

static entry_ptr first_inode = NULL;


ino_t create_inode(const char *file_name, const int fd)
{
  int length;
  bool file_name_found;
  entry_ptr last_inode, inode;


  if (!first_inode)
  {
    first_inode = xmalloc(sizeof(inode_table_entry));
    first_inode->inode_number = 0;
    first_inode->fd = -1;
    first_inode->file_name = NULL;
    first_inode->next_inode = NULL;
  }
  inode = first_inode;


  /*
   *  Check if file name has already been registred.
   */
  for (file_name_found = true, last_inode = inode; inode && inode->file_name; last_inode = inode, inode = inode->next_inode, file_name_found = true)
  {
    int i;

    for (i = 0; inode->file_name[i] && file_name[i]; i++)
      if (!filename_char_eq(inode->file_name[i], file_name[i]))
      {
        file_name_found = false;
        break;
      }

    if (file_name_found && inode->file_name[i] == file_name[i])
    {
      if (inode->fd == -1)
        inode->fd = fd;
      return inode->inode_number;
    }
  }


  /*
   *  File name has not already been registred
   *  so allocate a new entry in the list and
   *  return the list index as inode number.
   */
  for (length = 0; file_name[length]; length++)
    ;
  length++;

  if (!inode)
  {
    inode = xmalloc(sizeof(inode_table_entry));
    last_inode->next_inode = inode;
  }
  inode->next_inode = NULL;
  inode->inode_number = last_inode->inode_number + 1;
  inode->fd = fd;
  inode->file_name = xmalloc(length * sizeof(char));;
  memcpy(inode->file_name, file_name, length);
 
  return inode->inode_number;
}


ino_t get_inode(const int fd)
{
  entry_ptr inode;


  for (inode = first_inode; inode; inode = inode->next_inode)
    if (inode->fd == fd)
      return inode->inode_number;

  return (ino_t)(-1);
}


void reset_descriptor(int fd)
{
  entry_ptr inode;


  for (inode = first_inode; inode; inode = inode->next_inode)
    if (inode->fd == fd)
    {
      inode->fd = -1;
      return;
    }
}


/*  Use DJGPP's stat function in stat_wrapper.  */
#undef stat

int stat_wrapper(const char *file_name, struct stat *stat_buf)
{
  int status;


  status = stat(file_name, stat_buf);
  stat_buf->st_ino = create_inode(file_name, -1);

  return status;
}


/*  Use DJGPP's fstat function in fstat_wrapper.  */
#undef fstat

int fstat_wrapper(int fd, struct stat *stat_buf)
{
  int status;
  ino_t inode_number;


  status = fstat(fd, stat_buf);
  if (isatty(fd))
    return status;

  if ((inode_number = get_inode(fd)) != (ino_t)(-1))
    stat_buf->st_ino = inode_number;

  return status;
}


/*  Use DJGPP's open function in open_wrapper.  */
#undef open

int open_wrapper(const char *file_name, int mode, ...)
{
  int fd, permissions = *(&mode + 1);


  fd = open(file_name, mode, permissions);
  create_inode(file_name, fd);

  return fd;
}


/*  Use DJGPP's close function in close_wrapper.  */
#undef close

int close_wrapper(int fd)
{
  int status;


  status = close(fd);
  reset_descriptor(fd);

  return status;
}


/*  Use DJGPP's fopen function in fopen_wrapper.  */
#undef fopen

FILE *fopen_wrapper(const char *file_name, const char *mode)
{
  FILE *fp;


  fp = fopen(file_name, mode);
  if (fp)
    create_inode(file_name, fileno(fp));

  return fp;
}


/*  Use DJGPP's fclose function in fclose_wrapper.  */
#undef fclose

int fclose_wrapper(FILE *fp)
{
  int fd, status;


  fd = fileno(fp);
  status = fclose(fp);
  reset_descriptor(fd);

  return status;
}
