// LEVEL-EDITOR v1.02 --- Copyright (c) Henrik Stokseth. 1999.


// --------------------------------------------------------------------------
// Include files

#include <stdio.h>
#include <ctype.h>
#include "allegro.h"
#include "tilemap.h"


// --------------------------------------------------------------------------
// Stuff that reduces EXE size

BEGIN_DIGI_DRIVER_LIST
END_DIGI_DRIVER_LIST

BEGIN_MIDI_DRIVER_LIST
END_MIDI_DRIVER_LIST

BEGIN_JOYSTICK_DRIVER_LIST
END_JOYSTICK_DRIVER_LIST


// --------------------------------------------------------------------------
// Definitions

#define LEDITOR_VERSION_STRING "LEVEL-EDITOR v1.02"
#define LEDITOR_LICENSE        "Freeware."
#define LEDITOR_COPYRIGHT      "Copyright 1999, Henrik Stokseth."

#define OK                     TRUE
#define NOT_OK                 FALSE

typedef struct undo
{
  struct undo *previous_entry;
  int         first_in_job;
  int         layer;
  int         x, y;
  int         sprite;
} UNDO;


// --------------------------------------------------------------------------
// Global variables

TILEMAP  *Map          = NULL;
BITMAP   *ScreenBuffer = NULL;
DATAFILE *Data         = NULL;
PALETTE  *EditPalette  = NULL;
UNDO     *UndoList     = NULL;

int EditSprite  = NO_SPRITE;
int FillSprite  = NO_SPRITE;
int EditLayer   = 0;
int EditX       = 0;
int EditY       = 0;
int FillFlood   = FALSE;
int UpdateTile  = TRUE;
int EditorExit  = FALSE;

int VisibleLayer[9] = {TRUE, TRUE, TRUE, TRUE, TRUE, TRUE, TRUE, TRUE, TRUE};

int WHITE, BLACK, GRAY, RED, GREEN, BLUE;


// --------------------------------------------------------------------------
// Function prototypes

void destroy_undo_list();
void add_to_undo_list(int x, int y, int first_in_job);
int  remove_from_undo_list(UNDO *entry);
int  undo();
void edit_sprite(int x, int y, int first_in_job);
void Wait();
int  InSquare(int X1, int Y1, int X2, int Y2);
void UnInitialize();
void Initialize();
void Welcome();
int  About();
void ToolPalette();
int  FillLayer();
int  Eraser();
int  Flooder();
void hfiller(int x, int y);
void vfiller(int x, int y);
void FloodFill();
int  MapInfo();
int  input_number(BITMAP *bmp, int x, int y, int max_digits);
void show_map();
void show_tile();
void scroll_map(int delta_x, int delta_y);
void SelectLayer(int layernum);
void process_keyboard_input();
void MainMenu();
int  QuitEditor();
void Editor();
int  NewMap();
int  LoadMap();
int  SaveMap();
int  ResizeMap();
void SelectPalette();
int  LoadSprites();
int  SelectSprite();
void Error(int ErrorNumber);


// --------------------------------------------------------------------------
// Menus

MENU tool_palette[] =
{
  { "TOOL PALETTE",        NULL,         NULL,        D_DISABLED },
  { "",                    NULL,         NULL                    },
  { "&Eraser",             Eraser,       NULL                    },
  { "Fill la&yer",         FillLayer,    NULL                    },
  { "&Flood Fill",         Flooder,      NULL                    },
  { "&Undo",               undo,         NULL                    },
  { "Tilemap &Info",       MapInfo,      NULL                    },
  { NULL,                  NULL,         NULL                    }
};

MENU main_menu[] =
{
  { "MAIN MENU",           NULL,         NULL,        D_DISABLED },
  { "",                    NULL,         NULL                    },
  { "&New tilemap",        NULL,         NULL                    },
  { "&Load tilemap",       NULL,         NULL                    },
  { "&Save tilemap",       NULL,         NULL                    },
  { "&Resize tilemap",     NULL,         NULL                    },
  { "Loa&d sprites",       NULL,         NULL                    },
  { "Selec&t sprite",      NULL,         NULL                    },
  { "Tool palette",        NULL,         tool_palette            },
  { "&About",              NULL,         NULL                    },
  { "",                    NULL,         NULL                    },
  { "E&xit editor",        NULL,         NULL                    },
  { NULL,                  NULL,         NULL                    }
};


// --------------------------------------------------------------------------
// The destroy_undo_list function

void destroy_undo_list()
{
  while(remove_from_undo_list(NULL));
}


// --------------------------------------------------------------------------
// The add_to_undo_list function

void add_to_undo_list(int x, int y, int first_in_job)
{
  UNDO *list_entry;

  list_entry                 = malloc(sizeof(UNDO)); if(!list_entry) Error(4);
  list_entry->first_in_job   = first_in_job;
  list_entry->layer          = EditLayer;
  list_entry->x              = x;
  list_entry->y              = y;
  list_entry->sprite         = get_sprite_number(Map, EditLayer, x, y);
  list_entry->previous_entry = UndoList;

  UndoList                   = list_entry;
}


// --------------------------------------------------------------------------
// The remove_from_undo_list function

int remove_from_undo_list(UNDO *entry)
{
  UNDO *list_entry;
  
  if(!UndoList) return FALSE;
  list_entry = UndoList;
  UndoList = UndoList->previous_entry;
  if(entry) *entry = *list_entry;
  free(list_entry);
  return TRUE;
}


// --------------------------------------------------------------------------
// The undo function

int undo()
{
  UNDO entry;
  
  while(remove_from_undo_list(&entry))
  {
    Map->data[get_sprite_pos(Map, entry.layer, entry.x, entry.y)] = entry.sprite;
    if(entry.first_in_job) return OK;
  }

  return OK;
}


// --------------------------------------------------------------------------
// The edit_sprite function

void edit_sprite(int x, int y, int first_in_job)
{
  if(Map && Data && (x+1) && (y+1) && (x<Map->xsize) && (y<Map->ysize))
  if(get_sprite_number(Map, EditLayer, x, y) != EditSprite)
  {
    add_to_undo_list(x, y, first_in_job);
    Map->data[get_sprite_pos(Map, EditLayer, x, y)] = EditSprite;
    if(UpdateTile) show_tile(x, y);
  }
}


// --------------------------------------------------------------------------
// The main function

int main()
{
  Initialize();
  Welcome();
  Editor();
  UnInitialize();

  return 0;
}


// --------------------------------------------------------------------------
// The wait function

void Wait()
{
  clear_keybuf();
  while(mouse_b);
  while(!keypressed()) if(mouse_b) break;
}


// --------------------------------------------------------------------------
// The insquare function

int InSquare(int X1, int Y1, int X2, int Y2)
{
  if((mouse_x>=X1) && (mouse_x<=X2) && (mouse_y>=Y1) && (mouse_y<=Y2)) return TRUE;
  else return FALSE;
}


// --------------------------------------------------------------------------
// The uninitialize function

void UnInitialize()
{
  destroy_tilemap(Map);
  Map = NULL;
  destroy_undo_list();
  unload_datafile(Data);
  destroy_bitmap(ScreenBuffer);
  set_gfx_mode(GFX_TEXT, 80, 25, 0, 0);
  printf("Bye, bye ...\n");
}


// --------------------------------------------------------------------------
// The initialize function

void Initialize()
{
  int card, w, h, coldepth, ok = FALSE;

  // initialize allegro
  allegro_init();
  if(!install_mouse()) Error(3);
  install_keyboard();
  install_timer();
  if(set_gfx_mode(GFX_VGA, 320, 200, 0, 0)) Error(2);
  set_palette(desktop_palette);

  // select screen mode
  while(!ok)
  {
    // show mode selection window
    if(!gfx_mode_select_ex(&card, &w, &h, &coldepth))
    {
      set_gfx_mode(GFX_TEXT, 80, 25, 0, 0);
      printf("Bye, bye ...\n");
      exit(0);
    }

    // make user aware of bad choise
    if(coldepth == 8) alert("ATTENTION:", "Please select a color-depth", "other than 8-bits.", "OK", "Shut up", 13, 27);
    if((w < 640) || (h < 400)) alert("ATTENTION:", "Please select a resolution", "of 640x400 or better.", "OK", "Shut up", 13, 27);

    // if the choises are ok then try the settings
    if((w >= 640) && (h >= 400) && (coldepth != 8))
    {
      set_color_depth(coldepth);
      if(!set_gfx_mode(card, w, h, 0, 0)) ok = TRUE;
    }
  }
  
  // make a double buffer
  ScreenBuffer = create_bitmap(SCREEN_W, SCREEN_H);
  if(!ScreenBuffer) Error(4);

  // define colors
  WHITE = makecol(255, 255, 255);
  BLACK = makecol(  0,   0,   0);
  GRAY  = makecol(127, 127, 127);
  RED   = makecol(255,   0,   0);
  GREEN = makecol(  0, 255,   0);
  BLUE  = makecol(  0,   0, 255);

  // init GUI colors
  gui_fg_color = BLACK;
  gui_bg_color = WHITE;
  gui_mg_color = BLUE;

  // initialize textmode and mouse
  show_mouse(screen);
  text_mode(-1);
}


// --------------------------------------------------------------------------
// The welcome function

void Welcome()
{
  clear_to_color(ScreenBuffer, GRAY);
  rectfill(ScreenBuffer, SCREEN_W/2-140, SCREEN_H/2-40, SCREEN_W/2+140, SCREEN_H/2+40, WHITE);
  rect(ScreenBuffer, SCREEN_W/2-140, SCREEN_H/2-40, SCREEN_W/2+140, SCREEN_H/2+40, BLACK);
  textout_centre(ScreenBuffer, font, LEDITOR_VERSION_STRING, SCREEN_W/2, SCREEN_H/2-24, BLACK);
  textout_centre(ScreenBuffer, font, LEDITOR_LICENSE, SCREEN_W/2, SCREEN_H/2-4, BLACK);
  textout_centre(ScreenBuffer, font, LEDITOR_COPYRIGHT, SCREEN_W/2, SCREEN_H/2+16, BLACK);
  textout(ScreenBuffer, font, "Press a button on the mouse or on the keyboard to continue ...", 10, SCREEN_H-20, GREEN);
  scare_mouse();
  blit(ScreenBuffer, screen, 0, 0, 0, 0, SCREEN_W, SCREEN_H);
  Wait();
  unscare_mouse();
}


// --------------------------------------------------------------------------
// The about function

int About()
{
  text_mode(-1);

  clear_to_color(ScreenBuffer, GRAY);
  rectfill(ScreenBuffer, 10, 10, SCREEN_W-10, SCREEN_H-10, WHITE);
  rect(ScreenBuffer, 10, 10, SCREEN_W-10, SCREEN_H-10, BLACK);
  textprintf_centre(ScreenBuffer, font, SCREEN_W/2, 20, RED, "%s %s %s", LEDITOR_VERSION_STRING, LEDITOR_LICENSE, LEDITOR_COPYRIGHT);
  textout_centre(ScreenBuffer, font, "Hello folks! Welcome to THE leveleditor for your tilebased games.         ", SCREEN_W/2, 40,  BLACK);
  textout_centre(ScreenBuffer, font, "See the README.TXT file for instructions on how to use this editor.       ", SCREEN_W/2, 50,  BLACK);
  textout_centre(ScreenBuffer, font, "If you want to contact me for any reason, please feel free to do so.      ", SCREEN_W/2, 70,  BLACK);
  textout_centre(ScreenBuffer, font, "              Snail mail : Henrik Stokseth                                ", SCREEN_W/2, 90,  BLUE);
  textout_centre(ScreenBuffer, font, "                           Porsevegen 18b                                 ", SCREEN_W/2, 100, BLUE);
  textout_centre(ScreenBuffer, font, "                           6011 lesund                                   ", SCREEN_W/2, 110, BLUE);
  textout_centre(ScreenBuffer, font, "                           Norway                                         ", SCREEN_W/2, 120, BLUE);
  textout_centre(ScreenBuffer, font, "                  E-mail : hstokset@hotmail.com                           ", SCREEN_W/2, 140, BLUE);
  textout_centre(ScreenBuffer, font, "                           hstokset@c2i.net                               ", SCREEN_W/2, 150, BLUE);
  textout_centre(ScreenBuffer, font, "       Official homepage : http://thunder.prohosting.com/~hstokset/       ", SCREEN_W/2, 170, BLUE);
  textout_centre(ScreenBuffer, font, "Thanks goes to DJ Delorie and Shawn Hargreaves for beeing so generous that", SCREEN_W/2, 190, BLACK);
  textout_centre(ScreenBuffer, font, "they actually give away powerful development tools for free.              ", SCREEN_W/2, 200, BLACK);
  textout_centre(ScreenBuffer, font, "How to use my tilemap editor:                                             ", SCREEN_W/2, 220, RED);
  textout_centre(ScreenBuffer, font, "Use the cursor keys to scroll around the tilemap inside the editor.       ", SCREEN_W/2, 240, BLACK);
  textout_centre(ScreenBuffer, font, "The left mouse button is used for drawing tiles on the map, the right     ", SCREEN_W/2, 250, BLACK);
  textout_centre(ScreenBuffer, font, "mouse button pops up the main menu. You can use the keys from 1 to 9      ", SCREEN_W/2, 260, BLACK);
  textout_centre(ScreenBuffer, font, "to select the current layer. The current layer is the one you place       ", SCREEN_W/2, 270, BLACK);
  textout_centre(ScreenBuffer, font, "the tiles on. That's it... is that simple or what? Other keys on the      ", SCREEN_W/2, 280, BLACK);
  textout_centre(ScreenBuffer, font, "keyboard can also be used. Please read the README.TXT for a complete list.", SCREEN_W/2, 290, BLACK);

  textout(ScreenBuffer, font, "Press a button on the mouse or on the keyboard to continue ...", 20, SCREEN_H-30, RED);
  scare_mouse();
  blit(ScreenBuffer, screen, 0, 0, 0, 0, SCREEN_W, SCREEN_H);
  Wait(); while(mouse_b);
  unscare_mouse();

  return OK;
}


// --------------------------------------------------------------------------
// The filllayer function

int FillLayer()
{
  int x, y;
  
  if(Map && Data)
  {
    UpdateTile = FALSE;
    for(x=0; x<Map->xsize; x++) for(y=0; y<Map->ysize; y++)
    edit_sprite(x, y, (!x && !y) ? TRUE : FALSE);
    UpdateTile = TRUE;
  }

  return OK;
}


// --------------------------------------------------------------------------
// The eraser function

int Eraser()
{
  EditSprite = NO_SPRITE;

  return OK;
}


// --------------------------------------------------------------------------
// The flooder function

int Flooder()
{
  FillFlood = TRUE;

  return OK;
}


// --------------------------------------------------------------------------
// The hfiller function

void hfiller(int startx, int y)
{
  int x    = startx;
  int maxx = startx;
  int minx = startx;

  while(++x < Map->xsize)
  {
    if(get_sprite_number(Map, EditLayer, x, y) == FillSprite)
      maxx = x;
    else
      break;
  }
  x = startx;
  while(--x >= 0)
  {
    if(get_sprite_number(Map, EditLayer, x, y) == FillSprite)
      minx = x;
    else
      break;
  }
  for(x = minx; x < startx; x++)
    edit_sprite(x, y, FALSE);
  for(x = startx+1; x <= maxx; x++)
    edit_sprite(x, y, FALSE);
  for(x = minx; x < startx; x++)
    vfiller(x, y);
  for(x = startx+1; x <= maxx; x++)
    vfiller(x, y);
}


// --------------------------------------------------------------------------
// The vfiller function

void vfiller(int x, int starty)
{
  int y    = starty;
  int maxy = starty;
  int miny = starty;

  while(++y < Map->ysize)
  {
    if(get_sprite_number(Map, EditLayer, x, y) == FillSprite)
      maxy = y;
    else
      break;
  }
  y = starty;
  while(--y >= 0)
  {
    if(get_sprite_number(Map, EditLayer, x, y) == FillSprite)
      miny = y;
    else
      break;
  }
  for(y = miny; y < starty; y++)
    edit_sprite(x, y, FALSE);
  for(y = starty+1; y <= maxy; y++)
    edit_sprite(x, y, FALSE);
  for(y = miny; y < starty; y++)
    hfiller(x, y);
  for(y = starty+1; y <= maxy; y++)
    hfiller(x, y);
}


// --------------------------------------------------------------------------
// The floodfill function

void FloodFill()
{
  int x, y;

  if(Map && Data)
  {
    x = EditX+mouse_x/Map->xgrid;
    y = EditY+mouse_y/Map->ygrid;
    if((x >= 0) && (y >= 0) && (x < Map->xsize) && (y < Map->ysize))
    {
      FillSprite = get_sprite_number(Map, EditLayer, x, y);
      if(FillSprite != EditSprite)
      {
        UpdateTile = FALSE;
        edit_sprite(x, y, TRUE);
        hfiller(x, y);
        vfiller(x, y);
        UpdateTile = TRUE;
      }
    }
  }
  FillFlood = FALSE;
}


// --------------------------------------------------------------------------
// The mapinfo function

int MapInfo()
{
  int c;

  if(Map && Data)
  {
    text_mode(-1);

    clear_to_color(ScreenBuffer, GRAY);
    rectfill(ScreenBuffer, 10, 10, SCREEN_W-10, SCREEN_H-10, WHITE);
    rect(ScreenBuffer, 10, 10, SCREEN_W-10, SCREEN_H-10, BLACK);
    textout_centre(ScreenBuffer, font, "TILEMAP INFORMATION :", SCREEN_W/2, 20, RED);

    textprintf(ScreenBuffer, font, 20, 40, BLACK, "TileMap Version     : %u", Map->version);
    textprintf(ScreenBuffer, font, 20, 50, BLACK, "TileMap X-Grid      : %u", Map->xgrid);
    textprintf(ScreenBuffer, font, 20, 60, BLACK, "TileMap Y-Grid      : %u", Map->ygrid);
    textprintf(ScreenBuffer, font, 20, 70, BLACK, "TileMap X-Size      : %u", Map->xsize);
    textprintf(ScreenBuffer, font, 20, 80, BLACK, "TileMap Y-Size      : %u", Map->ysize);
    textprintf(ScreenBuffer, font, 20, 90, BLACK, "TileMap Layers      : %u", Map->layers);

    textprintf(ScreenBuffer, font, 20, 110, BLACK, "Current Edit Layer  : %u", EditLayer+1);
    textprintf(ScreenBuffer, font, 20, 120, BLACK, "Current Edit Sprite : %s", (EditSprite == NO_SPRITE) ? "NONE / ERASER" : get_object_name(Data, EditSprite));

    textout(ScreenBuffer, font, "Visible layers      :", 20, 140, BLACK);
    for(c=0; c<Map->layers; c++) if(VisibleLayer[c]) textprintf(ScreenBuffer, font, 196+c*16, 140, BLACK, "%u", c+1);
    
    textout(ScreenBuffer, font, "Press a button on the mouse or on the keyboard to continue ...", 20, SCREEN_H-30, RED);
    scare_mouse();
    blit(ScreenBuffer, screen, 0, 0, 0, 0, SCREEN_W, SCREEN_H);
    Wait();
    while(mouse_b);
    unscare_mouse();
  }

  return OK;
}


// --------------------------------------------------------------------------
// The input_number function

int input_number(BITMAP *bmp, int x, int y, int max_digits)
{
  int  editpos = 0;
  char *tempstr;
  int  tempval = 0;
  int  tempkey;
  int  tempexit = FALSE;

  tempstr = malloc(max_digits+1);
  tempstr[0] = 0;
  
  while(!tempexit)
  {
    rectfill(bmp, x, y, x+(8*(max_digits+1)), y+8, gui_bg_color);
    textout(bmp, font, tempstr, x, y, gui_fg_color);
    rectfill(bmp, x+(8*editpos), y, x+(8*(editpos+1)), y+8, gui_fg_color);
    scare_mouse();
    blit(bmp, screen, 0, 0, 0, 0, SCREEN_W, SCREEN_H);
    unscare_mouse();
    tempkey = readkey();
    rectfill(bmp, x+(8*editpos), y, x+(8*(editpos+1)), y+8, gui_bg_color);
    if(isdigit(tempkey & 0xFF) && (editpos<max_digits))
    {
      tempstr[editpos] = (tempkey & 0xFF);
      editpos++;
      tempstr[editpos] = 0;
    }
    else if((tempkey >> 8 == KEY_BACKSPACE) && editpos)
    {
      editpos--;
      tempstr[editpos] = 0;
    }
    else if(tempkey >> 8 == KEY_ENTER)
    {
      tempstr[editpos] = 0;
      tempexit = TRUE;
    }
  }
  tempval = atoi(tempstr);
  free(tempstr);
  return tempval;
}


// --------------------------------------------------------------------------
// The show_map function

void show_map()
{
  int x, y, layer;

  clear_to_color(ScreenBuffer, GRAY);
  for(layer=0; layer<Map->layers; layer++) if(VisibleLayer[layer])
  {
    for(x=EditX; ((x-EditX)*Map->xgrid<SCREEN_W) && (x < Map->xsize); x++)
    for(y=EditY; ((y-EditY)*Map->ygrid<SCREEN_H) && (y < Map->ysize); y++)
    if(get_sprite_number(Map, layer, x, y) != NO_SPRITE)
    draw_rle_sprite(ScreenBuffer, Data[get_sprite_number(Map, layer, x, y)].dat, (x-EditX)*Map->xgrid, (y-EditY)*Map->ygrid);
  }
  scare_mouse();
  blit(ScreenBuffer, screen, 0, 0, 0, 0, SCREEN_W, SCREEN_H);
  unscare_mouse();
}


// --------------------------------------------------------------------------
// The show_tile function

void show_tile(int x, int y)
{
  int layer;

  scare_mouse();
  rectfill(screen, (x-EditX)*Map->xgrid, (y-EditY)*Map->ygrid, (x-EditX+1)*Map->xgrid-1, (y-EditY+1)*Map->ygrid-1, GRAY);
  for(layer=0; layer<Map->layers; layer++)
  if(get_sprite_number(Map, layer, x, y) != -1)
  draw_rle_sprite(screen, Data[get_sprite_number(Map, layer, x, y)].dat, (x-EditX)*Map->xgrid, (y-EditY)*Map->ygrid);
  unscare_mouse();
}


// --------------------------------------------------------------------------
// The scroll_map function

void scroll_map(int delta_x, int delta_y)
{
  if(Map && Data)
  {
    // Scroll horizontally
    if((EditX+delta_x < 0) || (Map->xsize < SCREEN_W/Map->xgrid))
      EditX = 0;
    else if(EditX+delta_x > Map->xsize-SCREEN_W/Map->xgrid)
      EditX = Map->xsize-SCREEN_W/Map->xgrid;
    else
      EditX += delta_x;
    
    // Scroll vertically
    if((EditY+delta_y < 0) || (Map->ysize < SCREEN_H/Map->ygrid))
      EditY = 0;
    else if(EditY+delta_y > Map->ysize-SCREEN_H/Map->ygrid)
      EditY = Map->ysize-SCREEN_H/Map->ygrid;
    else
      EditY += delta_y;
  }
}


// --------------------------------------------------------------------------
// The selectlayer function

void SelectLayer(int layernum)
{
  if(Map && Data && (layernum <= Map->layers)) EditLayer = layernum;
}


// --------------------------------------------------------------------------
// The process_keyboard_input function

void process_keyboard_input()
{
  int tempkey;

  if(keypressed())
  {
    tempkey = readkey();
    switch(tempkey >> 8)
    {
      // scroll
      case KEY_UP:
        scroll_map(0, (key_shifts & KB_CTRL_FLAG) ? -SCREEN_H/Map->ygrid : -1);
      break;
      case KEY_DOWN:
        scroll_map(0, (key_shifts & KB_CTRL_FLAG) ? SCREEN_H/Map->ygrid : 1);
      break;
      case KEY_LEFT:
        scroll_map((key_shifts & KB_CTRL_FLAG) ? -SCREEN_W/Map->xgrid : -1, 0);
      break;
      case KEY_RIGHT:
        scroll_map((key_shifts & KB_CTRL_FLAG) ? SCREEN_W/Map->xgrid : 1, 0);
      break;

      // select current edit layer or turn on/off layers
      case KEY_1:
        (key_shifts & KB_CTRL_FLAG) ? (VisibleLayer[0] = VisibleLayer[0] ? FALSE : TRUE) : SelectLayer(0);
      break;
      case KEY_2:
        (key_shifts & KB_CTRL_FLAG) ? (VisibleLayer[1] = VisibleLayer[1] ? FALSE : TRUE) : SelectLayer(1);
      break;
      case KEY_3:
        (key_shifts & KB_CTRL_FLAG) ? (VisibleLayer[2] = VisibleLayer[2] ? FALSE : TRUE) : SelectLayer(2);
      break;
      case KEY_4:
        (key_shifts & KB_CTRL_FLAG) ? (VisibleLayer[3] = VisibleLayer[3] ? FALSE : TRUE) : SelectLayer(3);
      break;
      case KEY_5:
        (key_shifts & KB_CTRL_FLAG) ? (VisibleLayer[4] = VisibleLayer[4] ? FALSE : TRUE) : SelectLayer(4);
      break;
      case KEY_6:
        (key_shifts & KB_CTRL_FLAG) ? (VisibleLayer[5] = VisibleLayer[5] ? FALSE : TRUE) : SelectLayer(5);
      break;
      case KEY_7:
        (key_shifts & KB_CTRL_FLAG) ? (VisibleLayer[6] = VisibleLayer[6] ? FALSE : TRUE) : SelectLayer(6);
      break;
      case KEY_8:
        (key_shifts & KB_CTRL_FLAG) ? (VisibleLayer[7] = VisibleLayer[7] ? FALSE : TRUE) : SelectLayer(7);
      break;
      case KEY_9:
        (key_shifts & KB_CTRL_FLAG) ? (VisibleLayer[8] = VisibleLayer[8] ? FALSE : TRUE) : SelectLayer(8);
      break;

      // editor hotkeys
      case KEY_X:
      case KEY_ESC:
        QuitEditor();
      break;
      case KEY_A:
        About();
      break;
      case KEY_D:
        LoadSprites();
      break;
      case KEY_N:
        NewMap();
      break;
      case KEY_S:
        SaveMap();
      break;
      case KEY_L:
        LoadMap();
      break;
      case KEY_R:
        ResizeMap();
      break;
      case KEY_E:
        EditSprite = NO_SPRITE;
      break;
      case KEY_T:
        SelectSprite();
      break;
      case KEY_I:
        MapInfo();
      break;
      case KEY_F:
        FillFlood = TRUE;
      break;
      case KEY_Y:
        FillLayer();
      break;
      case KEY_U:
        undo();
      break;
    }
  }
}


// --------------------------------------------------------------------------
// The quiteditor function

int QuitEditor()
{
  if(alert("Exit LEVEL-EDITOR:", NULL, "Are you sure?", "Yes", "No", 'Y', 'N') == 1) EditorExit = TRUE;
  text_mode(-1);

  return OK;
}


// --------------------------------------------------------------------------
// The mainmenu function

void MainMenu()
{
  int menu_ret_value;

  menu_ret_value = do_menu(main_menu, mouse_x, mouse_y);

  switch(menu_ret_value)
  {
    case  2: NewMap();       break;
    case  3: LoadMap();      break;
    case  4: SaveMap();      break;
    case  5: ResizeMap();    break;
    case  6: LoadSprites();  break;
    case  7: SelectSprite(); break;
    case  9: About();        break;
    case 11: QuitEditor();   break;
  }
  
  text_mode(-1);
}


// --------------------------------------------------------------------------
// The editor function

void Editor()
{
  while(!EditorExit)
  {
    // output screen
    if(Map && Data)
    {
      show_map();
    }
    else if(!Map && !Data)
    {
      clear_to_color(ScreenBuffer, GRAY);
      textout_centre(ScreenBuffer, font, "Neither tilemap nor sprites loaded ...", SCREEN_W/2, SCREEN_H/2, BLACK);
      textout_centre(ScreenBuffer, font, "Press the right mouse button to bring up the menu.", SCREEN_W/2, SCREEN_H/2+20, BLACK);
      textout_centre(ScreenBuffer, font, "Use the left mouse button to place the tiles on the map.", SCREEN_W/2, SCREEN_H/2+30, BLACK);
      scare_mouse();
      blit(ScreenBuffer, screen, 0, 0, 0, 0, SCREEN_W, SCREEN_H);
      unscare_mouse();
    }
    else if(!Data)
    {
      clear_to_color(ScreenBuffer, GRAY);
      textout_centre(ScreenBuffer, font, "No sprites loaded ...", SCREEN_W/2, SCREEN_H/2, BLACK);
      textout_centre(ScreenBuffer, font, "Press the right mouse button to bring up the menu.", SCREEN_W/2, SCREEN_H/2+20, BLACK);
      textout_centre(ScreenBuffer, font, "Use the left mouse button to place the tiles on the map.", SCREEN_W/2, SCREEN_H/2+30, BLACK);
      scare_mouse();
      blit(ScreenBuffer, screen, 0, 0, 0, 0, SCREEN_W, SCREEN_H);
      unscare_mouse();
    }
    else if(!Map)
    {
      clear_to_color(ScreenBuffer, GRAY);
      textout_centre(ScreenBuffer, font, "No tilemap loaded ...", SCREEN_W/2, SCREEN_H/2, BLACK);
      textout_centre(ScreenBuffer, font, "Press the right mouse button to bring up the menu.", SCREEN_W/2, SCREEN_H/2+20, BLACK);
      textout_centre(ScreenBuffer, font, "Use the left mouse button to place the tiles on the map.", SCREEN_W/2, SCREEN_H/2+30, BLACK);
      scare_mouse();
      blit(ScreenBuffer, screen, 0, 0, 0, 0, SCREEN_W, SCREEN_H);
      unscare_mouse();
    }

    // wait for user input
    Wait();

    // keyboard input
    process_keyboard_input();

    // left mouse button
    if((mouse_b & 1) && Map && Data)
    {
      if(FillFlood) FloodFill();
      else
      {
        edit_sprite(EditX+mouse_x/Map->xgrid, EditY+mouse_y/Map->ygrid, TRUE);
        while(mouse_b & 1)
        edit_sprite(EditX+mouse_x/Map->xgrid, EditY+mouse_y/Map->ygrid, FALSE);
      }
    }

    // right mouse button
    if(mouse_b & 2) MainMenu();
  }
}


// --------------------------------------------------------------------------
// The newmap function

int NewMap()
{
  int ok = FALSE;
  int layers = 0, xsize = 0, ysize = 0, xgrid = 0, ygrid = 0;
  
  text_mode(-1);
  if(Map)
  if(alert("Create a new tilemap:", "Your current will be lost.", "Are you sure?", "Yes", "No", 'Y', 'N') == 2) return NOT_OK;
  text_mode(-1);

  destroy_undo_list();
  destroy_tilemap(Map);
  Map = NULL;

  while(!ok)
  {
    // write background
    clear_to_color(ScreenBuffer, GRAY);
    rectfill(ScreenBuffer, 10, 10, SCREEN_W-10, SCREEN_H-10, WHITE);
    rect(ScreenBuffer, 10, 10, SCREEN_W-10, SCREEN_H-10, BLACK);
    textout_centre(ScreenBuffer, font, "Please specify tilemap properties ...", SCREEN_W/2, 20, BLACK);

    // make new map
    textout(ScreenBuffer, font, "X-Grid:", 20, 40, BLACK);
    xgrid  = input_number(ScreenBuffer, 80, 40, 3);
    textout(ScreenBuffer, font, "Y-Grid:", 20, 50, BLACK);
    ygrid  = input_number(ScreenBuffer, 80, 50, 3);
    textout(ScreenBuffer, font, "X-Size:", 20, 60, BLACK);
    xsize  = input_number(ScreenBuffer, 80, 60, 4);
    textout(ScreenBuffer, font, "Y-Size:", 20, 70, BLACK);
    ysize  = input_number(ScreenBuffer, 80, 70, 4);
    textout(ScreenBuffer, font, "Layers:", 20, 80, BLACK);
    layers = input_number(ScreenBuffer, 80, 80, 1);

    // check if input data is okay
    if(xgrid && ygrid && xsize && ysize && layers)
    {
      ok = TRUE;
    }
    else
    {
      textout(screen, font, "You didn't specify the numbers right, please try again ...", 20, SCREEN_H-30, RED);
      Wait();
      clear_keybuf();
    }
  }
  
  // create a new map
  Map = create_tilemap(layers, xsize, ysize, xgrid, ygrid);

  // make some corrections
  EditLayer  = 0;
  EditX      = 0;
  EditY      = 0;
  EditSprite = NO_SPRITE;

  return OK;
}


// --------------------------------------------------------------------------
// The loadmap function

int LoadMap()
{
  char filename[MAX_STRING_LENGTH] = "";

  if(file_select("Load MAP:", filename, "MAP"))
  {
    if(Map && Data)
    {
      unload_datafile(Data);
      Data = NULL;
    }
    destroy_tilemap(Map);
    Map = NULL;
    destroy_undo_list();
    Map = load_tilemap(filename);
    if(Map->version > 1)
    {
      alert("WARNING:", "", "Unsupported tilemap version.", "Ok", NULL, 13, 27);
      destroy_tilemap(Map);
      Map = NULL;
    }
    text_mode(-1);
    
    // correct stuff
    if(Map && Data)
    {
      if(!Map->names)
      {
      	create_namelist(Map, Data);
      }
      else
      {
        correct_tilemap(Map, Data);
        create_namelist(Map, Data);
      }
    }

    // make some corrections
    EditLayer  = 0;
    EditX      = 0;
    EditY      = 0;
    EditSprite = NO_SPRITE;
  }

  return OK;
}


// --------------------------------------------------------------------------
// The savemap function

int SaveMap()
{
  char filename[MAX_STRING_LENGTH] = "";

  if(Map && file_select("Save MAP:", filename, "MAP"))
  save_tilemap(Map, filename);

  return OK;
}


// --------------------------------------------------------------------------
// The resizemap function

int ResizeMap()
{
  TILEMAP *tempmap;

  int     xgrid, ygrid, xsize, ysize, layers;
  int     xcut = FALSE, ycut = FALSE;
  int     lcount, xcount, ycount;
  int     ok = FALSE;
  int     tempkey;
  char    tempstr[MAX_STRING_LENGTH];
  char    buffer[MAX_STRING_LENGTH];

  if(Map && Data)
  {
    text_mode(-1);

    while(!ok)
    {
      // write background
      clear_to_color(ScreenBuffer, GRAY);
      rectfill(ScreenBuffer, 10, 10, SCREEN_W-10, SCREEN_H-10, WHITE);
      rect(ScreenBuffer, 10, 10, SCREEN_W-10, SCREEN_H-10, BLACK);
      textout_centre(ScreenBuffer, font, "Please enter new TILEMAP preferences ...", SCREEN_W/2, 20, BLACK);
      scare_mouse();
      blit(ScreenBuffer, screen, 0, 0, 0, 0, SCREEN_W, SCREEN_H);
      unscare_mouse();

      // read in the new values
      textprintf(ScreenBuffer, font, 20, 40, BLACK, "X-Grid (%u) : ", Map->xgrid);
      xgrid  = input_number(ScreenBuffer, 150, 40, 3);
      textprintf(ScreenBuffer, font, 20, 50, BLACK, "Y-Grid (%u) : ", Map->ygrid);
      ygrid  = input_number(ScreenBuffer, 150, 50, 3);
      textprintf(ScreenBuffer, font, 20, 60, BLACK, "X-Size (%u) : ", Map->xsize);
      xsize  = input_number(ScreenBuffer, 150, 60, 4);
      textprintf(ScreenBuffer, font, 20, 70, BLACK, "Y-Size (%u) : ", Map->ysize);
      ysize  = input_number(ScreenBuffer, 150, 70, 4);
      textprintf(ScreenBuffer, font, 20, 80, BLACK, "Layers (%u) : ", Map->layers);
      layers = input_number(ScreenBuffer, 150, 80, 1);

      scare_mouse();
      blit(ScreenBuffer, screen, 0, 0, 0, 0, SCREEN_W, SCREEN_H);
      unscare_mouse();
      
      if(xsize < Map->xsize)
      {
        textout(screen, font, "Your new X-Size is smaller than the one you're currently using.", 20, 100, BLACK);
        textout(screen, font, "Shall I cut away the left or the right side of the tilemap? (L/R) :", 20, 110, BLACK);
        tempkey = 0;
        while((tempkey != KEY_R) && (tempkey != KEY_L)) tempkey = readkey() >> 8;
        xcut = (tempkey == KEY_L) ? TRUE : FALSE;
      }

      if(ysize < Map->ysize)
      {
        textout(screen, font, "Your new Y-Size is smaller than the one you're currently using.", 20, 120, BLACK);
        textout(screen, font, "Shall I cut away the upper or the lower part of the tilemap? (U/L) :", 20, 130, BLACK);
        tempkey = 0;
        while((tempkey != KEY_U) && (tempkey != KEY_L)) tempkey = readkey() >> 8;
        ycut = (tempkey == KEY_U) ? TRUE : FALSE;
      }
      
      // check if input data is okay
      if(xgrid && ygrid && xsize && ysize && layers)
      {
        ok = TRUE;
      }
      else
      {
        textout(screen, font, "You didn't specify the numbers right, please try again ...", 20, SCREEN_H-30, RED);
        Wait();
        xcut = FALSE;
        ycut = FALSE;
        clear_keybuf();
      }
      
    }

    // make a new resized tilemap
    Map = resize_tilemap(Map, layers, xsize, ysize, xgrid, ygrid, xcut, ycut);
    destroy_undo_list();

    // make some corrections
    EditLayer  = 0;
    EditX      = 0;
    EditY      = 0;
  }

  return OK;
}


// --------------------------------------------------------------------------
// The selectpalette function

void SelectPalette()
{
  int counter;
  int ok = FALSE;

  text_mode(-1);

  while(!ok)
  {
    // list up palettes
    clear_to_color(ScreenBuffer, GRAY);
    rectfill(ScreenBuffer, 10, 10, SCREEN_W-10, SCREEN_H-10, WHITE);
    rect(ScreenBuffer, 10, 10, SCREEN_W-10, SCREEN_H-10, BLACK);
    textout_centre(ScreenBuffer, font, "Choose the palette you will be using ...", SCREEN_W/2, 20, BLACK);

    for(counter=1; counter <= number_of_type(Data, DAT_PALETTE); counter++)
    textout(ScreenBuffer, font, get_object_name(Data, get_object_number(Data, counter, DAT_PALETTE)), 20, 30+counter*10, BLACK);

    scare_mouse();
    blit(ScreenBuffer, screen, 0, 0, 0, 0, SCREEN_W, SCREEN_H);
    unscare_mouse();

    // wait for user input
    Wait();

    // left mouse button
    if(mouse_b & 1)
    for(counter=1; counter <= number_of_type(Data, DAT_PALETTE); counter++)
    if(InSquare(20, 30+counter*10, 20+strlen(get_object_name(Data, get_object_number(Data, counter, DAT_PALETTE)))*8, 40+counter*10))
    {
      ok = TRUE;
      EditPalette = Data[get_object_number(Data, counter, DAT_PALETTE)].dat;
    }

    // check if palette is not selected
    if(!ok)
    {
      textout(screen, font, "Please try again...", 20, SCREEN_H-30, RED);
      Wait();
    }
  }
}


// --------------------------------------------------------------------------
// The loadsprites function

int LoadSprites()
{
  char filename[MAX_STRING_LENGTH] = "";

  if(file_select("Load Sprites:", filename, "DAT"))
  {
    // load the datafile
    if(Map && Data)
    {
      destroy_tilemap(Map);
      Map = NULL;
    }
    if(Data)
    {
      unload_datafile(Data);
      Data = NULL;
    }
    
    Data = load_datafile(filename);
    
    // correct stuff
    if(Map && Data)
    {
      if(!Map->names)
      {
      	create_namelist(Map, Data);
      }
      else
      {
        correct_tilemap(Map, Data);
        create_namelist(Map, Data);
      }
    }

    // if neccesary load a palette
    if(object_of_type_exists(Data, DAT_PALETTE))
    {
      SelectPalette();
      select_palette(*EditPalette);
      unload_datafile(Data);
      Data = load_datafile(filename);
      unselect_palette();
    }
    text_mode(-1);
  }

  return OK;
}


// --------------------------------------------------------------------------
// The selectsprite function

int SelectSprite()
{
  RLE_SPRITE *sprite;
  int        sprites, spritesperline, maxwidth, maxheight, x, y;
  int        offset = 1, spritenum, finished  = FALSE;
  
  if(!Data) return NOT_OK;

  sprites        = number_of_type(Data, DAT_RLE_SPRITE);
  maxwidth       = get_max_sprite_width(Data);
  maxheight      = get_max_sprite_height(Data);
  spritesperline = (SCREEN_W-40)/(maxwidth+5);

  text_mode(-1);
  
  while(!finished)
  {
    // write background
    clear_to_color(ScreenBuffer, GRAY);
    rectfill(ScreenBuffer, 10, 10, SCREEN_W-10, SCREEN_H-10, WHITE);
    rect(ScreenBuffer, 10, 10, SCREEN_W-10, SCREEN_H-10, BLACK);
    textout_centre(ScreenBuffer, font, "Choose the sprite you want to use ...", SCREEN_W/2, 20, BLACK);
    textout(ScreenBuffer, font, "Use the arrow keys to navigate thru multiple screens ...", 20, SCREEN_H-30, BLACK);

    // display one page
    spritenum = offset;
    x = 20;
    y = 40;
    while(spritenum<=sprites)
    {
      // draw the current sprite
      sprite = Data[get_object_number(Data, spritenum, DAT_RLE_SPRITE)].dat;
      rect(ScreenBuffer, x-2, y-2, x+sprite->w+1, y+sprite->h+1, BLACK);
      draw_rle_sprite(ScreenBuffer, sprite, x, y);

      // make ready for the next sprite
      spritenum++;
      x+= maxwidth+5;
      if(x>=SCREEN_W-maxwidth-25)
      {
        x = 20;
        y+= maxheight+5;
      }
      if(y>=SCREEN_H-maxheight-25) break;
    }
    
    // update screen
    scare_mouse();
    blit(ScreenBuffer, screen, 0, 0, 0, 0, SCREEN_W, SCREEN_H);
    unscare_mouse();
    
    // await user input
    Wait();
    if(key[KEY_ESC]) finished = TRUE;
    if(key[KEY_UP] || key[KEY_LEFT])
    {
      if(offset-spritesperline>=1) offset-=spritesperline;
    }
    if(key[KEY_DOWN] || key[KEY_RIGHT])
    {
      if(offset+spritesperline<=sprites) offset+=spritesperline;
    }

    // left mouse button
    if(mouse_b & 1)
    {
      spritenum = offset;
      x = 20;
      y = 40;
      while(mouse_b & 1);
      
      while(get_object_number(Data, spritenum, DAT_RLE_SPRITE) != get_object_number(Data, spritenum+1, DAT_RLE_SPRITE))
      {
        sprite = Data[get_object_number(Data, spritenum, DAT_RLE_SPRITE)].dat;
        if(InSquare(x-2, y-2, x+sprite->w+1, y+sprite->h+1))
        EditSprite = get_object_number(Data, spritenum, DAT_RLE_SPRITE);

        // make ready for the next sprite
        spritenum++;
        x+= maxwidth+5;
        if(x>=SCREEN_W-maxwidth-25)
        {
          x = 20;
          y+= maxheight+5;
        }
        if(y>=SCREEN_H-maxheight-25) break;
      }
      finished = TRUE;
    }

  }

  return OK;
}


// --------------------------------------------------------------------------
// The error function

void Error(int ErrorNumber)
{
  set_gfx_mode(GFX_TEXT, 80, 25, 0, 0);
  printf("\nERROR %u: ", ErrorNumber);
  switch(ErrorNumber) {
    case 1:
      printf("Error reading datafile...\n");
    break;
    case 2:
      printf("Can't set the proper video mode (%s)\n", allegro_error);
    break;
    case 3:
      printf("Mouse driver not loaded...");
    break;
    case 4:
      printf("Memory allocation error...\n");
    break;
    default:
      printf("Undetermined error...\n");
    break;
  }
  exit(ErrorNumber);
}


// --------------------------------------------------------------------------
// End of program

