// --------------------------------------------------------------------------
// Level-Editor v1.02 --- Tilemap helper functions.
// Copyright (c) Henrik Stokseth, 1999.


#ifndef __MAPFILE__
#define __MAPFILE__

#ifdef __cplusplus
extern "C" {
#endif // __cplusplus


// --------------------------------------------------------------------------
// Included files

#include <string.h>
#include <stdio.h>


// --------------------------------------------------------------------------
// Definitions

#define TILEMAP_VERSION    1
#define TILEMAP_DATA_SIZE  (map->xsize * map->ysize * map->layers)
#define TILEMAP_LAYER_SIZE (map->xsize * map->ysize)
#define NAMELIST_SIZE      (sizeof(NAMELIST)*map->names)
#define NO_SPRITE          -1
#define MAX_STRING_LENGTH  80

// --------------------------------------------------------------------------
// Structure definitions

typedef struct NAMELIST
{
  int  index;                   // the sprite's datafile index number
  char name[MAX_STRING_LENGTH]; // the sprite's datafile name
} NAMELIST;

typedef struct INDEXLIST
{
  int from; // old datafile index number
  int to;   // new datafile index number
} INDEXLIST;

typedef struct TILEMAP
{
  int      version;      // map-file version
  int      xsize, ysize; // size of map measured in tiles
  int      xgrid, ygrid; // size of grid measured in pixels
  int      layers;       // number of layers
  int      names;        // number of sprite-names (used for auto-correction)
  int      *data;        // map data
  NAMELIST *name;        // sprite names (used for auto-correction)
} TILEMAP;


// --------------------------------------------------------------------------
// Function prototypes

// datafile helper functions  (mostly for internal use)
char *get_object_name(DATAFILE *dat, int objnum);
int  object_of_type_exists(DATAFILE *dat, int type);
int  number_of_type(DATAFILE *dat, int type);
int  get_object_number(DATAFILE *dat, int number_of_type, int type);
int  get_max_sprite_width(DATAFILE *dat);
int  get_max_sprite_height(DATAFILE *dat);

// tilemap helper functions
TILEMAP *load_tilemap(char *filename);
TILEMAP *register_tilemap(DATAFILE *dat, int index);
void    save_tilemap(TILEMAP *map, char *filename);
TILEMAP *create_tilemap(int layers, int xsize, int ysize, int xgrid, int ygrid);
TILEMAP *resize_tilemap(TILEMAP *map, int layers, int xsize, int ysize, int xgrid, int ygrid, int xcut, int ycut);
void    destroy_tilemap(TILEMAP *map);

// drawing helper functions
void draw_tilemap_layer(BITMAP *dest_bmp, DATAFILE *dat, TILEMAP *map, int layer, int dest_x, int dest_y);

// tile helper functions
inline int  get_sprite_number(TILEMAP *map, int layer, int x, int y);
inline void set_sprite_number(TILEMAP *map, int layer, int x, int y, int sprite_num);
inline int  get_sprite_pos(TILEMAP *map, int layer, int x, int y);

// auto-correction helper functions
void create_namelist(TILEMAP *map, DATAFILE *dat);
void correct_tilemap(TILEMAP *map, DATAFILE *dat);


// --------------------------------------------------------------------------
// The get_object_name function

char *get_object_name(DATAFILE *dat, int objnum)
{
  int counter;

  // make sure the object is there
  if(objnum<0) return NULL;
  for(counter=0; dat[counter].type != DAT_END; counter++);
  if(objnum>counter) return NULL;
  
  // return name
  for(counter=0; dat[objnum].prop[counter].type != DAT_END; counter++)
  {
    if(dat[objnum].prop[counter].type == DAT_NAME)
    return dat[objnum].prop[counter].dat;
  }    
  return NULL;
}


// --------------------------------------------------------------------------
// The object_of_type_exists function

int object_of_type_exists(DATAFILE *dat, int type)
{
  int counter;

  for(counter=0; dat[counter].type != DAT_END; counter++)
  {
    if(dat[counter].type == type) return TRUE;
  }
  return FALSE;
}


// --------------------------------------------------------------------------
// The number_of_type function

int number_of_type(DATAFILE *dat, int type)
{
  int counter, counter2 = 0;

  for(counter=0; dat[counter].type != DAT_END; counter++)
  if(dat[counter].type == type) counter2++;
  return counter2;
}


// --------------------------------------------------------------------------
// The get_object_number function

int get_object_number(DATAFILE *dat, int number_of_type, int type)
{
  int counter, counter2 = 0;

  // if found
  for(counter=0; dat[counter].type != DAT_END; counter++)
  {
    if(dat[counter].type == type) counter2++;
    if(number_of_type == counter2) return counter;
  }

  // if not found
  return NO_SPRITE;
}


// --------------------------------------------------------------------------
// The get_max_sprite_width function

int get_max_sprite_width(DATAFILE *dat)
{
  RLE_SPRITE *sprite = NULL;
  int        counter;
  int        tempx = 0;

  for(counter=0; dat[counter].type != DAT_END; counter++)
  {
    if(dat[counter].type == DAT_RLE_SPRITE)
    {
      sprite = dat[counter].dat;
      if(sprite->w > tempx) tempx = sprite->w;
    }
  }
  return tempx;
}


// --------------------------------------------------------------------------
// The get_max_sprite_height function

int get_max_sprite_height(DATAFILE *dat)
{
  RLE_SPRITE *sprite = NULL;
  int        counter;
  int        tempy = 0;

  for(counter=0; dat[counter].type != DAT_END; counter++)
  {
    if(dat[counter].type == DAT_RLE_SPRITE)
    {
      sprite = dat[counter].dat;
      if(sprite->h > tempy) tempy = sprite->h;
    }
  }
  return tempy;
}


// --------------------------------------------------------------------------
// The load_tilemap function

TILEMAP *load_tilemap(char *filename)
{
  FILE    *mapfile;
  TILEMAP *map;
  int     count;

  mapfile = fopen(filename, "rb");

  map = malloc(sizeof(TILEMAP));
  fread(map, sizeof(TILEMAP), 1, mapfile);

  map->data = malloc(TILEMAP_DATA_SIZE*sizeof(int));
  fread(map->data, sizeof(int), TILEMAP_DATA_SIZE, mapfile);

  if(map->names)
  {
    map->name = malloc(NAMELIST_SIZE);
    for(count=0; count<map->names; count++)
    {
      fread(&map->name[count].index, sizeof(int), 1, mapfile);
      fgets(map->name[count].name, MAX_STRING_LENGTH, mapfile);
      map->name[count].name[strlen(map->name[count].name)-1] = 0;
    }
  }

  fclose(mapfile);

  return map;
}


// --------------------------------------------------------------------------
// The register_tilemap function

TILEMAP *register_tilemap(DATAFILE *dat, int index)
{
  TILEMAP *map = dat[index].dat;
  char    *list_ptr;
  int     count;

  // initialize tilemap for use with allegro
  map->data = (int*)  ((char*) map+sizeof(TILEMAP));
  list_ptr  = (char*) ((char*) map+sizeof(TILEMAP)+TILEMAP_DATA_SIZE*sizeof(int));

  if(map->names)
  {
    map->name = malloc(NAMELIST_SIZE);
    for(count=0; count<map->names; count++)
    {
      map->name[count].index = *list_ptr;
      list_ptr += sizeof(int);
      list_ptr[strcspn(list_ptr, "\n")] = 0;
      strcpy(map->name[count].name, list_ptr);
      list_ptr += strlen(list_ptr)+1;
    }
  }

  return map;
}


// --------------------------------------------------------------------------
// The save_tilemap function

void save_tilemap(TILEMAP *map, char *filename)
{
  FILE *mapfile;
  int  count;

  mapfile = fopen(filename, "wb");

  fwrite(map, sizeof(TILEMAP), 1, mapfile);
  fwrite(map->data, sizeof(int), TILEMAP_DATA_SIZE, mapfile);

  for(count=0; count<map->names; count++)
  {
    fwrite(&(map->name[count].index), sizeof(int), 1, mapfile);
    fputs(map->name[count].name, mapfile);
    fputc('\n', mapfile);
  }
  
  fclose(mapfile);
}


// --------------------------------------------------------------------------
// The create_tilemap function

TILEMAP *create_tilemap(int layers, int xsize, int ysize, int xgrid, int ygrid)
{
  TILEMAP *map;
  int     counter;

  // allocate stuff
  map          = malloc(sizeof(TILEMAP));
  map->version = TILEMAP_VERSION;
  map->xgrid   = xgrid;
  map->ygrid   = ygrid;
  map->xsize   = xsize;
  map->ysize   = ysize;
  map->layers  = layers;
  map->names   = 0;
  map->data    = malloc(TILEMAP_DATA_SIZE*sizeof(int));
  map->name    = NULL;

  // clear tilemap
  for(counter=0; counter<TILEMAP_DATA_SIZE; counter++)
  map->data[counter] = NO_SPRITE;

  return map;
}


// --------------------------------------------------------------------------
// The resize_tilemap function

TILEMAP *resize_tilemap(TILEMAP *map, int layers, int xsize, int ysize, int xgrid, int ygrid, int xcut, int ycut)
{
  TILEMAP *tempmap;
  int     lcount, xcount, ycount;

  if(map)
  {
    // make a new resized tilemap
    tempmap = create_tilemap(layers, xsize, ysize, xgrid, ygrid);

    for(lcount=0; lcount<tempmap->layers; lcount++)
    for(xcount=0; xcount<tempmap->xsize ; xcount++)
    for(ycount=0; ycount<tempmap->ysize ; ycount++)
    {
      tempmap->data[get_sprite_pos(tempmap, lcount, xcount, ycount)] =
      get_sprite_number(map, lcount,
      (xcut) ? (xcount+map->xsize-xsize) : xcount,
      (ycut) ? (ycount+map->ysize-ysize) : ycount);
    }
    
    destroy_tilemap(map);
    return tempmap;
  }
  else return NULL;
}


// --------------------------------------------------------------------------
// The destroy_tilemap function

void destroy_tilemap(TILEMAP *map)
{
  if(map)
  {
    if(map->data) free(map->data);
    if(map->name) free(map->name);
    free(map);
  }
}


// --------------------------------------------------------------------------
// The draw_tilemap_layer function

void draw_tilemap_layer(BITMAP *dest_bmp, DATAFILE *dat, TILEMAP *map, int layer, int dest_x, int dest_y)
{
  int x, y;
  
  for(x = dest_x>0 ? 0 : -dest_x/map->xgrid; (x*map->xgrid+dest_x<dest_bmp->w) && (x<map->xsize); x++)
  for(y = dest_y>0 ? 0 : -dest_y/map->ygrid; (y*map->ygrid+dest_y<dest_bmp->h) && (y<map->ysize); y++)
  if(get_sprite_number(map, layer, x, y) != NO_SPRITE)
  draw_rle_sprite(dest_bmp, dat[get_sprite_number(map, layer, x, y)].dat, dest_x+x*map->xgrid, dest_y+y*map->ygrid);
}


// --------------------------------------------------------------------------
// The get_sprite_number function

inline int get_sprite_number(TILEMAP *map, int layer, int x, int y)
{
  if((layer < map->layers) && (x < map->xsize) && (y < map->ysize))
    return map->data[get_sprite_pos(map, layer, x, y)];
  else
    return NO_SPRITE;
}


// --------------------------------------------------------------------------
// The set_sprite_number function

inline void set_sprite_number(TILEMAP *map, int layer, int x, int y, int sprite_num)
{
  if((layer < map->layers) && (x < map->xsize) && (y < map->ysize))
  map->data[get_sprite_pos(map, layer, x, y)] = sprite_num;
}


// --------------------------------------------------------------------------
// The get_sprite_pos function

inline int get_sprite_pos(TILEMAP *map, int layer, int x, int y)
{
  return (x+y*map->xsize) + (map->xsize*map->ysize*layer);
}


// --------------------------------------------------------------------------
// The create_namelist function

void create_namelist(TILEMAP *map, DATAFILE *dat)
{
  int count, names;

  // count up how many sprites the datafile contains
  names = 0;
  for(count = 0; dat[count].type != DAT_END; count++)
  if(dat[count].type == DAT_RLE_SPRITE) names++;

  // create namelist
  map->names = names;
  map->name  = malloc(NAMELIST_SIZE);

  // fill namelist
  names = 0;
  for(count = 0; dat[count].type != DAT_END; count++)
  if(dat[count].type == DAT_RLE_SPRITE)
  {
    map->name[names].index = count;
    strcpy(map->name[names].name, get_object_name(dat, count));
    names++;
  }
}


// --------------------------------------------------------------------------
// The correct_tilemap function

void correct_tilemap(TILEMAP *map, DATAFILE *dat)
{
  INDEXLIST *indexlist = malloc(sizeof(INDEXLIST)*map->names);
  int       count, count2;
  
  // build indexlist
  for(count=0; count<map->names; count++)
  {
    indexlist[count].from = map->name[count].index;
    indexlist[count].to   = NO_SPRITE;

    for(count2=0; dat[count2].type != DAT_END; count2++)
    {
      if(!strcmp(map->name[count].name, get_object_name(dat, count2)))
      {
        indexlist[count].to = count2;
        break;
      }
    }
  }

  // correct tilemap using the indexlist
  for(count=0; count<TILEMAP_DATA_SIZE; count++)
  {
    for(count2=0; count2<map->names; count2++)
    {
      if(map->data[count] == indexlist[count2].from)
      {
        map->data[count] = indexlist[count2].to;
        break;
      }
    }
  }

  // clean up
  free(indexlist);
}


// --------------------------------------------------------------------------
// End of header file

#ifdef __cplusplus
}
#endif // __cplusplus

#endif // __MAPFILE__
