/* Project SWORD
   V2.0

   SubSystem : Mathematical toolbox
   File      : LibSrc/ToolBox/Math/Matrix.CC
   Author    : Eric NICOLAS
   Overview  : Variable size matrix arithmetics
   UpDate    : May 06, 1995

** Copyright (C) 1993,1995 The SWORD Group
**
** This file is distributed under the terms listed in the document
** "copying.en". A copy of "copying.en" should accompany this file.
** if not, a copy should be available from where this file was obtained.
** This file may not be distributed without a verbatim copy of "copying.en".
**
** This file is distributed WITHOUT ANY WARRANTY; without even the implied
** warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
*/

#include <math.h>
#include "Common/Common.H"
#include "ToolBox/Math/MError.H"
#include "ToolBox/Math/Complex.H"
#include "ToolBox/Math/Vector.H"
#include "ToolBox/Math/Matrix.H"

#define Epsilon 1E-6

int     MatrixDefaultSizeI=1;
int     MatrixDefaultSizeJ=1;

static TMatrix tempMatrix;
static TVector tempVector;

TMatrix::TMatrix(int eSizeI, int eSizeJ)
{ SizeI=eSizeI;
  SizeJ=eSizeJ;
  Tabl=new complex[SizeI*SizeJ];
}

TMatrix::TMatrix(TMatrix& M)
{ int i;
  SizeI=M.SizeI;
  SizeJ=M.SizeJ;
  Tabl=new complex[SizeI*SizeJ];
  for(i=0;i<SizeI*SizeJ;i++) Tabl[i]=M.Tabl[i];
}

TMatrix::~TMatrix()
{ delete Tabl;
}

void TMatrix::Reset(void)
{ int i;
  for(i=0;i<SizeI*SizeJ;i++) Tabl[i]=0;
}

TMatrix TMatrix::operator=(TMatrix& M)
{ int i;
  if ((SizeI!=M.SizeI)&&(SizeJ!=M.SizeJ))
  { // Ajust sizes
    delete Tabl;
    SizeI=M.SizeI;
    SizeJ=M.SizeJ;
    Tabl=new complex[SizeI*SizeJ];
  }
  for(i=0;i<SizeI*SizeJ;i++) Tabl[i]=M.Tabl[i];
  return (*this);
}

TMatrix TMatrix::operator+(TMatrix& M)
{ int i;
  if (M.SizeI!=SizeI) MathError=MError_Matrix_MixingSizes;
  if (M.SizeJ!=SizeJ) MathError=MError_Matrix_MixingSizes;
  if (MathError) return (*this);
  tempMatrix=*this;
  for(i=0;i<SizeI*SizeJ;i++) tempMatrix.Tabl[i]+=M.Tabl[i];
  return tempMatrix;
}

TMatrix TMatrix::operator+=(TMatrix& M)
{ int i;
  if (M.SizeI!=SizeI) MathError=MError_Matrix_MixingSizes;
  if (M.SizeJ!=SizeJ) MathError=MError_Matrix_MixingSizes;
  if (MathError) return (*this);
  for(i=0;i<SizeI*SizeJ;i++) Tabl[i]+=M.Tabl[i];
  return (*this);
}

TMatrix TMatrix::operator-(TMatrix& M)
{ int i;
  if (M.SizeI!=SizeI) MathError=MError_Matrix_MixingSizes;
  if (M.SizeJ!=SizeJ) MathError=MError_Matrix_MixingSizes;
  if (MathError) return (*this);
  tempMatrix=*this;
  for(i=0;i<SizeI*SizeJ;i++) tempMatrix.Tabl[i]-=M.Tabl[i];
  return tempMatrix;
}

TMatrix TMatrix::operator-=(TMatrix& M)
{ int i;
  if (M.SizeI!=SizeI) MathError=MError_Matrix_MixingSizes;
  if (M.SizeJ!=SizeJ) MathError=MError_Matrix_MixingSizes;
  if (MathError) return (*this);
  for(i=0;i<SizeI*SizeJ;i++) Tabl[i]-=M.Tabl[i];
  return (*this);
}

TMatrix TMatrix::operator*(TMatrix& M)
{ int     i,j,k;
  complex C;
  if (SizeJ!=M.SizeI) MathError=MError_Matrix_MixingSizes;
  if (MathError) return (*this);
  tempMatrix=TMatrix(SizeI,M.SizeJ);
  for(i=0;i<SizeI;i++)
    for(j=0;j<M.SizeJ;j++)
    { C=0;
      for(k=0;k<SizeJ;k++) C+=operator()(i,k)*M(k,j);
      tempMatrix(i,j)=C;
    }
  return tempMatrix;
}

TMatrix TMatrix::operator*=(TMatrix& M)
{ int     i,j,k;
  complex C;
  if (SizeI!=M.SizeI) MathError=MError_Matrix_MixingSizes;
  if (SizeJ!=M.SizeJ) MathError=MError_Matrix_MixingSizes;
  if (MathError) return (*this);
  tempMatrix=*this;
  for(i=0;i<SizeI;i++)
    for(j=0;j<M.SizeJ;j++)
    { C=0;
      for(k=0;k<SizeJ;k++) C+=operator()(i,k)*M(k,j);
      tempMatrix(i,j)=C;
    }
  *this=tempMatrix;
  return (*this);
}

TMatrix T(TMatrix& M)
{ int i,j;
  if (M.SizeI!=M.SizeJ) MathError=MError_Matrix_Rectangular;
  if (MathError) return M;
  tempMatrix=M;
  for(i=0;i<M.SizeI;i++)
    for(j=0;j<M.SizeJ;j++)
      tempMatrix(j,i)=conj(M(i,j));
  return tempMatrix;
}

boolean Pivot(TMatrix& M, TMatrix *I, TVector *V, int ReferenceRow)
{ // Search for one row where ReferenceRow is non-zero the switch rows
  // in order to make diagonal ReferenceRow element non-zero.
  // Apply the same operations on matrix I
  int     NewRow;
  complex C;
  for(NewRow=ReferenceRow+1;NewRow<M.SizeI;NewRow++)
  { if (abs(M(NewRow,ReferenceRow))>Epsilon)
    { M.ExchangeRows(NewRow,ReferenceRow);
      if (I!=NULL) I->ExchangeRows(NewRow,ReferenceRow);
      if (V!=NULL)
      { C=V->Tabl[NewRow];
        V->Tabl[NewRow]=V->Tabl[ReferenceRow];
        V->Tabl[ReferenceRow]=C;
      }
      return FALSE;
    }
  }
  return TRUE;
}

complex Det(TMatrix& M)
{ if (M.SizeI!=M.SizeJ) MathError=MError_Matrix_Rectangular;
  if (MathError) return 0;
  complex PartialDet=1;
  complex DiagCoef,Multiplier;
  int     Row,ReferenceRow=0;
  tempMatrix=M;
  // Make the matrix upper-triangular
  for(ReferenceRow=0;ReferenceRow<tempMatrix.SizeI-1;ReferenceRow++)
  { // If Diagonal element is 0, then switch rows
    if (abs(tempMatrix(ReferenceRow,ReferenceRow))<Epsilon)
    { if (Pivot(tempMatrix,NULL,NULL,ReferenceRow))
      { MathError=MError_Matrix_Singularity;
        return 0;
      }
      PartialDet=-PartialDet;
    }
    DiagCoef=tempMatrix(ReferenceRow,ReferenceRow);
    for(Row=ReferenceRow+1;Row<tempMatrix.SizeI;Row++)
    { // Make the ReferenceRow Element of this Row zero
      if (abs(tempMatrix(Row,ReferenceRow))>Epsilon)
      { Multiplier=-tempMatrix(Row,ReferenceRow)/DiagCoef;
        tempMatrix.MultAddRow(ReferenceRow,Row,Multiplier);
      }
    }
    // Multiply the diagonal term into PartialDeter
    PartialDet*=DiagCoef;
  }
  return PartialDet*tempMatrix(tempMatrix.SizeI-1,tempMatrix.SizeI-1);
}

TMatrix Inv(TMatrix& M)
{ if (M.SizeI!=M.SizeJ) MathError=MError_Matrix_Rectangular;
  if (MathError) return M;
  int i,j;
  // Create the result
  TMatrix InvMatrix=M;
  // Check for trivial case : SizeI==SizeJ==1
  if (M.SizeI==1)
    if (abs(M.Tabl[0])<Epsilon)
    { // Exception : Singular Matrix
      MathError=MError_Matrix_Singularity;
      return M;
    }
    else
    { InvMatrix.Tabl[0]=1/M.Tabl[0];
      return InvMatrix;
    }
  // Make the inverse to be Identity Matrix
  for(i=0;i<M.SizeI;i++)
    for(j=0;j<M.SizeJ;j++)
      if (i==j) InvMatrix(i,j)=1; else InvMatrix(i,j)=0;
  // Copy the original matrix in order to proceed operations on it
  tempMatrix=M;
  // Datas for computing
  int     ReferenceRow,Row;
  complex Divisor;
  complex Multiplier;
  for(ReferenceRow=0;ReferenceRow<tempMatrix.SizeI;ReferenceRow++)
  { // Make the diagonal element non-zero
    if (abs(tempMatrix(ReferenceRow,ReferenceRow))<Epsilon)
      if (Pivot(tempMatrix,&InvMatrix,NULL,ReferenceRow))
      { // Exception : Singular Matrix
        MathError=MError_Matrix_Singularity;
        return M;
      }
    Divisor=tempMatrix(ReferenceRow,ReferenceRow);
    tempMatrix.DivideRow(ReferenceRow,Divisor);
    InvMatrix.DivideRow(ReferenceRow,Divisor);
    //
    for(Row=0;Row<M.SizeI;Row++)
    { if (Row!=ReferenceRow)
        if (abs(tempMatrix(Row,ReferenceRow))>Epsilon)
        { // Make the ReferenceRow element of this Row zero
          Multiplier=-tempMatrix(Row,ReferenceRow);
          tempMatrix.MultAddRow(ReferenceRow,Row,Multiplier);
          InvMatrix.MultAddRow(ReferenceRow,Row,Multiplier);
        }
    }
  }
  return InvMatrix;
}

// ------

TVector TMatrix::operator*(TVector& V)
{ if (SizeJ!=V.Size) MathError=MError_Matrix_VectorSize;
  if (MathError) return V;
  tempVector=TVector(SizeI);
  complex C;
  int     i,k;
  for(i=0;i<SizeI;i++)
  { C=0;
    for(k=0;k<SizeJ;k++)
      C+=operator()(i,k)*V(k);
    tempVector(i)=C;
  }
  return tempVector;
}

TVector TMatrix::SolveSystem(TVector& Constant)
{ if (SizeJ!=Constant.Size) MathError=MError_Matrix_VectorSize;
  if (SizeI!=SizeJ)         MathError=MError_Matrix_Rectangular;
  if (MathError) return Constant;
  // Create the solution vector
  TVector Sol(SizeJ);
  // Solve the trivial case
  if (SizeI==1)
  { if (abs(Tabl[0])<Epsilon) MError_Matrix_Singularity;
    if (MathError) return Constant;
    Sol(0)=Constant(0)/Tabl[0];
    return Sol;
  }
  // Copy the coefficient matrix and the constant vector in order to perform
  // transformations on it
  TMatrix Coefs=*this;
  TVector Const=Constant;
  // Makes the coefficient matrix upper-triangular and perform the same
  // operations on the constant vector
  int     ReferenceRow,Row,Term;
  complex DiagCoef,Multiplier,C;
  for(ReferenceRow=0;ReferenceRow<SizeI-1;ReferenceRow++)
  { // Check if the diagonal element is zero
    if (abs(Coefs(ReferenceRow,ReferenceRow))<Epsilon)
      if (Pivot(Coefs,NULL,&Const,ReferenceRow))
      { // Exception : Singular Matrix
        MathError=MError_Matrix_Singularity;
        return Const;
      }
    // Make other coefficients of this ReferenceRow column zero
    DiagCoef=Coefs(ReferenceRow,ReferenceRow);
    for(Row=ReferenceRow+1;Row<SizeI;Row++)
    { if (abs(Coefs(Row,ReferenceRow))>Epsilon)
      { Multiplier=-Coefs(Row,ReferenceRow)/DiagCoef;
        Coefs.MultAddRow(ReferenceRow,Row,Multiplier);
        Const(Row)+=Multiplier*Const(ReferenceRow);
      }
    }
  }
  if (abs(Coefs(SizeI-1,SizeJ-1))<Epsilon)
  { MathError=MError_Matrix_Singularity;
    return Const;
  }
  // Apply Backwards substitution to the upper triangular Coefs Matrix and
  // Constant vector.
  for(Term=SizeI-1;Term>=0;Term--)
  { C=0;
    for(Row=Term+1;Row<SizeI;Row++)
      C+=Coefs(Term,Row)*Sol(Row);
    Sol(Term)=(Const(Term)-C)/Coefs(Term,Term);
  }
  return Sol;
}

// ------
         
TMatrix operator*(TMatrix& M, complex  C)
{ int i;
  tempMatrix=M;
  for(i=0;i<tempMatrix.SizeI*tempMatrix.SizeJ;i++) tempMatrix.Tabl[i]*=C;
  return tempMatrix;
}

TMatrix operator*(complex  C, TMatrix& M)
{ int i;
  tempMatrix=M;
  for(i=0;i<tempMatrix.SizeI*tempMatrix.SizeJ;i++) tempMatrix.Tabl[i]*=C;
  return tempMatrix;
}

TMatrix TMatrix::operator*=(complex  C)
{ int i;
  for(i=0;i<SizeI*SizeJ;i++) Tabl[i]*=C;
  return (*this);
}

TMatrix operator/(TMatrix& M, complex  C)
{ int i;
  tempMatrix=M;
  for(i=0;i<tempMatrix.SizeI*tempMatrix.SizeJ;i++) tempMatrix.Tabl[i]/=C;
  return tempMatrix;
}

TMatrix operator/(complex  C, TMatrix& M)
{ int i;
  tempMatrix=M;
  for(i=0;i<tempMatrix.SizeI*tempMatrix.SizeJ;i++) tempMatrix.Tabl[i]=C/M.Tabl[i];
  return tempMatrix;
}

TMatrix TMatrix::operator/=(complex  C)
{ int i;
  for(i=0;i<SizeI*SizeJ;i++) Tabl[i]/=C;
  return (*this);
}

void TMatrix::SetColumn(int J, TVector& V)
{ if (V.Size!=SizeI) MathError=MError_Matrix_VectorSize;
  if ((J<0)||(J>=SizeJ)) MathError=MError_Matrix_OutOfSize;
  if (MathError) return;
  int i;
  for(i=0;i<SizeI;i++)
    operator()(i,J)=V(i);
}

void TMatrix::SetRow(int I, TVector& V)
{ if (V.Size!=SizeJ) MathError=MError_Matrix_VectorSize;
  if ((I<0)||(I>=SizeI)) MathError=MError_Matrix_OutOfSize;
  if (MathError) return;
  int j;
  for(j=0;j<SizeJ;j++)
    operator()(I,j)=V(j);
}

TVector TMatrix::Column(int J)
{ if (J<0)      { MathError=MError_Matrix_OutOfSize; J=0;       }
  if (J>=SizeJ) { MathError=MError_Matrix_OutOfSize; J=SizeJ-1; }
  int i;
  tempVector=TVector(SizeI);
  for(i=0;i<SizeI;i++) tempVector(i)=operator()(i,J);
  return tempVector;
}

TVector TMatrix::Row(int I)
{ if (I<0)      { MathError=MError_Matrix_OutOfSize; I=0;       }
  if (I>=SizeI) { MathError=MError_Matrix_OutOfSize; I=SizeI-1; }
  int j;
  tempVector=TVector(SizeJ);
  for(j=0;j<SizeJ;j++) tempVector(j)=operator()(I,j);
  return tempVector;
}

void TMatrix::ExchangeRows(int I1, int I2)
{ if (I1<0)      { MathError=MError_Matrix_OutOfSize; I1=0;       }
  if (I1>=SizeI) { MathError=MError_Matrix_OutOfSize; I1=SizeI-1; }
  if (I2<0)      { MathError=MError_Matrix_OutOfSize; I2=0;       }
  if (I2>=SizeI) { MathError=MError_Matrix_OutOfSize; I2=SizeI-1; }
  int      j;
  complex  C;
  complex *P1,*P2;
  P1=Tabl+I1*SizeJ;
  P2=Tabl+I2*SizeJ;
  for(j=0;j<SizeJ;j++)
  { C=*P1; *P1=*P2; *P2=C;
    P1++;
    P2++;
  }
}

void TMatrix::ExchangeColumns(int J1, int J2)
{ if (J1<0)      { MathError=MError_Matrix_OutOfSize; J1=0;       }
  if (J1>=SizeJ) { MathError=MError_Matrix_OutOfSize; J1=SizeJ-1; }
  if (J2<0)      { MathError=MError_Matrix_OutOfSize; J2=0;       }
  if (J2>=SizeJ) { MathError=MError_Matrix_OutOfSize; J2=SizeJ-1; }
  int      i;
  complex  C;
  complex *P1,*P2;
  P1=Tabl+J1;
  P2=Tabl+J2;
  for(i=0;i<SizeI;i++)
  { C=*P1; *P1=*P2; *P2=C;
    P1+=SizeJ;
    P2+=SizeJ;
  }
}

void TMatrix::MultAddRow(int I1, int I2, complex C)
{ if (I1<0)      { MathError=MError_Matrix_OutOfSize; I1=0;       }
  if (I1>=SizeI) { MathError=MError_Matrix_OutOfSize; I1=SizeI-1; }
  if (I2<0)      { MathError=MError_Matrix_OutOfSize; I2=0;       }
  if (I2>=SizeI) { MathError=MError_Matrix_OutOfSize; I2=SizeI-1; }
  int      j;
  complex *P1,*P2;
  P1=Tabl+I1*SizeJ;
  P2=Tabl+I2*SizeJ;
  for(j=0;j<SizeJ;j++)
  { *P2+=*P1*C;
    P1++;
    P2++;
  }
}

void TMatrix::MultAddColumn(int J1, int J2, complex C)
{ if (J1<0)      { MathError=MError_Matrix_OutOfSize; J1=0;       }
  if (J1>=SizeJ) { MathError=MError_Matrix_OutOfSize; J1=SizeJ-1; }
  if (J2<0)      { MathError=MError_Matrix_OutOfSize; J2=0;       }
  if (J2>=SizeJ) { MathError=MError_Matrix_OutOfSize; J2=SizeJ-1; }
  int      i;
  complex *P1,*P2;
  P1=Tabl+J1;
  P2=Tabl+J2;
  for(i=0;i<SizeI;i++)
  { *P2+=*P1*C;
    P1+=SizeJ;
    P2+=SizeJ;
  }
}

void TMatrix::DivideRow(int I, complex C)
{ if (I<0)      { MathError=MError_Matrix_OutOfSize; I=0;       }
  if (I>=SizeI) { MathError=MError_Matrix_OutOfSize; I=SizeI-1; }
  int      j;
  complex *P=Tabl+I*SizeJ;
  for(j=0;j<SizeJ;j++)
  { *P/=C;
    P++;
  }
}

void TMatrix::DivideColumn(int J, complex C)
{ if (J<0)      { MathError=MError_Matrix_OutOfSize; J=0;       }
  if (J>=SizeJ) { MathError=MError_Matrix_OutOfSize; J=SizeJ-1; }
  int      i;
  complex *P=Tabl+J;
  for(i=0;i<SizeI;i++)
  { *P/=C;
    P+=SizeI;
  }
}
