/* Project SWORD
   V2.0

   SubSystem : Mathematical toolbox
   File      : Include/ToolBox/Math/Complex.H
   Author    : Eric NICOLAS
   Overview  : Complex arithmetics in signle and double precision
   UpDate    : Apr 29, 1995

** Copyright (C) 1993,1995 The SWORD Group
**
** This file is distributed under the terms listed in the document
** "copying.en". A copy of "copying.en" should accompany this file.
** if not, a copy should be available from where this file was obtained.
** This file may not be distributed without a verbatim copy of "copying.en".
**
** This file is distributed WITHOUT ANY WARRANTY; without even the implied
** warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
*/

#ifndef _TOOLBOX_MATH_COMPLEX_H_
#define _TOOLBOX_MATH_COMPLEX_H_

struct complex;
struct lcomplex;

struct complex
{ // Datas
  float re,im;
  // Constructors
  complex(float _re, float _im=0);
  complex();
  complex(complex& c);
  complex(lcomplex& c);
  // Using complexes
  friend float   real(complex& c);
  friend float   imag(complex& c);
  friend complex conj(complex& c);
  friend float   norm(complex& c);
  friend float   abs(complex& c);
  friend float   arg(complex& c);
  // Functions on complexes
  friend complex sqrt(complex& c);
  friend complex polar(float r, float th);
  friend complex exp(complex& c);
  friend complex log(complex& c);
  friend complex log10(complex& c);
  friend complex sin(complex& c);
  friend complex cos(complex& c);
  friend complex tan(complex& c);
  friend complex sinh(complex& c);
  friend complex tanh(complex& c);
  friend complex cosh(complex& c);
  friend complex pow(complex& x, complex& y);
  // operators
  // :: +
  friend complex operator+(complex& c1, complex& c2);
  friend complex operator+(float    f,  complex& c);
  friend complex operator+(complex& c,  float    f);
  // :: -
  friend complex operator-(complex& c1, complex& c2);
  friend complex operator-(float    f,  complex& c);
  friend complex operator-(complex& c,  float    f);
  // :: *
  friend complex operator*(complex& c1, complex& c2);
  friend complex operator*(float    f,  complex& c);
  friend complex operator*(complex& c,  float    f);
  // :: /
  friend complex operator/(complex& c1, complex& c2);
  friend complex operator/(float    f,  complex& c);
  friend complex operator/(complex& c,  float    f);
  // :: logical
  friend int     operator==(complex& c1, complex& c2);
  friend int     operator!=(complex& c1, complex& c2);
  // :: x=
  complex &operator+=(complex& c);
  complex &operator+=(float    f);
  complex &operator-=(complex& c);
  complex &operator-=(float    f);
  complex &operator*=(complex& c);
  complex &operator*=(float    f);
  complex &operator/=(complex& c);
  complex &operator/=(float    f);
  // :: +,-
  complex operator+();
  complex operator-();
};

struct lcomplex
{ // datas
  double re,im;
  // Constructors
  lcomplex(double _re, double _im=0);
  lcomplex();
  lcomplex(lcomplex& c);
  lcomplex(complex& c);
  // Using complexes
  friend double   real(lcomplex& c);
  friend double   imag(lcomplex& c);
  friend lcomplex conj(lcomplex& c);
  friend double   norm(lcomplex& c);
  friend double   abs(lcomplex& c);
  friend double   arg(lcomplex& c);
  // operators
  // :: +
  friend lcomplex operator+(lcomplex& c1, lcomplex& c2);
  friend lcomplex operator+(double    f,  lcomplex& c);
  friend lcomplex operator+(lcomplex& c,  double    f);
  // :: -
  friend lcomplex operator-(lcomplex& c1, lcomplex& c2);
  friend lcomplex operator-(double    f,  lcomplex& c);
  friend lcomplex operator-(lcomplex& c,  double    f);
  // :: *
  friend lcomplex operator*(lcomplex& c1, lcomplex& c2);
  friend lcomplex operator*(double    f,  lcomplex& c);
  friend lcomplex operator*(lcomplex& c,  double    f);
  // :: /
  friend lcomplex operator/(lcomplex& c1, lcomplex& c2);
  friend lcomplex operator/(double    f,  lcomplex& c);
  friend lcomplex operator/(lcomplex& c,  double    f);
  // :: logical
  friend int     operator==(lcomplex& c1, lcomplex& c2);
  friend int     operator!=(lcomplex& c1, lcomplex& c2);
  // :: x=
  lcomplex &operator+=(lcomplex& c);
  lcomplex &operator+=(double    f);
  lcomplex &operator-=(lcomplex& c);
  lcomplex &operator-=(double    f);
  lcomplex &operator*=(lcomplex& c);
  lcomplex &operator*=(double    f);
  lcomplex &operator/=(lcomplex& c);
  lcomplex &operator/=(double    f);
  // :: +,-
  lcomplex operator+();
  lcomplex operator-();
};

// ===== Usefull complexes

extern complex NUL;
extern complex cI;
extern complex cJ;
extern complex cPi;
extern complex cE;

// ===== inline Functions

inline complex::complex(float _re, float _im)
{ re=_re;
  im=_im;
}

inline complex::complex()
{ re=im=0;
}

inline complex::complex(complex& c)
{ re=c.re;
  im=c.im;
}

inline complex::complex(lcomplex& c)
{ re=(float)c.re;
  im=(float)c.im;
}

inline float real(complex &c)
{ return c.re;
}

inline float imag(complex &c)
{ return c.im;
}

inline complex conj(complex &c)
{ return complex(c.re, -c.im);
}

inline complex polar(float r, float th)
{ return complex(r*cos(th), r*sin(th));
}

inline complex complex::operator+()
{ return *this;
}

inline complex complex::operator-()
{ return complex(-re, -im);
}

inline complex &complex::operator+=(complex &c)
{ re += c.re;
  im += c.im;
  return *this;
}

inline complex &complex::operator+=(float f)
{ re += f;
  return *this;
}

inline complex &complex::operator-=(complex &c)
{ re -= c.re;
  im -= c.im;
  return *this;
}

inline complex &complex::operator-=(float f)
{ re -= f;
  return *this;
}

inline complex &complex::operator*=(float f)
{ re *= f;
  im *= f;
  return *this;
}

inline complex &complex::operator/=(float f)
{ re /= f;
  im /= f;
  return *this;
}

inline complex operator+(complex &c1, complex &c2)
{ return complex(c1.re + c2.re, c1.im + c2.im);
}

inline complex operator+(float f, complex &c)
{  return complex(f + c.re, c.im);
}

inline complex operator+(complex &c, float f)
{ return complex(c.re + f, c.im);
}

inline complex operator-(complex &c1, complex &c2)
{ return complex(c1.re - c2.re, c1.im - c2.im);
}

inline complex operator-(float f, complex &c)
{ return complex(f - c.re, -c.im);
}

inline complex operator-(complex &c, float f)
{ return complex(c.re - f, c.im);
}

inline complex operator*(complex &c, float f)
{ return complex(c.re*f, c.im*f);
}

inline complex operator*(float f, complex &c)
{ return complex(c.re*f, c.im*f);
}

inline complex operator*(complex& c1, complex& c2)
{ return complex(c1.re*c2.re-c1.im*c2.im, c1.re*c2.im+c1.im*c2.re);
}

inline complex operator/(complex &c, float f)
{ return complex(c.re/f, c.im/f);
}

inline int operator==(complex &c1, complex &c2)
{ return ((c1.re == c2.re) && (c1.im == c2.im));
}

inline int operator!=(complex &c1, complex &c2)
{ return ((c1.re != c2.re) || (c1.im != c2.im));
}

// ===== Inline functions : lcomplex

inline lcomplex::lcomplex(double _re, double _im)
{ re=_re;
  im=_im;
}

inline lcomplex::lcomplex()
{ re=im=0;
}

inline lcomplex::lcomplex(lcomplex& c)
{ re=c.re;
  im=c.im;
}

inline lcomplex::lcomplex(complex& c)
{ re=(double)c.re;
  im=(double)c.im;
}

inline double real(lcomplex &c)
{ return c.re;
}

inline double imag(lcomplex &c)
{ return c.im;
}

inline lcomplex conj(lcomplex &c)
{ return lcomplex(c.re, -c.im);
}

inline lcomplex polar(double r, double th)
{ return lcomplex(r*cos(th), r*sin(th));
}

inline lcomplex lcomplex::operator+()
{ return *this;
}

inline lcomplex lcomplex::operator-()
{ return lcomplex(-re, -im);
}

inline lcomplex &lcomplex::operator+=(lcomplex &c)
{ re += c.re;
  im += c.im;
  return *this;
}

inline lcomplex &lcomplex::operator+=(double f)
{ re += f;
  return *this;
}

inline lcomplex &lcomplex::operator-=(lcomplex &c)
{ re -= c.re;
  im -= c.im;
  return *this;
}

inline lcomplex &lcomplex::operator-=(double f)
{ re -= f;
  return *this;
}

inline lcomplex &lcomplex::operator*=(double f)
{ re *= f;
  im *= f;
  return *this;
}

inline lcomplex &lcomplex::operator/=(double f)
{ re /= f;
  im /= f;
  return *this;
}

inline lcomplex operator+(lcomplex &c1, lcomplex &c2)
{ return lcomplex(c1.re + c2.re, c1.im + c2.im);
}

inline lcomplex operator+(double f, lcomplex &c)
{  return lcomplex(f + c.re, c.im);
}

inline lcomplex operator+(lcomplex &c, double f)
{ return lcomplex(c.re + f, c.im);
}

inline lcomplex operator-(lcomplex &c1, lcomplex &c2)
{ return lcomplex(c1.re - c2.re, c1.im - c2.im);
}

inline lcomplex operator-(double f, lcomplex &c)
{ return lcomplex(f - c.re, -c.im);
}

inline lcomplex operator-(lcomplex &c, double f)
{ return lcomplex(c.re - f, c.im);
}

inline lcomplex operator*(lcomplex &c, double f)
{ return lcomplex(c.re*f, c.im*f);
}

inline lcomplex operator*(double f, lcomplex &c)
{ return lcomplex(c.re*f, c.im*f);
}

inline lcomplex operator*(lcomplex& c1, lcomplex& c2)
{ return lcomplex(c1.re*c2.re-c1.im*c2.im, c1.re*c2.im+c1.im*c2.re);
}

inline lcomplex operator/(lcomplex &c, double f)
{ return lcomplex(c.re/f, c.im/f);
}

inline int operator==(lcomplex &c1, lcomplex &c2)
{ return ((c1.re == c2.re) && (c1.im == c2.im));
}

inline int operator!=(lcomplex &c1, lcomplex &c2)
{ return ((c1.re != c2.re) || (c1.im != c2.im));
}

#endif // _TOOLBOX_MATH_COMPLEX_H_
