/**********************************************************************
 *  
 *  NAME:           basewin.h
 *  
 *  DESCRIPTION:    
 *  
 *  copyright (c) 1990 J. Alan Eldridge
 * 
 *  M O D I F I C A T I O N   H I S T O R Y
 *
 *  when        who                 what
 *  -------------------------------------------------------------------
 *  11/29/90    JAE                 created
 *  
 *********************************************************************/

#ifndef __BASEWIN_H
#define __BASEWIN_H

//  define EXTENDED_IO_MANIPS to 1 if you want the extra
//  i/o manipulators "<< setpos(y,x)" and "<< setattr(a)".

#define EXTENDED_IO_MANIPS  1

class basewin: 
    public vidbuf,
    public viewport {

private:

    uchar   fWrap:      1,  //  line wrap flag
            fScroll:    1,  //  scroll port flag
            fFrame:     1,  //  has a frame
            fCloseBox:  1;  //  has a close box
            
    void    zflags()        //  zap the flags
        { fWrap = fScroll = fFrame = fCloseBox = FALSE; }

    uchar   att;            //  video attribute
    int     ydirty[2],      //  first,last dirty row
            (*xdirty)[2];   //  first,last dirty col on each row
                            //  note: decl. is "ptr to array of 2 uchars"

protected:

    //  dirty region handling
    void    mark(int row);
    void    markline(int row, int col);
    void    unmark();
    void    touch(int yul, int xul, int ylr, int xlr);

    //  scroll a region of the window
    void    scroll(int yul, int xul, int ylr, int xlr, int nLines = 1);

    //  draw a box in the window
    void    box(int yul, int xul, int ylr, int xlr, int vchr, int hchr,
        int fillflg = 0, int fillchr = ' ');

public:    

    //  constructor
    basewin(int yul, int xul, int ylr, int xlr);

    //  destructor
    virtual ~basewin();

    //  video attribute control
    //  these are virtual because a Window has more intelligent
    //  mechanisms for handling screen colors
    virtual void    setattr(uchar a)
        { att = a; }
    virtual int     getattr()
        { return att; }
    int iscolor()   //  for convenience
        { return vid_ISCOLOR; }
        
    //  touch all of port or window
    void    touch()         //  just the current port
        { touch(yUL, xUL, yLR, xLR); }
    void    touchall()      //  the whole window
        { touch(0, 0, rows-1, cols-1); }

    //  scroll current port
    void    scroll(int nLines = 1) 
        { scroll(yUL, xUL, yLR, xLR, nLines); }
        
    //  set/get cursor (relative to port)
    virtual void    setpos(int y, int x)
        { yCur = y+yUL; xCur = x+xUL; }
    virtual void    getpos(int &y, int &x)
        { y = yCur-yUL; x = xCur-xUL; }

    //  change viewport & home cursor
    virtual void    setport(int yul, int xul, int ylr, int xlr)
        { viewport::setport(yul,xul,ylr,xlr); setpos(0,0); }
    virtual void    setport(int *box)
        { basewin::setport(box[0],box[1],box[2],box[3]); }
    virtual void    setport(viewport &vp)
        { viewport::setport(vp); setpos(0,0); }
    void  fullport()
        { setport(0,0,rows-1,cols-1); }
                
    //  character output
    //  all of these return refs to the window so they can be
    //  chained together, e.g., w.put("hello").NL().put("world").refresh()
    //  (syntactically, it's a little ugly, but ...)    
    basewin &NL();      //  newline
    basewin &CR();      //  carriage return
    basewin &BS();      //  backspace
     
    basewin &put(uchar c);              //  write a character
    basewin &put(uchar *s);             //  write a string
    basewin &center(int line, uchar *s);    //  center a string
    basewin &center(uchar *s)           //  center a string
        { return center(yCur, s); }
    basewin &printf(uchar *fmt, ...);   //  printf() to window
    basewin &operator<<(uchar c)        //  alternate character output
        { return put(c); }
    basewin &operator<<(uchar *s)       //  alternate string output
        { return put(s); }

    //  clear part/all of the current port
    basewin &clear(uchar ch = ' ');  //  the whole port
    basewin &clreol(uchar ch = ' '); //  clear to end of line

    //  "i/o manipulator" helper
    basewin &operator<<(basewin &(*f)(basewin &))
        { return f(*this); }

    //  draw a box 
    enum {
        boxLine1 = UCHAR_MAX + 1,
        boxLine2
    };
    void    box(int vchr, int hchr, int fillflg = FALSE, int fillchr = ' ')
        { box(yUL, xUL, yLR, xLR, vchr, hchr, fillflg, fillchr); }
    void    frame(int vchr, int hchr, int fillflg = FALSE, int fillchr = ' ');
    void    closebox();
        
    //  read character/attrib at current position
    int     get()
        { return vidptr(yCur, xCur)->get(); }
    int     geta()
        { return vidptr(yCur, xCur)->geta(); }
        
    //  get a string from the user
    
    enum {
        trimStr     =   0x01,               //  trim trailing blanks
        expandStr   =   0x02,               //  keep padded to max length
        autoTab     =   0x04|0x02,          //  cursor left/right out of field
        zapField    =   0x08,               //  first key is print ==> erase field
        noErrBeep   =   0x10                //  no beep if can't process key
    };
    
    int editfld(
            uchar   *pBuf,
            int     bMax,
            int     *pTermKeys = 0,
            int     eFlags = trimStr,
            int     (*OKFunc)(int c) = 0,
            int     fillChr = ' ');

    int egetstr(uchar *pBuf);
    int getstr(uchar *pBuf);
    int operator>>(uchar *pBuf)
        { return getstr(pBuf); }
        
    //  update to display screen
    virtual void    refresh(int vflag=1);   //  refresh dirty area only

    //  manipulate flags
    int     wrapok()                    //  get line wrap state
        { return fWrap; }
    void    wrapok(int flag)            //  turn line wrap on/off
        { fWrap = flag != 0; }
    int     scrollok()                  //  get scrolling state
        { return fScroll; }
    void    scrollok(int flag)          //  turn scrolling on/off
        { fScroll = flag != 0; }
    
    //  read keyboard
    int     keyrdy()                    //  is a key ready?
        {   return kbd_ready(); }
    int     getkey()                    //  wait for & return key
        { return kbd_getkey(); }
    void    operator>>(int &c)          //  wait for & return key
        { c = getkey(); }
};

//  these are like parameterless i/o manipulators
//  e.g., w << "Hello, world" << nl << refresh;
    
inline basewin  &nl(basewin &w)
    {   return w.NL(); }
inline basewin  &cr(basewin &w)
    {   return w.CR(); }
inline basewin  &bs(basewin &w)
    {   return w.BS(); }
inline basewin  &refresh(basewin &w)
    {   w.refresh(); return w; }
inline basewin  &clear(basewin &w)
    {   return w.clear(); }
inline basewin  &clreol(basewin &w)
    {   return w.clreol(); }

#if EXTENDED_IO_MANIPS

//  this class allows w << setpos(y,x)

class basewin_cpos {
private:
    int y, x;
    basewin &setcursor(basewin &w)
        { w.setpos(y,x); return w; }
protected:
    basewin_cpos(int r, int c): y(r), x(c) { }
public:
    friend basewin_cpos setpos(int r, int c)
        { return basewin_cpos(r,c); }
    friend basewin &
    operator<<(basewin &w, basewin_cpos cp)
        { return cp.setcursor(w); }
};            

//  this class allows w << setattr(a)

class basewin_attr {
private:
    uchar   a;
    basewin &setattr(basewin &w)
        { w.setattr(a); return w; }
protected:
    basewin_attr(int att): a(att) { }
public:
    friend basewin_attr setattr(int a)
        { return basewin_attr(a); }
    friend basewin &
    operator<<(basewin &w, basewin_attr a)
        { return a.setattr(w); }
};            

#endif

//  lookup tables for ibm line drawing characters

extern  uchar   UWLineChars     [][2];
extern  uchar   UWTeeChars      [][2][2];
extern  uchar   UWCornerChars   [][2][2];

//  values for args to teechar(), cornerchar(), and linechar()

#define SINGLE_LINE     0
#define DOUBLE_LINE     1

#define LINE_HORZ       0
#define LINE_VERT       1

#define TEE_TOP         0
#define TEE_RIGHT       1
#define TEE_BOTTOM      2
#define TEE_LEFT        3
#define TEE_CENTER      4

#define CORNER_UL       0
#define CORNER_UR       1
#define CORNER_LR       2
#define CORNER_LL       3

//  these inline functions access the line drawing char arrays

inline int  linechar(int vertp, int dblp)
    { return UWLineChars[vertp][dblp]; }
inline int  teechar(int pos, int v, int h)
    { return UWTeeChars[pos][v][h]; }
inline int  cornerchar(int pos, int v, int h)
    { return UWCornerChars[pos][v][h]; }

//  defines for gray box characters (incl. alternate spelling)

#define GRAYBOX0    176
#define GREYBOX0    176
#define GRAYBOX1    177
#define GREYBOX1    177
#define GRAYBOX2    178
#define GREYBOX2    178
#define BLACKBOX    219

//  defines for partial box characters

#define BOX_ENTIRE  BLACKBOX
#define BOX_LOWER   220
#define BOX_LEFT    221
#define BOX_RIGHT   222
#define BOX_UPPER   223

//  initialize the window library

int initscr(
    int     vidmode = LASTMODE,
    char    *vidvar = 0);

//  function to annoy the user

void beep(
    int     nTimes = 1,
    uint    howLong = 333,
    uint    Hz = 880);

#endif
