/*-----------------------------------------------------------------------*/
/*   Mach64 library source file LIBM64.C  version  0.1                   */
/*                                                                       */
/*   This library is written for DJGPP version >= 2.0                    */
/*   All information is taken from the VGADOC4B package.                 */
/*                                                                       */
/*   Andreas Leipelt,  Hamburg April 1997                                */
/*   Please send comments, suggestions, bug reports etc. to              */
/*   leipelt@math.uni-hamburg.de                                         */
/*-----------------------------------------------------------------------*/
#include <stdlib.h>
#include <dpmi.h>
#include <go32.h>
#include <sys/farptr.h>
#include <pc.h>
#include <string.h>
#include "mach64.h"
#include "m64regs.h"

static int call_m64_vbe(unsigned char func, unsigned short bx_data,
			unsigned short cx_data, unsigned short dx_data,
			unsigned char  *al_ret, unsigned short *bx_ret,
			unsigned short *cx_ret, unsigned short *dx_ret)
{
  __dpmi_regs r;

  r.h.ah = 0xA0;
  r.h.al = func;
  r.x.bx = bx_data;
  r.x.cx = cx_data;
  r.x.dx = dx_data;
  __dpmi_int(0x10, &r);
  if (al_ret != NULL) *al_ret = r.h.al;
  if (bx_ret != NULL) *bx_ret = r.x.bx;
  if (cx_ret != NULL) *cx_ret = r.x.cx;
  if (dx_ret != NULL) *dx_ret = r.x.dx;
  return (int)r.h.ah;
}

int M64_detect(void)
{
   char            signature[] = " 761295520";
   int             i;
   long            BIOS_addr = 0xC0030;
   unsigned long   tmp;

   for (i=0; i < 10; i++) {
     if (signature[i] != _farpeekb(_dos_ds, BIOS_addr + (long)i)) return M64_ERROR;
   }

   tmp = inportl(Scratch_Reg0);
   outportl(Scratch_Reg0, 0x55555555);
   if (inportl(Scratch_Reg0) != 0x55555555) return M64_ERROR;
   else {
     outportl(Scratch_Reg0, 0xaaaaaaaa);
     if (inportl(Scratch_Reg0) != 0xaaaaaaaa) return M64_ERROR;
   }
   outportl(Scratch_Reg0, tmp);

   return M64_OK;
}

int M64_load_CRTC_params(unsigned short params)
{
  return call_m64_vbe(0x00, 0, params, 0, NULL, NULL, NULL, NULL);
}

int M64_set_display_mode(unsigned short params)
{
  return call_m64_vbe(0x01, 0, params, 0, NULL, NULL, NULL, NULL);
}

int M64_load_CRTC_params_and_set_mode(unsigned short params)
{
  return call_m64_vbe(0x02, 0, params, 0, NULL, NULL, NULL, NULL);
}

int M64_read_EEPROM(unsigned short index, unsigned short *data)
{
  return call_m64_vbe(0x03, index, 0, 0, NULL, NULL, NULL, data);
}

int M64_write_EEPROM(int index, int data)
{
  return call_m64_vbe(0x04, index, 0, data, NULL, NULL, NULL, NULL);
}

int M64_memory_aperture_service(unsigned short enable)
{
  if (enable) enable = 1;
  return call_m64_vbe(0x05, 0, enable, 0, NULL, NULL, NULL, NULL);
}

int M64_short_query(unsigned char  *aperture_cfg,
		    unsigned short *aperture_adr,
		    unsigned short *memory_size_and_color_depth,
		    unsigned short *ASIC_ident)
{
  return call_m64_vbe(0x06, 0, 0, 0, aperture_cfg, aperture_adr,
		  memory_size_and_color_depth, ASIC_ident);
}

/* AL = 07h  Return hardware capability list
   Input  ?
   Output ? */
int M64_hardware_capabilty_list(void)
{
  return M64_FUNCTION_NOT_SUPPORTED;
}

int M64_query_device_data_structure_size(unsigned short fullinfo,
					 unsigned short *size)
{
  if (fullinfo) fullinfo = 1;
  return call_m64_vbe(0x08, 0, fullinfo, 0, NULL, NULL, size, NULL);
}

int M64_query_device(unsigned short fullinfo, unsigned short size, void *infobuf)
{
  int err;

  if (fullinfo) fullinfo = 1;
  /* Use the transfer buffer to store the results of Mach64 BIOS call */
  err = call_m64_vbe(0x09, 0, fullinfo, __tb / 16, NULL, NULL, NULL, NULL);
  if (err != M64_OK) return err;
  dosmemget(__tb, size, infobuf);
  return M64_OK;
}

int M64_clock_chip_frequency_table(unsigned short *freq_table,
				   M64_clock_info_structure *clock_chip_info)
{
  unsigned short seg, fofs, cofs;
  int err;
  long i, ptr;
  unsigned char *tmp;

  err = call_m64_vbe(0x0A, 0, 0, 0, NULL, &fofs, &cofs, &seg);
  if (err != M64_OK) return err;

  ptr = (((long)seg) << 4) + (long)fofs;
  for (i=0; i < 16; i++) {
    freq_table[i] = _farpeekw(_dos_ds, ptr + i * 2);
  }
  ptr = (((long)seg) << 4) + (long)cofs;
  tmp = (unsigned char*)clock_chip_info;
  for (i=0; i < sizeof(M64_clock_info_structure); i++) {
    *tmp++ = _farpeekb(_dos_ds, ptr + i);
  }
  return M64_OK;
}

int M64_program_clock_entry(unsigned short  clock_table_entry,
			    unsigned short  clock_value,
			    unsigned char  *clock_chip_type,
			    unsigned short *prog_word)
{
  return call_m64_vbe(0x0B, clock_value, clock_table_entry << 8, 0,
		    clock_chip_type, prog_word, 0, 0);
}

int M64_set_DPMS_mode(unsigned short DPMS_mode)
{
  return call_m64_vbe(0x0C, 0, DPMS_mode & 0x00FF, 0, NULL, NULL, NULL, NULL);
}

int M64_get_DPMS_state(unsigned short *DPMS_state)
{
  return call_m64_vbe(0x0D, 0, 0, 0, NULL, NULL, DPMS_state, NULL);
}

int M64_set_GCPM_state(unsigned short GCPM_state)
{
  return call_m64_vbe(0x0E, 0, GCPM_state & 0x00FF, 0, NULL, NULL, NULL, NULL);
}

int M64_get_GCPM_state(unsigned short *GCPM_state)
{
  return call_m64_vbe(0x0F, 0, 0, 0, NULL, NULL, GCPM_state, NULL);
}

int M64_set_DAC_state(unsigned short DAC_state)
{
  return call_m64_vbe(0x10, 0, DAC_state & 0x0001, 0, NULL, NULL, NULL, NULL);
}

int M64_external_device_info(void) { return M64_FUNCTION_NOT_SUPPORTED; }

int M64_get_IO_base_address(unsigned short *io_base)
{
  return call_m64_vbe(0x12, 0, 0, 0, NULL, NULL, NULL, io_base);
}

void M64_set_palette_entry(int entry, int r, int g, int b)
{
  outportb(Dac_Regs, (unsigned char)entry);
  outportb(Dac_Regs+1, (unsigned char)r);
  outportb(Dac_Regs+1, (unsigned char)g);
  outportb(Dac_Regs+1, (unsigned char)b);
}

void M64_get_palette_entry(int entry, int *r, int *g, int *b)
{
  *r = (int)inportb(Dac_Regs+3);
  *g = (int)inportb(Dac_Regs+3);
  *b = (int)inportb(Dac_Regs+3);
}

void M64_set_palette(int first, int last, unsigned char *pal)
{
  int i;
  unsigned char *p = pal;

  outportb(Dac_Regs, (unsigned char)first);
  for (i=first; i <= last; i++) {
    outportb(Dac_Regs+1, *p++);
    outportb(Dac_Regs+1, *p++);
    outportb(Dac_Regs+1, *p++);
  }
}

void M64_get_palette(int first, int last, unsigned char *pal)
{
  int i;
  unsigned char *p = pal;

  outportb(Dac_Regs, (unsigned char)first);
  for (i=first; i <= last; i++) {
    *p++ = inportb(Dac_Regs+3);
    *p++ = inportb(Dac_Regs+3);
    *p++ = inportb(Dac_Regs+3);
  }
}

int M64_set_display_start(unsigned long offset)
{
  unsigned long tmp;

  tmp = inportl(Crtc_Off_Pitch) & 0xFFF00000;
  offset = (offset / 8) & 0x000FFFFF;
  tmp += offset;
  outportl(Crtc_Off_Pitch, tmp);

  if (inportl(Crtc_Off_Pitch) != tmp) return M64_ERROR;
  else return M64_OK;
}

unsigned long M64_get_display_start(void)
{
  return (inportl(Crtc_Off_Pitch) & 0x000FFFFF) * 8;
}

int M64_set_pitch(int pitch)
{
  unsigned long tmp;

  tmp = inportl(Crtc_Off_Pitch) & 0x000FFFFF;
  tmp |= ((unsigned long)pitch / 8) << 22;
  outportl(Crtc_Off_Pitch, tmp);
  if (inportl(Crtc_Off_Pitch) != tmp) return M64_ERROR;
  else return M64_OK;
}

int M64_get_pitch(void)
{
  return (inportl(Crtc_Off_Pitch) >> 22) * 8;
}

#define DAC_EXT_SEL_RS2		0x01
#define DAC_EXT_SEL_RS3		0x02

int M64_switch_dac(int dac_cfg)
{
   unsigned char gmr  = 0x83;
   unsigned char dsra = 0x60;
   unsigned char tmp;

   if (((inportw(Config_Stat0) >> 9) & 7) != 5)
				   return M64_FUNCTION_NOT_SUPPORTED;

   switch (inportb(Crtc_Gen_Cntl+1) & 7) {
     case 1: return M64_UNSUPPORTED_BIT_DEPTH;
     case 2: switch (dac_cfg) {
	       case DAC_8_8BIT_LUT: break;
	       case DAC_DEFAULT   :
	       case DAC_8_6BIT_LUT: dsra |= 0x01;
				    break;
	       default: return M64_INVALID_DAC_CFG;
	     }
	     break;
     case 3: if ((dac_cfg != DAC_16_555) && (dac_cfg != DAC_DEFAULT))
						return M64_INVALID_DAC_CFG;
	     return M64_OK;
     case 4: switch (dac_cfg) {
	       case DAC_16_555 : gmr = 0xA0; break;
	       case DAC_DEFAULT:
	       case DAC_16_565 : gmr = 0xA1; break;
	       case DAC_16_664 : gmr = 0xA2; break;
	       case DAC_16_655 : gmr = 0xA3; break;
	       default: return M64_INVALID_DAC_CFG;
	     }
	     break;
     case 5: switch (dac_cfg) {
	       case DAC_24_RGB : gmr = 0xC1; break;
	       case DAC_DEFAULT:
	       case DAC_24_BGR : gmr = 0xC0; break;
	       default: return M64_INVALID_DAC_CFG;
	     }
	     break;
     case 6: switch (dac_cfg) {
	       case DAC_32_RGBa: gmr = 0xE1; break;
	       case DAC_DEFAULT:
	       case DAC_32_aBGR: gmr = 0xE3; break;
	       default: return M64_INVALID_DAC_CFG;
	     }
	     break;
    default: return M64_UNKNOWN_BIT_DEPTH;
  }

  tmp = inportb(Dac_Cntl);
  outportb(Dac_Cntl, (tmp & ~DAC_EXT_SEL_RS2) | DAC_EXT_SEL_RS3);

  outportb(Dac_Regs+2, 0x1d);
  outportb(Dac_Regs+3, gmr);
  outportb(Dac_Regs,   0x02);

  tmp = inportb(Dac_Cntl);
  outportb(Dac_Cntl, tmp | DAC_EXT_SEL_RS2 | DAC_EXT_SEL_RS3);
  switch (inportb(Mem_Cntl) & 7) {
    case 0 : dsra |= 0x04; break;
    case 1 : dsra |= 0x08; break;
    default: dsra |= 0x0c; break;
  }

  tmp = inportb(Dac_Regs);
  outportb(Dac_Regs, dsra | (tmp & 0x80));
  tmp = inportb(Dac_Cntl);
  outportb(Dac_Cntl, (tmp & ~(DAC_EXT_SEL_RS2 | DAC_EXT_SEL_RS3)));

  return M64_OK;
}

int M64_get_mode_info(int *width, int *height, int *bpp, int *bpl)
{
  *width  = (int)(inportw(Crtc_H_Total_Disp+2) + 1) * 8;
  *height = (int)inportw(Crtc_V_Total_Disp+2) + 1;
  switch (inportb(Crtc_Gen_Cntl+1) & 7) {
    case 1 : *bpp = 4;   break;
    case 2 : *bpp = 8;   break;
    case 3 : *bpp = 15;  break;
    case 4 : *bpp = 16;  break;
    case 5 : *bpp = 24;  break;
    case 6 : *bpp = 32;  break;
    default: return M64_UNKNOWN_BIT_DEPTH;
  }
  *bpl = ((int)inportl(Crtc_Off_Pitch) >> 22) * ((*bpp+3) / 4) * 4;
  return M64_OK;
}

