/*
 * Copyright (C) 1996, jack
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/* $Id: setlocal.c,v 1.4 1996/10/25 14:00:59 jack Exp $ */

#include <locale.h>
#include <stddef.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>

#define NUMLOCALE (sizeof (current_locales) / sizeof (struct __current_locale))
#define MAXNAME (64)
#define MAXLOCALEALL (NUMLOCALE * MAXNAME)
#define LCN_C "C"
#define LCN_JA_JP_SJIS "ja_JP.SJIS"

struct __current_locale
{
  int category;
  char *lcname;
  char *locale;
};

struct __alias_locale
{
  char *alias;
  char *locale;
};

struct __tz_locale
{
  char *tzname;
  char *locale;
};

static char *locales[] =
{
  LCN_C,
  LCN_JA_JP_SJIS,
  NULL,
};

static struct __alias_locale alias_locales[] =
{
  { "POSIX", LCN_C },
  { "american", LCN_C },
  { "english-usa", LCN_C },
  { "japanese", LCN_JA_JP_SJIS },
  { NULL, NULL },
};

static struct __tz_locale tz_locales[] =
{
  { "JST", LCN_JA_JP_SJIS },
  { NULL, NULL },
};

static struct __current_locale current_locales[] =
{
  { LC_COLLATE, "LC_COLLATE", LCN_C },
  { LC_CTYPE, "LC_CTYPE", LCN_C },
  { LC_MONETARY, "LC_MONETARY", LCN_C },
  { LC_NUMERIC, "LC_NUMERIC", LCN_C },
  { LC_TIME, "LC_TIME", LCN_C },
  { -1, NULL, NULL },
};

static char locale_all[MAXLOCALEALL];

extern void __mbinit (int category, const char *locale);
extern void __wcinit (int category, const char *locale);


static char *
expand_alias (const char *locale)
{
  int i;

  /* serach in array */
  for (i = 0; alias_locales[i].alias != NULL; i++)
    if (strcmp (alias_locales[i].alias, locale) == 0)
      return alias_locales[i].locale;

  /* not found */
  return (char *) locale;
}

static char *
expand_getenv (const char *name)
{
  char *env;

  env = getenv (name);
  if (env != NULL)
    env = expand_alias (env);
  return env;
}

static char *
find_self (int category)
{
  char *env;
  int i;

  /* assume from environment variables */
  if (category == LC_ALL)
    {
      env = expand_getenv ("LC_ALL");
      if (env != NULL)
	return env;
    }
  else
    {
      for (i = 0; current_locales[i].lcname != NULL; i++)
	if (current_locales[i].category == category)
	  {
	    env = expand_getenv (current_locales[i].lcname);
	    if (env != NULL)
	      return env;
	  }
    }

  /* assume from locale environment variable */
  env = expand_getenv ("LOCALE");
  if (env != NULL)
    {
      for (i = 0; locales[i] != NULL; i++)
	if (strcmp (locales[i], env) == 0)
	  return locales[i];
    }

  /* assume from language */
  env = expand_getenv ("LANG");
  if (env != NULL)
    return env;
  env = expand_getenv ("LANGUAGE");
  if (env != NULL)
    return env;

  /* assume from timezone */
  env = expand_getenv ("TZ");
  if (env != NULL)
    {
      for (i = 0; tz_locales[i].tzname != NULL; i++)
	if (strncmp (tz_locales[i].tzname, env,
		     strlen (tz_locales[i].tzname)) == 0)
	  return tz_locales[i].locale;
    }

  /* I don't know */
  return NULL;
}

static void
put_locale_name (int category, const char *locale)
{
  int i;

  for (i = 0; current_locales[i].locale != NULL; i++)
    if (current_locales[i].category == category)
      {
	current_locales[i].locale = (char *) locale;
	break;
      }
}

static char *
get_locale_name (int category)
{
  int i;

  for (i = 0; current_locales[i].locale != NULL; i++)
    if (current_locales[i].category == category)
      return current_locales[i].locale;

  return "Unknown";
}

static void
change_current (int category, const char *locale)
{
  int i;

  /* change current name */
  if (category == LC_ALL)
    {
      for (i = 0; current_locales[i].locale != NULL; i++)
	current_locales[i].locale = (char *) locale;
    }
  else
    put_locale_name (category, locale);

  /* initialize other module */
  __mbinit (category, locale);
  __wcinit (category, locale);
}

static void
make_locale_all (char *s)
{
  int i;

  for (i = 0; current_locales[i].locale != NULL; i++)
    {
      sprintf (s, "%s=%s;", current_locales[i].lcname,
	       current_locales[i].locale);
      s += strlen (s);
    }
  s[-1] = '\0';
}

char *
setlocale (int category, const char *locale)
{
  if (locale != NULL)
    {
      if (*locale == '\0')
	{
	  locale = find_self (category);
	  if (locale == NULL)
	    return NULL;
	}
      locale = expand_alias (locale);

      if (strcmp (locale, LCN_C) == 0)
	{
	  change_current (category, locale);
	  /* ... */
	}
      else if (strcmp (locale, LCN_JA_JP_SJIS) == 0)
	{
	  change_current (category, locale);
	  /* ... */
	}
      else
	{
	  /* nothing to do */
	  return NULL;
	}
    }

  if (category == LC_ALL)
    {
      make_locale_all (locale_all);
      return locale_all;
    }
  else
    locale = get_locale_name (category);

  return (char *) locale;
}
