/****************************************************************************/
/* MOUSE                                                                    */
/*--------------------------------------------------------------------------*/
/* Fonctions de manipulation de la souris                                   */
/*--------------------------------------------------------------------------*/
/* Auteur     : DELPRAT Jean-Pierre                                         */
/* Cr le    : 10/01/95                                                    */
/****************************************************************************/

#include <conio.h>
#include <mem.h>
#include <stdlib.h>
#include <time.h>
#include <string.h>	// memcpy()

#include "Compat.h"
#include "Errors.h"
#include "JPData.h"
#include "Settings.h"
#include "Screen.h"

#include "Mouse.h"

/*ͻ*/
/*                                CONSTANTES                              */
/*ͼ*/

// Interruption souris

#define INT_MOUSE 0x33

// Hauteur maximum d'un pointer

#define MAX_HEIGHT  16

// Nombre de curseurs dfinis

#define NB_POINTERS  2


/*ͻ*/
/*                      VARIABLES STATIQUES EXTERNES                      */
/*ͼ*/

// A VGA card is present : is only significant when the screen has
// been initialized

extern bool   s_vga_card_present;


/*ͻ*/
/*                            VARIABLES STATIQUES                         */
/*ͼ*/

// Driver souris initialis

static bool s_mouse_driver_initialized=false;
// Souris initialise

static bool s_mouse_initialized=false;

// Graphical mouse pointer

static TMousePointerAspect s_mouse_pointer_aspect=MPA_GRAPHICAL;

// Curseur en cours

static TMousePointer s_mouse_pointer=MP_ARROW;

// Pointeurs souris

static char  s_pointers[((MAX_HEIGHT*2)+2)*NB_POINTERS];

// Pointeur sur le curseur en cours

static char  *s_pointer_definition_ptr=&(s_pointers[0]);

// Point chaud du curseur (p/r  l'origine)

static int    s_hot_spot_x=0;
static int    s_hot_spot_y=0;

// Coordonnes du curseur

static int  s_graph_x,s_graph_y;

// Stockage des caractres sous le curseur
// Coordonnes de ces caractres
// Pointeur sur la position en cours

static int  s_text_x=0,s_text_y=0;
static char s_pointer_background[4];
static int  s_screen_ptr=-1;

// Etat de visibilit de la souris (incr. ou dcr. par ShowMouse/HideMouse)
// (>=0 => visible, <0 => invisible)

static int  s_show_state=-1;

// Taille cran et fonte

static int s_screen_width=80;
static int s_font_height=16;

// Divers pour acclerer le pointer

static int s_last_line=24;
static int s_last_col=79;
static int s_gap_max_height=0;
static int s_gap_32=16;

#ifdef __TCPLUSPLUS__
#else

// Pour installer le gestionnaire d'vnements de la souris en mode protg

static _go32_dpmi_registers s_mouse_regs;
static _go32_dpmi_seginfo   s_mouse_seginfo;
static bool s_callback_allocated=false;

#endif

/*ͻ*/
/*                             FONCTIONS PRIVEES                          */
/*ͼ*/


#ifdef __TCPLUSPLUS__
#include "MouseTC.cpp"
#else
#include "MouseGCC.cpp"
#endif


#ifdef __TCPLUSPLUS__
#else

static void LockMouse() __attribute__ ((constructor));

static void LockMouse()
{
  LOCK_VARIABLE(s_mouse_pointer_aspect);
  LOCK_VARIABLE(s_pointers);
  LOCK_VARIABLE(s_pointer_definition_ptr);
  LOCK_VARIABLE(s_graph_x);
  LOCK_VARIABLE(s_graph_y);
  LOCK_VARIABLE(s_text_x);
  LOCK_VARIABLE(s_text_y);
  LOCK_VARIABLE(s_pointer_background);
  LOCK_VARIABLE(s_screen_ptr);
  LOCK_VARIABLE(s_show_state);
  LOCK_VARIABLE(s_screen_width);
  LOCK_VARIABLE(s_font_height);
  LOCK_VARIABLE(s_last_line);
  LOCK_VARIABLE(s_last_col);
  LOCK_VARIABLE(s_gap_max_height);
  LOCK_VARIABLE(s_gap_32);

  LOCK_FUNCTION(RestoreBackground);
  LOCK_FUNCTION(DisplayMousePointer);
  LOCK_FUNCTION(MouseMovedEvent);
}


static void ExitMouse() __attribute__ ((destructor));

static void ExitMouse()
{
  // On libre le callback

  if (s_callback_allocated)
    _go32_dpmi_free_real_mode_callback(&s_mouse_seginfo);
}

#endif


/****************************************************************************/
/* InitMouse                                                                */
/*--------------------------------------------------------------------------*/
/* Initialise la souris et retourne true si le pilote est install          */
/****************************************************************************/

bool InitMouse()
{
  char *pointer_file;
  char *buffer;
  long length;
  int  screen_height;

  __dpmi_regs    regs;

  #ifdef __TCPLUSPLUS__
  struct REGPACK regs2;
  #endif

  // L'cran doit etre initialise au prealable

  if (!ScreenInitialized())
    return false;

  // Souris dj initialise ?

  if (s_mouse_initialized)
    CloseMouse();

  // Driver souris install ?

  regs.x.ax = 0;
  __dpmi_int(INT_MOUSE,&regs);

  if (regs.x.ax!=0xFFFF)
    return false;

  s_mouse_driver_initialized=true;
  s_mouse_initialized=true;
  s_mouse_pointer=MP_ARROW;
  s_show_state=-1; // Curseur cach

  // C'est bon pour le mode "pointeur affiche par le driver souris"
  //  -> on laisse faire le gestionnaire

  if (s_mouse_pointer_aspect==MPA_MOUSE_DRIVER)
    return true;

  // Paramtres divers

  s_screen_width=GetScreenWidth();
  screen_height=GetScreenHeight();
  s_last_line=screen_height-1;
  s_last_col=s_screen_width-1;

  if (s_mouse_pointer_aspect==MPA_GRAPHICAL)
    {
      s_font_height=GetFontHeight();
      s_gap_max_height=MAX_HEIGHT-s_font_height;
      s_gap_32=32-s_font_height;

      // Chargement des pointeurs souris

      switch (s_font_height)
	{
	  case 8 : pointer_file="JP8x8.PTR";
		   break;
	  case 14: pointer_file="JP8x14.PTR";
		   break;
	  default: pointer_file="JP8x16.PTR";
		   break;
	}

      LoadDataFile(pointer_file,buffer,length);
      if ((buffer==NULL) || (length!=(NB_POINTERS*((s_font_height<<1)+2))))
	{
	  if (buffer!=NULL)
	    delete []buffer;
	  FatalError(GetInvalidDataFileMessage());
	}

      memcpy(s_pointers,buffer,(size_t)length);
      delete []buffer;

      // Rglage de la sensitivit

      regs.x.ax=0x000F;
      regs.x.cx=regs.x.dx=1;
      __dpmi_int(INT_MOUSE,&regs);

      // Modification du curseur
      // En mme temps, fixe les limites

      SetMousePointer(MP_ARROW);

      // Initialisation du curseur  l'cran

      s_graph_x=((s_screen_width*8)>>1)-1;
      s_graph_y=((screen_height*s_font_height)>>1)-1;
      s_graph_x-=s_hot_spot_x;
      s_graph_y-=s_hot_spot_y;

    }
  else
    {
      // Limites de la souris

      regs.x.ax=0x0007;
      regs.x.cx=0;
      regs.x.dx=(s_screen_width-1)<<3;
      __dpmi_int(INT_MOUSE,&regs);

      regs.x.ax=0x0008;
      regs.x.cx=0;
      regs.x.dx=(screen_height-1)<<3;
      __dpmi_int(INT_MOUSE,&regs);

      s_graph_x=(s_screen_width>>1)-1;
      s_graph_y=(screen_height>>1)-1;
    }

  s_screen_ptr=-1;

  regs.x.ax=0x4;
  regs.x.cx=s_graph_x<<3;
  regs.x.dx=s_graph_y<<3;
  __dpmi_int(INT_MOUSE,&regs);

  // Installe le gestionnaire d'vnement

  #ifdef __TCPLUSPLUS__

  regs2.r_ax=0x000C;
  regs2.r_cx=1;        // Masque d'vnement : mouvement de la souris
  regs2.r_es=FP_SEG(MouseMovedEvent);
  regs2.r_dx=FP_OFF(MouseMovedEvent);
  intr(INT_MOUSE,&regs2);

  #else

  if (!s_callback_allocated)
    {
      s_mouse_seginfo.pm_selector=_my_cs();
      s_mouse_seginfo.pm_offset=(int)MouseMovedEvent;
      _go32_dpmi_allocate_real_mode_callback_retf(&s_mouse_seginfo,
						  &s_mouse_regs);
      s_callback_allocated=true;
    }

  regs.x.ax=0x000C;
  regs.x.cx=1;        // Masque d'vnement : mouvement de la souris
  regs.x.es=s_mouse_seginfo.rm_segment;
  regs.x.dx=s_mouse_seginfo.rm_offset;
  __dpmi_int(INT_MOUSE,&regs);

  #endif

  return true;
}

/********************************/
/* CloseMouse : Ferme le driver */
/* ----------   de la souris    */
/********************************/

void CloseMouse()
{
  __dpmi_regs    regs;

  if (!s_mouse_initialized)
    return;

  // On rinitialise le driver souris

  regs.x.ax = 0;
  __dpmi_int(INT_MOUSE,&regs);

  // On cache la souris

  if (s_mouse_pointer_aspect!=MPA_MOUSE_DRIVER)
    RestoreBackground();


  s_mouse_initialized=false;
}


/*************************************/
/* ShowMouse : Affiche le curseur de */
/* ---------   la souris  l'cran   */
/*************************************/

void ShowMouse()
{
  __dpmi_regs    regs;


  if (!s_mouse_initialized)
    return;

  s_show_state++;

  if (s_mouse_pointer_aspect==MPA_MOUSE_DRIVER)
    {
      regs.x.ax = 1;
    __dpmi_int(INT_MOUSE,&regs);
    }
  else
    {
      if (s_show_state==0)
	{
	  disable();
	  DisplayMousePointer();
	  enable();
	}
    }
}

/*************************************/
/* HideMouse : Elimine le curseur de */
/* ---------   la souris de l'cran  */
/*************************************/

void HideMouse()
{
  __dpmi_regs    regs;

  if (!s_mouse_initialized)
    return;

 s_show_state--;

  if (s_mouse_pointer_aspect==MPA_MOUSE_DRIVER)
    {
      regs.x.ax = 2;
      __dpmi_int(INT_MOUSE,&regs);
    }
  else
    {
      if (s_show_state==-1)
	{
	  disable();
	  RestoreBackground();
	  s_screen_ptr=-1;
	  enable();
	}
    }
}

/**************************************************/
/* GetMouseState : Retourne la position et l'tat */
/* -------------   des boutons de la souris       */
/**************************************************/
/* x et y        : entre 1 et ..                  */
/* buttons_state :    LEFT_BUTTON_PRESSED         */
/*                 ou RIGHT_BUTTON_PRESSED        */
/*                 ou NO_BUTTON_PRESSED           */
/**************************************************/

void GetMouseState(int &x,int &y,int &button_state)
{
  __dpmi_regs regs;

  if (!s_mouse_driver_initialized)
    {
      x=1;
      y=1;
      button_state=NO_BUTTON_PRESSED;
      return;
    }

  regs.x.ax=0x3;
  __dpmi_int(INT_MOUSE,&regs);

  if ((s_mouse_initialized) && (s_mouse_pointer_aspect==MPA_GRAPHICAL))
    {
      x=((s_graph_x+s_hot_spot_x)>>3)+1;
      y=((s_graph_y+s_hot_spot_y)/s_font_height)+1;
    }
  else
    {
      x=(regs.x.cx>>3)+1;
      y=(regs.x.dx>>3)+1;
    }

  if ((1 & regs.x.bx)!=0)
    button_state=LEFT_BUTTON_PRESSED;
  else
    {
      if((2 & regs.x.bx)!=0)
	button_state=RIGHT_BUTTON_PRESSED;
      else
	button_state=NO_BUTTON_PRESSED;
    }
}

/****************************************************************************/
/* SetMousePointerAspect         					    */
/*--------------------------------------------------------------------------*/
/* Sets the mouse pointer aspect (graphical, textual or displayed by driver */
/****************************************************************************/

void SetMousePointerAspect(TMousePointerAspect aspect)
{
  int old_show_state;

  if ((ScreenInitialized()) &&
      (!s_vga_card_present) &&
      (aspect==MPA_GRAPHICAL))
    return;

  if (aspect!=s_mouse_pointer_aspect)
    {
     if (s_mouse_initialized)
       {
	 old_show_state=s_show_state;

	 CloseMouse();
	 s_mouse_pointer_aspect=aspect;
	 InitMouse();

	 while (old_show_state>s_show_state)
	   ShowMouse();
	 while (old_show_state<s_show_state)
	   HideMouse();
       }
     else
       s_mouse_pointer_aspect=aspect;
    }
}

/****************************************************************************/
/* SetMousePointer                                                          */
/*--------------------------------------------------------------------------*/
/* Modifie l'aspect du curseur souris, et les limites de dplacement.       */
/* Ne fait rien si on n'a pas le curseur graphique                          */
/****************************************************************************/

void SetMousePointer(TMousePointer pointer)
{
  __dpmi_regs regs;

  if (!s_mouse_initialized)
    return;

  s_mouse_pointer=pointer;

  if (s_mouse_pointer_aspect==MPA_GRAPHICAL)
    {
      // Cache la souris

      HideMouse();

      s_graph_x+=s_hot_spot_x;
      s_graph_y+=s_hot_spot_y;

      // Modifie le curseur

      s_pointer_definition_ptr=s_pointers+( ((int)pointer)*((s_font_height<<1)+2) ) ;

      s_hot_spot_x=s_pointer_definition_ptr[0];
      s_hot_spot_y=s_pointer_definition_ptr[1];

      // Limites de la souris

      regs.x.ax=0x0007;
      regs.x.cx=0;
      regs.x.dx=((s_screen_width*8)-1-s_hot_spot_x)<<3;
      __dpmi_int(INT_MOUSE,&regs);

      regs.x.ax=0x0008;
      regs.x.cx=0;
      regs.x.dx=((GetScreenHeight()*s_font_height)-1-s_hot_spot_y)<<3;
      __dpmi_int(INT_MOUSE,&regs);

      // Position de la souris

      s_graph_x-=s_hot_spot_x;
      s_graph_y-=s_hot_spot_y;

      regs.x.ax=0x4;
      regs.x.cx=s_graph_x<<3;
      regs.x.dx=s_graph_y<<3;
      __dpmi_int(INT_MOUSE,&regs);

      // Raffiche la souris

      ShowMouse();
   }
}

TMousePointer GetMousePointer()
{
  return(s_mouse_pointer);
}




/****************************************************************************/
/* WaitMouseLeftButtonRelease                                               */
/*--------------------------------------------------------------------------*/
/* Attend que le bouton gauche de la souris soit relch                    */
/****************************************************************************/

void WaitMouseLeftButtonRelease()
{
  int x,y,button_state;

  do
    GetMouseState(x,y,button_state);
  while
    (button_state==LEFT_BUTTON_PRESSED);
}
