#include <keys.h>

#include "appl.h"
#include "desktop.h"
#include "dlgobj.h"
#include "error.h"
#include "event.h"
#include "group.h"
#include "refresh.h"

/*

	group.c

		Handles the	object groups (ie dialogs, menu bars and the desktop)

		C-Desktop
		Copyright (C)1998, Brett Porter.

*/

/*
	FUNCTION	PROTOTYPES
*/

static	void	Group_Show( T_Group *aGroup );
static	void	Group_Hide( T_Group *aGroup );

extern	void	Screen_GetBuffer( byte *aBuffer, int aX, int aY, int aW, int aH );

/*
	FUNCTION DEFINITIONS
*/

T_Group	*Group_Initialise( int aX1, int aY1, int aX2, int aY2, int aFlags )
{
	T_Group	*lGroup = NULL;

	int	lChange;

	P_InitVar( lGroup, T_Group );

	lGroup->fFlags = aFlags;

	if ( aFlags & FLAG_CENTREX )
	{
		if ( gDesktop == NULL )
		{
			FatalError( "Can't centre group until desktop group is created" );
		}
		lChange = (( gDesktop->fX2-gDesktop->fX1+1 )-( aX2-aX1+1 ))/2-aX1;	/// should this be within the owner, or the screen, or what?
		aX1 += lChange;
		aX2 += lChange;
	}
	if ( aFlags & FLAG_CENTREY )
	{
		if ( gDesktop == NULL )
		{
			FatalError( "Can't centre group until desktop group is created" );
		}
		lChange = (( gDesktop->fY2-gDesktop->fY1+1 )-( aY2-aY1+1 ))/2-aY1;	/// see above...
		aY1 += lChange;
		aY2 += lChange;
	}

	lGroup->fX1 = aX1;
	lGroup->fY1 = aY1;
	lGroup->fX2 = aX2;
	lGroup->fY2 = aY2;
	lGroup->fVisible = false;
	lGroup->fCloseModalCommand = ENDMODAL_NONE;

	lGroup->fObjectList = NULL;

	return lGroup;
}

void	Group_AddObject( T_Group *aGroup, void *aObject, int aX1, int aY1, int aX2, int aY2, int aFlags )
{
	T_ObjectRec	*lRec, *lCurrent;

	T_MinimumDesktopObject	*lObjectHeader = aObject;

	int	lChange;

	P_InitVar( lRec, T_ObjectRec );

	if ( aFlags & FLAG_CENTREX )
	{
		lChange = (( aGroup->fX2-aGroup->fX1+1 )-( aX2-aX1+1 ))/2-aX1;
		aX1 += lChange;
		aX2 += lChange;
	}
	if ( aFlags & FLAG_CENTREY )
	{
		lChange = (( aGroup->fY2-aGroup->fY1+1 )-( aY2-aY1+1 ))/2-aY1;
		aY1 += lChange;
		aY2 += lChange;
	}

	lRec->fFlags = aFlags;
	lRec->fX1 = aX1;
	lRec->fY1 = aY1;
	lRec->fX2 = aX2;
	lRec->fY2 = aY2;
	lRec->fOwner = aGroup;
	lRec->fObject.fData = aObject;
	lRec->fNext = NULL;
	lRec->fObject.fHeader->fOwner = lRec;

	Refresh_AddDirtyRectangle( lRec );

	lObjectHeader->fWidth = aX2-aX1+1;
	lObjectHeader->fHeight = aY2-aY1+1;

	if ( aGroup->fObjectList == NULL )
	{
		aGroup->fObjectList = lRec;
	}
	else
	{
		lCurrent = aGroup->fObjectList;

		while ( lCurrent->fNext != NULL )
		{
			lCurrent = lCurrent->fNext;
		}
		lCurrent->fNext = lRec;
	}

	Group_FocusObject( aGroup, aObject );
}

void	Group_RemoveObject( T_Group *aGroup, void *aObject )
{
	T_ObjectRec	*lPrev = NULL, *lCurrent = aGroup->fObjectList;

	while ( lCurrent != NULL )
	{
		if ( lCurrent->fObject.fData == aObject )
		{
			break;
		}
		lPrev = lCurrent;
		lCurrent = lCurrent->fNext;
	}

	if ( lCurrent == NULL )
	{
		FatalError( "Object to be removed not found" );
	}

	if ( lPrev == NULL )
	{
		aGroup->fObjectList = lCurrent->fNext;
	}
	else
	{
		lPrev->fNext = lCurrent->fNext;
	}
	Refresh_AddDirtyRectangle( lCurrent );

	DesktopObj_Kill( lCurrent );
}

void	Group_Kill( T_Group *aGroup )
{
	T_ObjectRec	*lRec = aGroup->fObjectList;

	while ( lRec != NULL )
	{
		DesktopObj_Kill( lRec );
		lRec = lRec->fNext;
	}
	free( aGroup );
}

bool	Group_HandleLeftMousePress( T_Group *aGroup, int aColumn, int aRow )
{
	T_ObjectRec	*lRec = aGroup->fObjectList;

	bool	lRetVal = false;

	while ( lRec != NULL )
	{
		if ( aColumn >= lRec->fX1 && aColumn <= lRec->fX2 && aRow >= lRec->fY1 && aRow <= lRec->fY2 )
		{
			if ( DesktopObj_HandleLeftClick( lRec, aColumn-lRec->fX1, aRow-lRec->fY1 ))
			{
				lRetVal = true;
				break;
			}
		}
		lRec = lRec->fNext;
	}
	return lRetVal;
}

bool	Group_HandleRightMousePress( T_Group *aGroup, int aColumn, int aRow )
{
	T_ObjectRec	*lRec = aGroup->fObjectList;

	bool	lRetVal = false;

	while ( lRec != NULL )
	{
		if ( aColumn >= lRec->fX1 && aColumn <= lRec->fX2 && aRow >= lRec->fY1 && aRow <= lRec->fY2 )
		{
			if ( DesktopObj_HandleRightClick( lRec, aColumn-lRec->fX1, aRow-lRec->fY1 ))
			{
				lRetVal = true;
				break;
			}
		}
		lRec = lRec->fNext;
	}
	return lRetVal;
}

void	Group_FocusNext( T_Group *aGroup )
{
	T_ObjectRec	*lRec;

	if ( aGroup->fFocused == NULL || aGroup->fFocused->fNext == NULL )
	{
		lRec = aGroup->fObjectList;
	}
	else
	{
		lRec = aGroup->fFocused->fNext;
	}

	while ( lRec != aGroup->fFocused )
	{
		if ( DesktopObj_CanFocus( lRec ))
		{
			DesktopObj_Focus( lRec );
			break;
		}
		lRec = lRec->fNext;

		if ( lRec == NULL && aGroup->fFocused != NULL )
		{
			lRec = aGroup->fObjectList;
		}
	}
}

void	Group_FocusPrevious( T_Group *aGroup )
{
	T_ObjectRec	*lRec, *lFound;

	if ( aGroup->fFocused == NULL || aGroup->fFocused->fNext == NULL )
	{
		lRec = aGroup->fObjectList;
		lFound = NULL;
	}
	else
	{
		lRec = aGroup->fFocused->fNext;
		lFound = lRec;
	}

	while ( lRec != aGroup->fFocused )
	{
		if ( DesktopObj_CanFocus( lRec ))
		{
			lFound = lRec;
		}
		lRec = lRec->fNext;

		if ( lRec == NULL && aGroup->fFocused != NULL )
		{
			lRec = aGroup->fObjectList;
		}
	}
	if ( lFound != NULL )
	{
		DesktopObj_Focus( lFound );
	}
}

T_ObjectRec	*Group_FindObject( T_Group *aGroup, void *aObject )
{
	T_ObjectRec	*lRec = aGroup->fObjectList;

	while ( lRec != NULL )
	{
		if ( lRec->fObject.fData == aObject )
		{
			break;
		}
		lRec = lRec->fNext;
	}
	if ( lRec == NULL )
	{
		FatalError( "Object not found." );
	}
	return lRec;
}

void	Group_FocusObject( T_Group *aGroup, void *aObject )
{
	DesktopObj_Focus( Group_FindObject( aGroup, aObject ));
}

bool	Group_HandleKeypress( T_Group *aGroup, int aKeypress )
{
	T_ObjectRec	*lRec;

	if ( aKeypress == K_Escape && !( aGroup->fFlags & FLAG_NOCLOSEBUTTON ))
	{
		aGroup->fCloseModalCommand = ENDMODAL_CLOSE;
		Event_SendMessage( MSG_CLOSEMODAL );
		return true;
	}
	if ( aGroup->fFocused != NULL && !( aKeypress & 0x100 ))
	{
		if ( DesktopObj_HandleKeypress( aGroup->fFocused, aKeypress ))
		{
			return true;
		}
	}
	lRec = aGroup->fObjectList;

	while ( lRec != NULL )
	{
		if ( DesktopObj_HandleKeypress( lRec, aKeypress ))
		{
			return true;
		}
		lRec = lRec->fNext;
	}
	if ( aKeypress == K_Tab )
	{
		Group_FocusNext( aGroup );
		return true;
	}
	if ( aKeypress == K_BackTab )
	{
		Group_FocusPrevious( aGroup );
		return true;
	}
	return false;
}

void	Group_RedrawObject( T_Group *aGroup, void *aObject )
{
	Refresh_AddDirtyRectangle( Group_FindObject( aGroup, aObject ));
}

bool	Group_Broadcast( T_Group *aGroup, T_EventRec *aEvent )
{
	T_ObjectRec	*lObject = aGroup->fObjectList;
	bool	lRetVal = false;

	while ( lObject != NULL )
	{
		lRetVal |= lObject->fObject.fHeader->fEventHandler( lObject->fObject.fData, aEvent );
		lObject = lObject->fNext;
	}
	return lRetVal;
}

void	Group_Show( T_Group *aGroup )
{
	aGroup->fVisible++;
	if ( aGroup->fVisible == 1 )	/* only just became visible */
	{
		T_EventRec	lEvent;

		lEvent.fMessage = EVENT_SHOW;
		Group_Broadcast( aGroup, &lEvent );
	}
	Refresh_AddDirtyRectangleHelper( aGroup->fX1, aGroup->fY1, aGroup->fX2, aGroup->fY2 );
	/// go thorugh each object and add its rectangle instead?
}

void	Group_Hide( T_Group *aGroup )
{
	if ( aGroup->fVisible > 0 )
	{
		aGroup->fVisible--;
	}
	if ( !aGroup->fVisible )
	{
		T_EventRec	lEvent;

		lEvent.fMessage = EVENT_HIDE;
		Group_Broadcast( aGroup, &lEvent );
	}
	Refresh_AddDirtyRectangleHelper( aGroup->fX1, aGroup->fY1, aGroup->fX2, aGroup->fY2 );
	/// go thorugh each object and add its rectangle instead?
}

void	Group_GoModal( T_Group *aGroup )
{
	T_Group	*lPrevGroup = gEvent_ModalGroup;

	gEvent_ModalGroup = aGroup;

	Group_Show( aGroup );

	Application_Wait( MSG_CLOSEMODAL );

	gEvent_ModalGroup = lPrevGroup;

	Group_Hide( aGroup );

	switch ( aGroup->fCloseModalCommand )
	{
		case	ENDMODAL_OK:
		case	ENDMODAL_CLOSE:
		case	ENDMODAL_YES:
		case	ENDMODAL_CANCEL:
		case	ENDMODAL_NO:
			break;
		case	ENDMODAL_EXECUTECOMMAND:
			Event_HandleCommand( aGroup->fCloseModalID );
			break;
		default:
			FatalError( "Unhandled modal exit method" );
			break;
	}
}

