#include <conio.h>
#include <ctype.h>
#include <keys.h>
#include <stdlib.h>
#include <stdio.h>

#include "buffer.h"
#include "colour.h"
#include "desktop.h"
#include "dialog.h"
#include "dlgobj.h"
#include "error.h"
#include "event.h"
#include "input.h"
#include "misc.h"
#include "refresh.h"
#include "scrollbr.h"

/*

	dlgobj.c

		Dialog box object stuff

		C-Desktop
		Copyright (C)1998, Brett Porter.

*/

/*
	FUNCTION	PROTOTYPES
*/

static	bool	DlgString_EventHandler( void *aString, T_EventRec *aEvent );
static	void	DlgString_Draw( T_DlgString *aString, T_EventDrawParameters *aParams );
static	void	DlgString_Kill( T_DlgString *aString );

static	bool	DlgButton_EventHandler( void *aButton, T_EventRec *aEvent );
static	void	DlgButton_Draw( T_DlgButton *aButton, T_EventDrawParameters *aParams );
static	bool	DlgButton_LeftMouseClick( T_DlgButton *aButton, int aColumn, int aRow );
static	bool	DlgButton_HandleKeypress( T_DlgButton *aButton, int aKeypress );
static	void	DlgButton_Kill( T_DlgButton *aButton );

static	bool	DlgInput_EventHandler( void *aInput, T_EventRec *aEvent );
static	void	DlgInput_Draw( T_DlgInput *aInput, T_EventDrawParameters *aParams );
static	bool	DlgInput_LeftMouseClick( T_DlgInput *aInput, int aColumn, int aRow );
static	bool	DlgInput_HandleKeypress( T_DlgInput *aInput, int aKeypress );
static	void	DlgInput_Kill( T_DlgInput *aInput );
static	inline	void	DlgInput_SetCursorPosition( T_DlgInput *aInput );
static	inline	void	DlgInput_Focus( T_DlgInput *aInput );

static	bool	DlgLabel_EventHandler( void *aLabel, T_EventRec *aEvent );
static	void	DlgLabel_Draw( T_DlgLabel *aLabel, T_EventDrawParameters *aParams );
static	bool	DlgLabel_LeftMouseClick( T_DlgLabel *aLabel );
static	bool	DlgLabel_HandleKeypress( T_DlgLabel *aLabel, int aKeypress );
static	void	DlgLabel_Kill( T_DlgLabel *aLabel );

static	bool	DlgRadioButtons_EventHandler( void *aRadioButtons, T_EventRec *aEvent );
static	void	DlgRadioButtons_Draw( T_DlgRadioButtons *aRadioButtons, T_EventDrawParameters *aParams );
static	bool	DlgRadioButtons_LeftMouseClick( T_DlgRadioButtons *aRadioButtons, int aRow );
static	bool	DlgRadioButtons_HandleKeypress( T_DlgRadioButtons *aRadioButtons, int aKeypress );
static	void	DlgRadioButtons_Kill( T_DlgRadioButtons *aRadioButtons );
static	inline	void	DlgRadioButtons_SetCursorPosition( T_DlgRadioButtons *aRadioButtons );
static	inline	void	DlgRadioButtons_Focus( T_DlgRadioButtons *aRadioButtons );

static	bool	DlgListBox_EventHandler( void *aListBox, T_EventRec *aEvent );
static	void	DlgListBox_Draw( T_DlgListBox *aListBox, T_EventDrawParameters *aParams );
static	bool	DlgListBox_LeftMouseClick( T_DlgListBox *aListBox, int aRow );
static	bool	DlgListBox_HandleKeypress( T_DlgListBox *aListBox, int aKeypress );
static	void	DlgListBox_Kill( T_DlgListBox *aListBox );

/*
	FUNCTION DEFINITIONS
*/

T_DlgString	*DlgString_Initialise( const char *aString, bool aIsCentred )
{
	char			*lPtr;
	T_DlgString	*lString;
	int			lCurWidth;

	P_InitVar( lString, T_DlgString );
	P_CreateStr( lString->fString, aString );

	lString->fTextCentred = aIsCentred;

	lPtr = lString->fString;
	lString->fColumns = 0;
	lString->fLines = 1;
	lCurWidth = 0;

	while ( *lPtr != EOS )
	{
		if ( *lPtr == '\n' )
		{
			if ( lCurWidth > lString->fColumns )
			{
				lString->fColumns = lCurWidth;
			}
			lCurWidth = 0;
			lString->fLines++;
		}
		lCurWidth++;
		lPtr++;
	}
	if ( lCurWidth > lString->fColumns )
	{
		lString->fColumns = lCurWidth;
	}

	DesktopObj_Initialise( lString, DlgString_EventHandler, false );

	return lString;
}

bool	DlgString_EventHandler( void *aString, T_EventRec *aEvent )
{
	bool	lRetVal = false;

	switch ( aEvent->fMessage )
	{
		case	EVENT_CANFOCUS:
			lRetVal = false;
			break;
		case	EVENT_KILL:
			DlgString_Kill( aString );
			lRetVal = true;
			break;
		case	EVENT_DRAW:
			DlgString_Draw( aString, &aEvent->fParameters.fDrawParams );
			lRetVal = true;
			break;
		case	EVENT_LEFTMOUSECLICK:
		case	EVENT_RIGHTMOUSECLICK:
		case	EVENT_KEYPRESS:
		case	EVENT_SHOW:
		case	EVENT_HIDE:
			break;
		default:
			FatalError( "Unhandled message" );
			break;
	}
	return lRetVal;
}

void	DlgString_Draw( T_DlgString *aString, T_EventDrawParameters *aParams )
{
	char	*lPtr, lTempCh, *lStr = aString->fString;
	int	lCurY = aParams->fY, lCentreFactor, lYTarget, lX1, lX2;
	bool	lFlag = false;

	lYTarget = lCurY+aString->fLines-1;

	lX1 =	aParams->fX;
	lX2 = lX1+aString->fWidth-1;

	lPtr = lStr-1;

	do
	{
		if ( lCurY > lYTarget )
		{
			break;
		}

		lPtr++;
		if ( *lPtr == '\n' || *lPtr == EOS )
		{
			lTempCh = *lPtr;

			*lPtr = EOS;

			lCentreFactor = 0;

			if ( aString->fTextCentred )
			{
				lCentreFactor = ( aString->fWidth-strlen( lStr ))/2;

				if ( lCentreFactor < 0 )
				{
					lCentreFactor = 0;
				}
			}

			Buffer_Print( aParams->fBuffer, lX1+lCentreFactor, lCurY, lX2, lCurY, lStr, COLOUR_DIALOG_TEXT );

			lCurY++;

			*lPtr = lTempCh;

			lStr = lPtr+1;

			if ( lTempCh == EOS )
			{
				lFlag = true;
			}
		}
	}
	while ( !lFlag );
}

void	DlgString_Kill( T_DlgString *aString )
{
	free( aString->fString );
	free( aString );
}

T_DlgButton	*DlgButton_Initialise( const char *aString, int aID, bool aIsDefault )
{
	T_DlgButton	*lButton;
	int	i = 0;
	bool	lFound = false;

	P_InitVar( lButton, T_DlgButton );
	lButton->fID = aID;
	lButton->fIsDefault = aIsDefault;

	DesktopObj_Initialise( lButton, DlgButton_EventHandler, true );

	while ( aString[i] != EOS )
	{
		if ( aString[i] == '&' && aString[i+1] != '&' )
		{
			if ( lFound )
			{
				FatalError( "Two hotkeys defined in button" );
			}
			lButton->fHotKeyPosition = i;
			i++;
			lButton->fHotKey = aString[i];
			if ( aString[i] == EOS )
			{
				FatalError( "Can't define EOS as a hotkey" );
			}
			lFound = true;
		}
		i++;
	}
	if ( !lFound )
	{
		lButton->fHotKey = EOS;
	}
	P_CreateStr( lButton->fText, aString );

	if ( lFound )
	{
		strcpy( lButton->fText+lButton->fHotKeyPosition, lButton->fText+lButton->fHotKeyPosition+1 );
	}

	return lButton;
}

bool	DlgButton_EventHandler( void *aButton, T_EventRec *aEvent )
{
	bool	lRetVal = false;

	switch ( aEvent->fMessage )
	{
		case	EVENT_FOCUS:
		case	EVENT_UNFOCUS:
		case	EVENT_CANFOCUS:
			lRetVal = true;
			break;
		case	EVENT_LEFTMOUSECLICK:
			lRetVal = DlgButton_LeftMouseClick( aButton, aEvent->fParameters.fMouseClick.fColumn, aEvent->fParameters.fMouseClick.fRow );
			break;
		case	EVENT_KEYPRESS:
			lRetVal = DlgButton_HandleKeypress( aButton, aEvent->fParameters.fKeypress );
			break;
		case	EVENT_KILL:
			DlgButton_Kill( aButton );
			lRetVal = true;
			break;
		case	EVENT_DRAW:
			DlgButton_Draw( aButton, &aEvent->fParameters.fDrawParams );
			lRetVal = true;
			break;
		case	EVENT_SHOW:
		case	EVENT_HIDE:
			break;
		default:
			FatalError( "Unhandled message" );
			break;
	}
	return lRetVal;
}

void	DlgButton_Draw( T_DlgButton *aButton, T_EventDrawParameters *aParams )
{
	char		*lText = aButton->fText;
	char		lButtonText[21];
	int		lChange, i, lPos, lColour;
	T_Buffer	*lBuffer = aParams->fBuffer;
	int		lX1, lY1, lX2, lY2;

	lX1 = aParams->fX;
	lY1 = aParams->fY;
	lX2 = lX1+aButton->fWidth-1;
	lY2 = lY1+aButton->fHeight-1;

	sprintf( lButtonText, "%20s", "" );
	lChange = (( aButton->fWidth-1 )-strlen( lText ))/2;

	for ( i = strlen( lText ); i--; )
	{
		lButtonText[i+lChange] = lText[i];
	}
	lPos = aButton->fHotKeyPosition+lChange;

	lColour = aButton->fFocused ? COLOUR_BUTTON_FOCUSED : ( aButton->fIsDefault ? COLOUR_BUTTON_DEFAULT : COLOUR_BUTTON_TEXT );

	if ( lColour == COLOUR_BUTTON_DEFAULT && aButton->fOwner->fOwner->fFocused->fObject.fHeader->fCanActivate )	/// yuk!
	{
		lColour = COLOUR_BUTTON_TEXT;
	}

	Buffer_Print( lBuffer, lX1, lY1, lX2-1, lY1, lButtonText, lColour );
	Buffer_Print( lBuffer, lX1+lPos, lY1, lX1+lPos, lY1, lButtonText+lPos, COLOUR_BUTTON_HILITE );
	Buffer_Print( lBuffer, lX2, lY1, lX2, lY1, "", COLOUR_BUTTON_SHDW );
	memset( lButtonText,	'', aButton->fWidth-1 );
	Buffer_Print( lBuffer, lX1+1, lY2, lX2, lY2, lButtonText, COLOUR_BUTTON_SHDW );
}

bool	DlgButton_LeftMouseClick( T_DlgButton *aButton, int aColumn, int aRow )
{
	T_DlgButton	*lButton = aButton;

	if ( aRow == 0 && aColumn < lButton->fWidth-1 )
	{
		Group_FocusObject( gEvent_ModalGroup, lButton );

		gEvent_ModalGroup->fCloseModalCommand = lButton->fID;

		Event_SendMessage( MSG_CLOSEMODAL );

		return true;
	}
	return false;
}

bool	DlgButton_HandleKeypress( T_DlgButton *aButton, int aKeypress )
{
	int	lHotKey;

	if ( aKeypress & 0x100 )
	{
		lHotKey = toupper( kScanToASCII[aKeypress-0x100] );
	}
	else
	{
		lHotKey = toupper( aKeypress );
	}
	if ( lHotKey == toupper( aButton->fHotKey ) || ( aKeypress == K_Return && ( aButton->fFocused || aButton->fIsDefault )))
	{
		Group_FocusObject( gEvent_ModalGroup, aButton );

		gEvent_ModalGroup->fCloseModalCommand = aButton->fID;

		Event_SendMessage( MSG_CLOSEMODAL );

		return true;
	}
	return false;
}

void	DlgButton_Kill( T_DlgButton *aButton )
{
	free( aButton->fText );
	free( aButton );
}

T_DlgInput	*DlgInput_Initialise( const char *aDefault, int aMaxChars, const char *aFilter )
{
	T_DlgInput	*lInput;
   bool			lTakesEnter = false;
   int			i;

	P_InitVar( lInput, T_DlgInput );
	lInput->fText = ( char* )malloc( aMaxChars+1 );
	lInput->fText[aMaxChars] = EOS;
	strncpy( lInput->fText, aDefault, aMaxChars );
	lInput->fMaxChars = aMaxChars;
	lInput->fFilter = aFilter;
	lInput->fCursorPosition = strlen( lInput->fText );

	i = 0;

   while ( aFilter[i] != EOS )
   {
   	if ( aFilter[i] == 13 )
      {
      	lTakesEnter = true;
      	break;
      }
		i++;
   }

	DesktopObj_Initialise( lInput, DlgInput_EventHandler, lTakesEnter );

	return lInput;
}

inline	void	DlgInput_Focus( T_DlgInput *aInput )
{
	if ( aInput->fOwner->fOwner->fVisible )
	{
		DlgInput_SetCursorPosition( aInput );
		Screen_SetCursorType( aInput->fOverwrite ? _SOLIDCURSOR : _NORMALCURSOR );
	}
}

bool	DlgInput_EventHandler( void *aInput, T_EventRec *aEvent )
{
	bool	lRetVal = false;

	T_DlgInput	*lInput = aInput;

	switch ( aEvent->fMessage )
	{
		case	EVENT_LEFTMOUSECLICK:
			lRetVal = DlgInput_LeftMouseClick( aInput, aEvent->fParameters.fMouseClick.fColumn, aEvent->fParameters.fMouseClick.fRow );
			break;
		case	EVENT_KEYPRESS:
			lRetVal = DlgInput_HandleKeypress( aInput, aEvent->fParameters.fKeypress );
			break;
		case	EVENT_KILL:
			DlgInput_Kill( aInput );
			lRetVal = true;
			break;
		case	EVENT_DRAW:
			DlgInput_Draw( aInput, &aEvent->fParameters.fDrawParams );
			lRetVal = true;
			break;
		case	EVENT_CANFOCUS:
			lRetVal = true;
			break;
		case	EVENT_SHOW:
			if ( !lInput->fFocused )
			{
				break;
			}
		case	EVENT_FOCUS:
			DlgInput_Focus( aInput );
			lRetVal = true;
			break;
		case	EVENT_HIDE:
			if ( !lInput->fFocused )
			{
				break;
			}
		case	EVENT_UNFOCUS:
			Screen_SetCursorType( _NOCURSOR );
			lRetVal = true;
			break;
		default:
			FatalError( "Unhandled message" );
			break;
	}
	return lRetVal;
}

void	DlgInput_Draw( T_DlgInput *aInput, T_EventDrawParameters *aParams )
{
	int	lX1 = aParams->fX;
	int	lY1 = aParams->fY;
	int	lX2 = lX1+aInput->fWidth-1;
	int	lY2 = lY1+aInput->fHeight-1;

	char	lStr[81], lOutputStr[81];

	T_Buffer	*lBuffer = aParams->fBuffer;

	while ( aInput->fCursorPosition-aInput->fStartLetter > aInput->fWidth-2 )
	{
		aInput->fStartLetter++;
	}
	if ( aInput->fCursorPosition < aInput->fStartLetter )
	{
		aInput->fStartLetter = aInput->fCursorPosition;
	}

	memset( lStr, ' ', 80 );
	lStr[80] = EOS;

	strncpy( lOutputStr, aInput->fText+aInput->fStartLetter, aInput->fWidth-2 );
	lOutputStr[aInput->fWidth-2] = EOS;
   sprintf( lStr+1, "%-*s", aInput->fWidth-2, lOutputStr );
	lStr[aInput->fWidth-1] = ' ';

	Buffer_Print( lBuffer, lX1, lY1, lX2, lY2, lStr, COLOUR_INPUT_TEXT );

	if ( aInput->fStartLetter > 0 )
	{
		Buffer_Print( lBuffer, lX1, lY1, lX1, lY2, "\x11", COLOUR_INPUT_ARROW );
	}
	if ( aInput->fStartLetter+aInput->fWidth-1 <= ( signed )strlen( aInput->fText ))
	{
		Buffer_Print( lBuffer, lX2, lY1, lX2, lY2, "\x10", COLOUR_INPUT_ARROW );
	}
	if ( aInput->fFocused && aInput->fOwner->fOwner->fVisible )
	{
		DlgInput_SetCursorPosition( aInput );
	}
}

inline	void	DlgInput_SetCursorPosition( T_DlgInput *aInput )
{
	T_ObjectRec	*lRec = aInput->fOwner;
	Screen_SetCursorPosition( lRec->fOwner->fX1+lRec->fX1+aInput->fCursorPosition-aInput->fStartLetter+1, lRec->fOwner->fY1+lRec->fY1 );
}

bool	DlgInput_LeftMouseClick( T_DlgInput *aInput, int aColumn, int aRow )
{
	int	lNewPos;

	if ( aRow != -1 && aColumn > 0 && aColumn < aInput->fWidth )
	{
		lNewPos = aColumn-1+aInput->fStartLetter;
		if ( lNewPos > ( signed )strlen( aInput->fText ))
		{
			lNewPos = strlen( aInput->fText );
		}
		aInput->fCursorPosition = lNewPos;
      Refresh_AddDirtyRectangle( aInput->fOwner );
	}
   Group_FocusObject( aInput->fOwner->fOwner, aInput );

	return true;
}

void	DlgInput_SetString( T_DlgInput *aInput, const char *aString )
{
	strncpy( aInput->fText, aString, aInput->fMaxChars );
	aInput->fCursorPosition = strlen( aInput->fText );
}

bool	DlgInput_HandleKeypress( T_DlgInput *aInput, int aKeypress )
{
	bool		lReturnValue = false, lAnything = ( aInput->fFilter[0] == EOS ) ? true : false;
	int		i, lStrLen = strlen( aInput->fText );

	if ( !aInput->fFocused )
	{
		return false;
	}

	if ( aKeypress == K_ELeft )
	{
		if ( aInput->fCursorPosition > 0 )
		{
			aInput->fCursorPosition--;
		}
		lReturnValue = true;
	}
	if ( aKeypress == K_ERight )
	{
		if ( aInput->fCursorPosition < lStrLen )
		{
			aInput->fCursorPosition++;
		}
		lReturnValue = true;
	}
	if ( aKeypress == K_EHome )
	{
		aInput->fCursorPosition = 0;
		lReturnValue = true;
	}
	if ( aKeypress == K_EEnd )
	{
		aInput->fCursorPosition = strlen( aInput->fText );
		lReturnValue = true;
	}
	if ( aKeypress == K_EInsert )
	{
		aInput->fOverwrite = !aInput->fOverwrite;
		lReturnValue = true;
	}
	if ( aKeypress == K_Control_ELeft )
	{
		if ( aInput->fCursorPosition )
		{
			for ( i = aInput->fCursorPosition-2; i > 0; i-- )
			{
				if ( aInput->fText[i] == ' ' )
				{
					i++;
					break;
				}
			}
			aInput->fCursorPosition = i;
			lReturnValue = true;
		}
	}
	if ( aKeypress == K_Control_ERight )
	{
		for ( i = aInput->fCursorPosition; i < lStrLen; i++ )
		{
			if ( aInput->fText[i] == ' ' )
			{
				while ( aInput->fText[i] == ' ' )
				{
					i++;
				}
				break;
			}
		}
		aInput->fCursorPosition = i;
		lReturnValue = true;
	}
   if ( aKeypress == K_BackSpace )
   {
   	if ( aInput->fCursorPosition > 0 )
      {
			aInput->fCursorPosition--;
			for ( i = aInput->fCursorPosition; i < lStrLen; i++ )
			{
				aInput->fText[i] = aInput->fText[i+1];
			}
      }
		lReturnValue = true;
   }
   if ( aKeypress == K_EDelete )
   {
      if ( aInput->fCursorPosition < lStrLen )
      {
			for ( i = aInput->fCursorPosition; i < lStrLen; i++ )
			{
				aInput->fText[i] = aInput->fText[i+1];
			}
      }
		lReturnValue = true;
   }
	if ( lReturnValue == false )
	{
  		if ( aKeypress < 128 && aKeypress >= ' ' )
      {
			lReturnValue = true;
	     	if ( strchr( aInput->fFilter, aKeypress ) || lAnything )
   	  	{
				if ( lStrLen < aInput->fMaxChars || ( aInput->fOverwrite && aInput->fCursorPosition != lStrLen ))
				{
					if ( !aInput->fOverwrite || ( aInput->fOverwrite && aInput->fCursorPosition == lStrLen ))
					{
						for ( i = lStrLen; i > aInput->fCursorPosition; i-- )
						{
							aInput->fText[i] = aInput->fText[i-1];
						}
						lStrLen++;
					}
	  		      aInput->fText[aInput->fCursorPosition] = aKeypress;
   			   aInput->fText[lStrLen] = EOS;
					aInput->fCursorPosition++;
				}
			}
   	}
	}
	if ( lReturnValue )
	{
		Refresh_AddDirtyRectangle( aInput->fOwner );
	}
	return lReturnValue;
}

void	DlgInput_Kill( T_DlgInput *aInput )
{
	free( aInput->fText );
	free( aInput );
}

T_DlgLabel	*DlgLabel_Initialise( const char *aText, char aHotKey, void *aObject )
{
	T_DlgLabel	*lLabel;
	int			i;

	P_InitVar( lLabel, T_DlgLabel );
	P_CreateStr( lLabel->fText, aText );

	lLabel->fHotKey = aHotKey;
	lLabel->fRelatedObject = aObject;

	DesktopObj_Initialise( lLabel, DlgLabel_EventHandler, false );

	for ( i = 0; i < ( signed )strlen( aText ); i++ )
	{
		if ( aText[i] == aHotKey )
		{
			lLabel->fHotKeyPosition = i;
			break;
		}
	}
	return lLabel;
}

bool	DlgLabel_EventHandler( void *aLabel, T_EventRec *aEvent )
{
	bool	lRetVal = false;

	switch ( aEvent->fMessage )
	{
		case	EVENT_LEFTMOUSECLICK:
			lRetVal = DlgLabel_LeftMouseClick( aLabel );
			break;
		case	EVENT_KEYPRESS:
			lRetVal = DlgLabel_HandleKeypress( aLabel, aEvent->fParameters.fKeypress );
			break;
		case	EVENT_KILL:
			DlgLabel_Kill( aLabel );
			lRetVal = true;
			break;
		case	EVENT_DRAW:
			DlgLabel_Draw( aLabel, &aEvent->fParameters.fDrawParams );
			lRetVal = true;
			break;
		case	EVENT_FOCUS:
		case	EVENT_UNFOCUS:
		case	EVENT_CANFOCUS:
		case	EVENT_SHOW:
		case	EVENT_HIDE:
			break;
		default:
			FatalError( "Unhandled message" );
			break;
	}
	return lRetVal;
}

void	DlgLabel_Draw( T_DlgLabel *aLabel, T_EventDrawParameters *aParams )
{
	T_Buffer	*lBuffer = aParams->fBuffer;

	int	lX1 = aParams->fX;
	int	lY1 = aParams->fY;
	int	lX2 = lX1+aLabel->fWidth-1;

	Buffer_Print( lBuffer, lX1, lY1, lX2, lY1, aLabel->fText, COLOUR_LABEL_TEXT );
	Buffer_Print( lBuffer, lX1+aLabel->fHotKeyPosition, lY1, lX1+aLabel->fHotKeyPosition, lY1, aLabel->fText+aLabel->fHotKeyPosition, COLOUR_LABEL_HILITE );
}

bool	DlgLabel_LeftMouseClick( T_DlgLabel *aLabel )
{
	T_ObjectRec	*lRec = aLabel->fRelatedObject->fOwner;

	Group_FocusObject( lRec->fOwner, lRec->fObject.fData );

	return true;
}

bool	DlgLabel_HandleKeypress( T_DlgLabel *aLabel, int aKeypress )
{
	int	lHotKey;

	if ( aKeypress & 0x100 )
	{
		lHotKey = toupper( kScanToASCII[aKeypress-0x100] );
	}
	else
	{
		lHotKey = toupper( aKeypress );
	}
	if ( lHotKey == aLabel->fHotKey )
	{
   	T_ObjectRec	*lRec = aLabel->fRelatedObject->fOwner;
      Group_FocusObject( lRec->fOwner, lRec->fObject.fData );
		return true;
	}
	return false;
}

void	DlgLabel_Kill( T_DlgLabel *aLabel )
{
	free( aLabel->fText );
	free( aLabel );
}

T_DlgRadioButtons	*DlgRadioButtons_Initialise( void )
{
	T_DlgRadioButtons	*lRadioButtons;

	P_InitVar( lRadioButtons, T_DlgRadioButtons );

	lRadioButtons->fSelectedNum = 0;
	lRadioButtons->fNumButtons = 0;
	lRadioButtons->fButtonList = NULL;

	DesktopObj_Initialise( lRadioButtons, DlgRadioButtons_EventHandler, false );

	return lRadioButtons;
}

inline	void	DlgRadioButtons_SetCursorPosition( T_DlgRadioButtons *aRadioButtons )
{
	T_ObjectRec	*lRec = aRadioButtons->fOwner;
	Screen_SetCursorPosition( lRec->fOwner->fX1+lRec->fX1+2, lRec->fOwner->fY1+lRec->fY1+aRadioButtons->fSelectedNum );
}

inline	void	DlgRadioButtons_Focus( T_DlgRadioButtons *aRadioButtons )
{
	if ( aRadioButtons->fOwner->fOwner->fVisible )
	{
		DlgRadioButtons_SetCursorPosition( aRadioButtons );
		Screen_SetCursorType( _NORMALCURSOR );
	}
}

bool	DlgRadioButtons_EventHandler( void *aRadioButtons, T_EventRec *aEvent )
{
	bool	lRetVal = false;

	T_DlgRadioButtons	*lRadioButtons = aRadioButtons;

	switch ( aEvent->fMessage )
	{
		case	EVENT_CANFOCUS:
			lRetVal = true;
			break;
		case	EVENT_SHOW:
			if ( !lRadioButtons->fFocused )
			{
				break;
			}
		case	EVENT_FOCUS:
			DlgRadioButtons_Focus( aRadioButtons );
			lRetVal = true;
			break;
		case	EVENT_HIDE:
			if ( !lRadioButtons->fFocused )
			{
				break;
			}
		case	EVENT_UNFOCUS:
			Screen_SetCursorType( _NOCURSOR );
			lRetVal = true;
			break;
		case	EVENT_LEFTMOUSECLICK:
			lRetVal = DlgRadioButtons_LeftMouseClick( aRadioButtons, aEvent->fParameters.fMouseClick.fRow );
			break;
		case	EVENT_KEYPRESS:
			lRetVal = DlgRadioButtons_HandleKeypress( aRadioButtons, aEvent->fParameters.fKeypress );
			break;
		case	EVENT_KILL:
			DlgRadioButtons_Kill( aRadioButtons );
			lRetVal = true;
			break;
		case	EVENT_DRAW:
			DlgRadioButtons_Draw( aRadioButtons, &aEvent->fParameters.fDrawParams );
			lRetVal = true;
			break;
		default:
			FatalError( "Unhandled message" );
			break;
	}
	return lRetVal;
}

void	DlgRadioButtons_AddButton( T_DlgRadioButtons *aRadioButtons, const char *aName, int aID )
{
	T_StringList	*lCurrent = aRadioButtons->fButtonList;

	if ( lCurrent == NULL )
	{
		P_InitVar( aRadioButtons->fButtonList, T_StringList );
		lCurrent = aRadioButtons->fButtonList;
	}
	else
	{
		while ( lCurrent->fNext != NULL )
		{
			lCurrent = lCurrent->fNext;
		}
		P_InitVar( lCurrent->fNext, T_StringList );
		lCurrent = lCurrent->fNext;
	}
	P_CreateStr( lCurrent->fText, aName );
	lCurrent->fID = aID;
	lCurrent->fNext = NULL;
	aRadioButtons->fNumButtons++;
}

void	DlgRadioButtons_Draw( T_DlgRadioButtons *aRadioButtons, T_EventDrawParameters *aParams )
{
	char	lStr[81], lOutputStr[81], lChar;
	int	lColour;

	T_Buffer	*lBuffer = aParams->fBuffer;

	int	lX1 = aParams->fX;
	int	lY1 = aParams->fY;
	int	lX2 = lX1+aRadioButtons->fWidth-1;
        int     lY2 = lY1+aRadioButtons->fHeight-1;
	int	lCurItem = 0;

	T_StringList	*lCurrent = aRadioButtons->fButtonList;

	while ( lCurrent != NULL )
	{
		strncpy( lOutputStr, lCurrent->fText, aRadioButtons->fWidth-6 );
		lOutputStr[aRadioButtons->fWidth-6] = EOS;
		lChar = ( aRadioButtons->fSelectedNum == lCurItem ) ? 7 : ' ';
	   sprintf( lStr, " (%c) %-*s ", lChar, aRadioButtons->fWidth-6, lOutputStr );
		lColour = COLOUR_RADBUT_TEXT;
		if ( aRadioButtons->fFocused && aRadioButtons->fSelectedNum == lCurItem )
		{
			lColour = COLOUR_RADBUT_FOCUSED;
		}
                Buffer_Print( lBuffer, lX1, lY1, lX2, lY2, lStr, lColour );

		lY1++;

		lCurrent = lCurrent->fNext;
		lCurItem++;
	}
}

bool	DlgRadioButtons_LeftMouseClick( T_DlgRadioButtons *aRadioButtons, int aRow )
{
	aRadioButtons->fSelectedNum = aRow;
	if ( !aRadioButtons->fFocused )
	{
		Group_FocusObject( aRadioButtons->fOwner->fOwner, aRadioButtons );
	}
	else
	{
   	Refresh_AddDirtyRectangle( aRadioButtons->fOwner );
	}
	return true;
}

bool	DlgRadioButtons_HandleKeypress( T_DlgRadioButtons *aRadioButtons, int aKeypress )
{
	bool	lHandled = false;

	if ( !aRadioButtons->fFocused )
	{
		return false;
	}

	if ( aKeypress == K_EUp && aRadioButtons->fSelectedNum > 0 )
	{
		aRadioButtons->fSelectedNum--;
		lHandled = true;
	}
	if ( aKeypress == K_EDown && aRadioButtons->fSelectedNum < aRadioButtons->fNumButtons-1 )
	{
		aRadioButtons->fSelectedNum++;
		lHandled = true;
	}
	if ( lHandled )
	{
		DlgRadioButtons_SetCursorPosition( aRadioButtons );
		Refresh_AddDirtyRectangle( aRadioButtons->fOwner );
	}
	return lHandled;
}

void	DlgRadioButtons_Kill( T_DlgRadioButtons *aRadioButtons )
{
	T_StringList	*lCurrent = aRadioButtons->fButtonList, *lNext;

	while ( lCurrent != NULL )
	{
		lNext = lCurrent->fNext;
		free( lCurrent->fText );
		free( lCurrent );
		lCurrent = lNext;
	}
	free( aRadioButtons );
}

T_DlgListBox	*DlgListBox_Initialise( T_ScrollBar *aHorizScroll, T_ScrollBar *aVertScroll )
{
	T_DlgListBox	*lListBox;

	P_InitVar( lListBox, T_DlgListBox );

	DesktopObj_Initialise( lListBox, DlgListBox_EventHandler, false );

	lListBox->fSelectedNum = 0;
	lListBox->fNumItems = 0;

	lListBox->fVertScroll = aVertScroll;

	if ( aVertScroll != NULL )
	{
		aVertScroll->fRelatedObject = ( T_MinimumDesktopObject* )lListBox;
	}

	lListBox->fHorizScroll = aHorizScroll;

	if ( aHorizScroll != NULL )
	{
		aHorizScroll->fRelatedObject = ( T_MinimumDesktopObject* )lListBox;
	}

	return lListBox;
}

bool	DlgListBox_EventHandler( void *aListBox, T_EventRec *aEvent )
{
	bool	lRetVal = false;

	switch ( aEvent->fMessage )
	{
		case	EVENT_LEFTMOUSECLICK:
			lRetVal = DlgListBox_LeftMouseClick( aListBox, aEvent->fParameters.fMouseClick.fRow );
			break;
		case	EVENT_KEYPRESS:
			lRetVal = DlgListBox_HandleKeypress( aListBox, aEvent->fParameters.fKeypress );
			break;
		case	EVENT_KILL:
			DlgListBox_Kill( aListBox );
			lRetVal = true;
			break;
		case	EVENT_DRAW:
			DlgListBox_Draw( aListBox, &aEvent->fParameters.fDrawParams );
			lRetVal = true;
			break;
		case	EVENT_SHOW:
		case	EVENT_HIDE:
			break;
		case	EVENT_FOCUS:
		case	EVENT_UNFOCUS:
		case	EVENT_CANFOCUS:
			lRetVal = true;
			break;
		default:
			FatalError( "Unhandled message" );
			break;
	}
	return lRetVal;
}

void	DlgListBox_AddItem( T_DlgListBox *aListBox, const char *aItem, int aID )
{
	T_StringList	*lCurrent = aListBox->fItemList;

	if ( lCurrent == NULL )
	{
		P_InitVar( aListBox->fItemList, T_StringList );
		lCurrent = aListBox->fItemList;
	}
	else
	{
		while ( lCurrent->fNext != NULL )
		{
			lCurrent = lCurrent->fNext;
		}
		P_InitVar( lCurrent->fNext, T_StringList );
		lCurrent = lCurrent->fNext;
	}
	P_CreateStr( lCurrent->fText, aItem );
	lCurrent->fID = aID;
	lCurrent->fNext = NULL;
	aListBox->fNumItems++;
}

void	DlgListBox_Draw( T_DlgListBox *aListBox, T_EventDrawParameters *aParams )
{
	T_Buffer	*lBuffer = aParams->fBuffer;

	int	lX1 = aParams->fX;
	int	lY1 = aParams->fY;
	int	lX2 = lX1+aListBox->fWidth;
	int	i, lColour;
	char	lOutputStr[81], *lPtr, lStr[81];

	T_StringList	*lString = aListBox->fItemList;

	i = aListBox->fYStart;

	while ( i )
	{
		i--;
		lString = lString->fNext;
	}

	for ( i = 0; i < aListBox->fHeight; i++ )
	{
		if ( lString != NULL )
		{
			strncpy( lOutputStr,	lString->fText, aListBox->fWidth-2 );
			lOutputStr[aListBox->fWidth-2] = EOS;
		}
		else
		{
			lOutputStr[0] = EOS;
		}
		if ( aListBox->fXStart < ( signed )strlen( lOutputStr ))
		{
			lPtr = lOutputStr+aListBox->fXStart;
		}
		else
		{
			lPtr = lOutputStr+strlen( lOutputStr );
		}
		sprintf( lStr, " %-*s ", aListBox->fWidth-2, lPtr );

		lColour = ( aListBox->fSelectedNum == i+aListBox->fYStart ) ? ( aListBox->fFocused ? COLOUR_LISTBOX_FOCUS : COLOUR_LISTBOX_SELECT ) : COLOUR_LISTBOX_TEXT;

		Buffer_Print( lBuffer, lX1, lY1+i, lX2, lY1+i, lStr, lColour );

		if ( lString != NULL )
		{
			lString = lString->fNext;
		}
	}
}

bool	DlgListBox_HandleKeypress( T_DlgListBox *aListBox, int aKeypress )
{
	bool	lHandled = false;

	if ( !aListBox->fFocused )
	{
		return false;
	}

	if ( aKeypress == K_EDown && aListBox->fSelectedNum < aListBox->fNumItems-1 )
	{
		aListBox->fSelectedNum++;
		lHandled = true;
	}
	if ( aKeypress == K_EUp && aListBox->fSelectedNum > 0 )
	{
		aListBox->fSelectedNum--;
		lHandled = true;
	}
	if ( aKeypress == K_EPageUp )
	{
		aListBox->fSelectedNum -= aListBox->fHeight;
		lHandled = true;
	}
	if ( aListBox->fSelectedNum < 0 )
	{
		aListBox->fSelectedNum = 0;
	}
	if ( aKeypress == K_EPageDown )
	{
		aListBox->fSelectedNum += aListBox->fHeight;
		lHandled = true;
	}
	if ( aListBox->fSelectedNum >= aListBox->fNumItems )
	{
		aListBox->fSelectedNum = aListBox->fNumItems-1;
	}
	if ( aKeypress == K_EHome )
	{
		aListBox->fSelectedNum = 0;
		lHandled = true;
	}
	if ( aKeypress == K_EEnd )
	{
		aListBox->fSelectedNum = aListBox->fNumItems-1;
		lHandled = true;
	}
	while ( aListBox->fSelectedNum-aListBox->fYStart >= aListBox->fHeight )
	{
		aListBox->fYStart++;
	}
	if ( aListBox->fSelectedNum < aListBox->fYStart )
	{
		aListBox->fYStart = aListBox->fSelectedNum;
	}
	if ( lHandled )
	{
		/// update scroll bars ///
		Refresh_AddDirtyRectangle( aListBox->fOwner );
	}
	return lHandled;
}

bool	DlgListBox_LeftMouseClick( T_DlgListBox *aListBox, int aRow )
{
	aListBox->fSelectedNum = aRow+aListBox->fYStart;

	if ( aListBox->fSelectedNum >= aListBox->fNumItems )
	{
		aListBox->fSelectedNum = aListBox->fNumItems-1;
	}

	if ( !aListBox->fFocused )
	{
		Group_FocusObject( aListBox->fOwner->fOwner, aListBox );
	}
	else
	{
   	Refresh_AddDirtyRectangle( aListBox->fOwner );
	}
	return true;
}

void	DlgListBox_Kill( T_DlgListBox *aListBox )
{
	T_StringList	*lCurrent = aListBox->fItemList, *lNext;

	while ( lCurrent != NULL )
	{
		lNext = lCurrent->fNext;
		free( lCurrent->fText );
		free( lCurrent );
		lCurrent = lNext;
	}
	free( aListBox );
}

bool	DesktopObj_HandleLeftClick( T_ObjectRec *aObject, int aColumn, int aRow )
{
	bool	lRetVal = false;

	T_EventRec	lEvent;

	if ( DesktopObj_Focus( aObject ))
	{
		lRetVal = true;
	}

	lEvent.fMessage = EVENT_LEFTMOUSECLICK;
	lEvent.fParameters.fMouseClick.fColumn = aColumn;
	lEvent.fParameters.fMouseClick.fRow = aRow;

	if ( aObject->fObject.fHeader->fEventHandler( aObject->fObject.fData, &lEvent ))
	{
		lRetVal = true;
	}
	return lRetVal;
}

bool	DesktopObj_HandleRightClick( T_ObjectRec *aObject, int aColumn, int aRow )
{
	bool	lRetVal = false;

	T_EventRec	lEvent;

	if ( DesktopObj_Focus( aObject ))
	{
		lRetVal = true;
	}

	lEvent.fMessage = EVENT_RIGHTMOUSECLICK;
	lEvent.fParameters.fMouseClick.fColumn = aColumn;
	lEvent.fParameters.fMouseClick.fRow = aRow;

	if ( aObject->fObject.fHeader->fEventHandler( aObject->fObject.fData, &lEvent ))
	{
		lRetVal = true;
	}
	return lRetVal;
}

void	DesktopObj_Draw( T_ObjectRec *aObject, int aX, int aY, T_Buffer *aBuffer )
{
	T_EventRec	lEvent;

	lEvent.fMessage = EVENT_DRAW;
	lEvent.fParameters.fDrawParams.fBuffer = aBuffer;
	lEvent.fParameters.fDrawParams.fX = aObject->fX1+aObject->fOwner->fX1-aX;
	lEvent.fParameters.fDrawParams.fY = aObject->fY1+aObject->fOwner->fY1-aY;

	if ( lEvent.fParameters.fDrawParams.fX+aObject->fObject.fHeader->fWidth > 0 &&
		  lEvent.fParameters.fDrawParams.fY+aObject->fObject.fHeader->fHeight > 0 &&
		  lEvent.fParameters.fDrawParams.fX < aBuffer->fWidth && lEvent.fParameters.fDrawParams.fY < aBuffer->fHeight )
	{
		aObject->fObject.fHeader->fEventHandler( aObject->fObject.fData, &lEvent );
	}
}

void	DesktopObj_Kill( T_ObjectRec *aObject )
{
	T_EventRec	lEvent;

	lEvent.fMessage = EVENT_KILL;

	aObject->fObject.fHeader->fEventHandler( aObject->fObject.fData, &lEvent );

	free( aObject );
}

bool	DesktopObj_HandleKeypress( T_ObjectRec *aObject, int aKeypress )
{
	T_EventRec	lEvent;

	lEvent.fMessage = EVENT_KEYPRESS;
	lEvent.fParameters.fKeypress = aKeypress;

	return aObject->fObject.fHeader->fEventHandler( aObject->fObject.fData, &lEvent );
}

bool	DesktopObj_Focus( T_ObjectRec *aObject )
{
	T_EventRec	lEvent;
	T_Group		*lGroup = aObject->fOwner;

	if ( DesktopObj_CanFocus( aObject ))
	{
		if ( lGroup->fFocused != aObject )
		{
			aObject->fObject.fHeader->fFocused = true;

			if ( lGroup->fFocused != NULL )
			{
				lEvent.fMessage = EVENT_UNFOCUS;

				lGroup->fFocused->fObject.fHeader->fFocused = false;

				if ( lGroup->fFocused->fObject.fHeader->fEventHandler( lGroup->fFocused->fObject.fData, &lEvent ))
				{
					Group_RedrawObject( lGroup, lGroup->fFocused->fObject.fData );
				}
			}
			lEvent.fMessage = EVENT_FOCUS;

			aObject->fObject.fHeader->fEventHandler( aObject->fObject.fData, &lEvent );

			Group_RedrawObject( lGroup, aObject->fObject.fData );
			lGroup->fFocused = aObject;
		}
      return true;
	}
   else
   {
   	return false;
   }
}

bool	DesktopObj_CanFocus( T_ObjectRec *aObject )
{
	T_EventRec	lEvent;

	lEvent.fMessage = EVENT_CANFOCUS;

	return aObject->fObject.fHeader->fEventHandler( aObject->fObject.fData, &lEvent );
}

void	DesktopObj_Initialise( void *aObject, bool ( *aEventHandler )( void*, T_EventRec* ), bool aCanActivate )
{
	T_MinimumDesktopObject	*lObject = aObject;

	lObject->fFocused = false;
   lObject->fCanActivate = aCanActivate;
	lObject->fEventHandler = aEventHandler;
}

T_StringList	*DlgRadioButtons_GetSelectedItem( T_DlgRadioButtons *aRadioButtons )
{
	T_StringList	*lRec = aRadioButtons->fButtonList;

	int	i = aRadioButtons->fSelectedNum;

	while ( i )
	{
		i--;
		lRec = lRec->fNext;
	}
	return lRec;
}

T_StringList	*DlgListBox_GetSelectedItem( T_DlgListBox *aListBox )
{
	T_StringList	*lRec = aListBox->fItemList;

	int	i = aListBox->fSelectedNum;

	while ( i )
	{
		i--;
		lRec = lRec->fNext;
	}
	return lRec;
}



