/* pane.cc
 */

#include "pane.h"
#include "mouse.h"
#include "keyboard.h"
#include <pc.h>
#include <iostream.h>
#include <conio.h>

#include <stdio.h>

screen_pane desktop( 0, 0, ScreenCols(), ScreenRows() );

const char * single_border_chars = "ڿٴ";
const char * double_border_chars = "ɻȼ";

/////////////////////////////////////////////////////////////////
// drawing functions
//
void draw_all_windows()
{
   desktop.refresh_image();
   desktop.draw_to_screen();
}


void screen_pane::
draw_children()
{
   int i = 0;
   while( i < _child_count )
   {
      if( _child[i] -> is_visible() )
      {
         _child[i] -> refresh_image();
         _child[i] -> draw_to( *this );
      }
      i++;
   }
}


void screen_pane::
refresh_image()
{
   if( _child_count > 0 )
   {
      fill();
      draw_children();
   }
   if( _parent )
   {
      draw_shadow();
      draw_border();
   }
}




/////////////////////////////////////////////////////////////////
// info functions
//
void screen_pane::show()         { _visible = true;  }
void screen_pane::hide()         { _visible = false; }

bool screen_pane::is_visible()   { return _visible;  }
bool screen_pane::is_invisible() { return !_visible; }

screen_pane & screen_pane::parent() { return *_parent; }


/////////////////////////////////////////////////////////////////
// moving and scrolling
//

void screen_pane::move_left( int cols ) {
   move( -cols, 0 );
}
void screen_pane::move_right( int cols ) {
   move( cols, 0 );
}
void screen_pane::move_up( int rows ) {
   move( 0, -rows );
}
void screen_pane::move_down( int rows ) {
   move( 0, rows );
}

void screen_pane::
move_to( int x, int y )
{
   int amount_right = x - left();
   int amount_down = y - top();
   move( amount_right, amount_down );
}

void screen_pane::
move( int amount_right, int amount_down )
{
   screen_sheet::move_to( left() + amount_right,
                          top()  + amount_down );
   int i = 0;
   while( i < _child_count )
   {
      _child[i] -> move( amount_right, amount_down );
      i++;
   }
}

void screen_pane::scroll_left( int cols )  {
   scroll(  cols, 0 );
   }
void screen_pane::scroll_right( int cols ) {
   scroll( -cols, 0 );
   }
void screen_pane::scroll_up( int rows )    {
   scroll( 0,  rows );
   }
void screen_pane::scroll_down( int rows )  {
   scroll( 0, -rows );
   }

void screen_pane::
scroll( int amount_left, int amount_up )
{
   int i = 0;
   while( i < _child_count )
   {
      _child[i] -> move_right( amount_left );
      _child[i] -> move_down( amount_up );
      i++;
   }
}

void screen_pane::
mouse_move_while( int condition )
{
   mouse.poll();
   if( ( mouse.clicked() & condition ) && mouse.is_over( *this ) )
   {
      mouse.hide_cursor();
      int offset_x = mouse.x() - left();
      int offset_y = mouse.y() - top();

      while( mouse.clicked() & condition )
      {
         move_to( mouse.x() - offset_x,
                  mouse.y() - offset_y );

         draw_all_windows();
         mouse.poll();
      }
      mouse.show_cursor();
   }
}

void screen_pane::
mouse_scroll_while( int condition, int direction )
{
   mouse.poll();
   if( !( mouse.clicked() & condition ) )
      return;

   int old_mouse_x = mouse.x();
   int old_mouse_y = mouse.y();

   mouse.hide_cursor();

   while( mouse.clicked() & condition )
   {
      if( mouse.is_over( *this ) && mouse.has_moved() )
      {
         if( direction == both_ways )
            scroll( mouse.x() - old_mouse_x, mouse.y() - old_mouse_y );
         else
         if( direction == vertically )
            scroll( 0, mouse.y() - old_mouse_y );
         else
         if( direction == horizontally )
            scroll( mouse.x() - old_mouse_x, 0 );

         draw_all_windows();
         old_mouse_x = mouse.x();
         old_mouse_y = mouse.y();
      }
      mouse.poll();
   }
   mouse.show_cursor();
}



/////////////////////////////////////////////////////////////////
// child functions
//
void screen_pane::
to_top()
{
   if( _parent )
   {
      _parent -> child_to_top( *this );
      _parent -> to_top();
   }
}

// moves child to top of heap, so it will be drawn last
void screen_pane::
child_to_top( screen_pane & new_top_child )
{
// find the child in this window's child array
   bool found = false;
   int i = 0;
   while( i < _child_count )
   {
      if( _child[i] == &new_top_child )
      {
         found = true;
         break;
      }
      i++;
   }

// swap with the child at the end of the list (top position)
   if( found )
   {
      int current_position = i;
      int end_of_list = _child_count - 1;

      screen_pane * temp = _child[end_of_list];
      _child[end_of_list] = _child[current_position];
      _child[current_position] = temp;
   }
}

screen_pane * screen_pane::
top_child_under( int x, int y )
{
   if( this -> contains( x, y ) )
   {
      int i = _child_count - 1;
      while( i >= 0 )
      {
         if( _child[i] -> contains( x, y ) )
            return _child[i];
         i--;
      }
   }
   return this;
}

screen_pane * screen_pane::
top_child()
{
   if( _child_count )
      return _child[_child_count-1];
   else
      return this;
}

screen_pane * screen_pane::
child( int index )
{
   if( index < 0 || index >= _child_count )
      return this;
   else
      return _child[index];
}

void screen_pane::
adopt( screen_pane & new_child )
{
   if( &new_child == &desktop
    || &new_child == this ) return;

   if( new_child._parent )
      new_child._parent -> remove_child( new_child );

   add_child( new_child );

   new_child._parent = this;
}

void screen_pane::
add_child( screen_pane & new_child )
{
   if( &new_child == &desktop
    || &new_child == this ) return;

   if( _child_count < MAXIMUM_CHILDREN )
   {
      _child[_child_count] = &new_child;
      _child_count++;
   }
}

void screen_pane::
remove_child( screen_pane & unwanted_child )
{
   if( _child_count > 0 )
   {
      child_to_top( unwanted_child );
      if( _child[_child_count - 1] == &unwanted_child )
         _child_count--;
   }
}



/////////////////////////////////////////////////////////////////
// constructors, destructor and intialisers
//
screen_pane::
screen_pane()
{
   _visible = true;
   _shadow_on = false;
   _border_on = false;
   _shadow_colour = DEFAULT_SHADOW_COLOUR;
   _border_colour = DEFAULT_BORDER_COLOUR;
   _title_colour = DEFAULT_TITLE_COLOUR;
   set_border_chars();

   _child_count = 0;
   _parent = 0;
   desktop.adopt( *this );
}

screen_pane::
screen_pane( int tlx, int tly, int width, int height )
: screen_sheet( tlx, tly, width, height )
{
   _visible = true;
   _shadow_on = false;
   _border_on = false;
   _border_colour = DEFAULT_BORDER_COLOUR;
   _title_colour = DEFAULT_TITLE_COLOUR;

   _child_count = 0;
   _parent = 0;
   if( this != &desktop )
      desktop.adopt( *this );
   else
      set_fill_char( DEFAULT_CHARACTER, DEFAULT_COLOUR );

   set_border_chars();
   resize( width, height );
   move_to( tlx, tly );
}

screen_pane::
~screen_pane()
{
   if( _parent )
      _parent -> remove_child( *this );

   int i = 0;
   while( i < _child_count )
   {
      _child[i] -> _parent = 0;
      i++;
   }
}

void screen_pane::
set_shadow( bool setting )
{
   if( _parent )
      _shadow_on = setting;
}


void screen_pane::
draw_shadow()
{
   if( !_shadow_on || !this -> overlaps( parent() ) )
      return;

   int offset = 1;
   int margin = 1;
   if( _border_on )
   {
      offset = 0;
      margin = 2;
   }
   int upper_limit = bigger_of( ( top()  + offset ) - _parent -> top(),  0 );
   int left_limit  = bigger_of( ( left() + offset ) - _parent -> left(), 0 );
   int lower_limit = smaller_of( ( bottom() + margin ) - _parent -> top(), parent().height() - 1 );
   int right_limit = smaller_of( ( right()  + margin ) - _parent -> left(),  parent().width() - 1 );

   if( right()+margin <= parent().right() )
   {
      int y = upper_limit;
      while( y <= lower_limit )
      {
         parent()[y][right_limit].colour = _shadow_colour;
         y++;
      }
   }
   if( bottom()+margin <= parent().bottom() )
   {
      int x = left_limit;
      while( x <= right_limit )
      {
         parent()[lower_limit][x].colour = _shadow_colour;
         x++;
      }
   }
}

void screen_pane::
set_border( bool setting )
{
   if( _parent )
      _border_on = setting;
}

void screen_pane::
draw_border()
{
   if( !_border_on || !this -> overlaps( parent() ) )
      return;

   int upper_limit = bigger_of( ( top()  - 1 ) - _parent -> top(),   0 );
   int left_limit  = bigger_of( ( left() - 1 ) - _parent -> left(), 0 );
   int lower_limit = smaller_of( ( bottom() + 1 ) - _parent -> top(), parent().height() - 1 );
   int right_limit = smaller_of( ( right()  + 1 ) - _parent -> left(),  parent().width() - 1 );
   screen_char border_char = { _vertical_bar, _border_colour };

   if( right() < parent().right() )
   {
      border_char.character = _vertical_bar;
      int y = upper_limit;
      while( y <= lower_limit )
      {
         (*_parent)[y][right_limit] = border_char;
         y++;
      }
      if( bottom() < parent().bottom() )
      {
         border_char.character = _br_corner;
         parent()[lower_limit][right_limit] = border_char;
      }
   }
   if( left() > parent().left() )
   {
      border_char.character = _vertical_bar;
      int y = upper_limit;
      while( y <= lower_limit )
      {
         parent()[y][left_limit] = border_char;
         y++;
      }
      if( top() > parent().top() )
      {
         border_char.character = _tl_corner;
         parent()[upper_limit][left_limit] = border_char;
      }
   }
   if( top() > parent().top() )
   {
      border_char.character = _horizontal_bar;
      int x = left_limit + 1;
      while( x <= right_limit - 1 )
      {
         parent()[upper_limit][x] = border_char;
         x++;
      }
      if( right() < parent().right() )
      {
         border_char.character = _tr_corner;
         parent()[upper_limit][right_limit] = border_char;
      }
      if( _title.length() > 0 )
      {
         if( (int)_title.length() >= width() - 3 )
            _title = _title.before( (int)width() - 3 );
         else
            parent()[upper_limit]
                    [left_limit+2+_title.length()].character = ' ';

         parent().put_text( _title, left_limit+2,
                            upper_limit, _title_colour );
         parent()[upper_limit]
                 [left_limit+1].character = _left_bookend;
      }
   }
   if( bottom() < parent().bottom() )
   {
      border_char.character = _horizontal_bar;
      int x = left_limit + 1;
      while( x <= right_limit - 1 )
      {
         parent()[lower_limit][x] = border_char;
         x++;
      }
      if( left() > parent().left() )
      {
         border_char.character = _bl_corner;
         parent()[lower_limit][left_limit] = border_char;
      }
   }
}

void screen_pane::
set_border_chars( const char * border_chars )
{
   if( !border_chars || strlen(border_chars) != 8 )
      return;

   _vertical_bar   = border_chars[0];
   _horizontal_bar = border_chars[1];
   _tl_corner      = border_chars[2];
   _tr_corner      = border_chars[3];
   _bl_corner      = border_chars[4];
   _br_corner      = border_chars[5];
   _left_bookend   = border_chars[6];
   _right_bookend  = border_chars[7];
}

void screen_pane::
set_title( const char * title_text )
{
   if( title_text )
      _title = title_text;
}
void screen_pane::
set_title_colour( char new_colour )
{
   _title_colour = new_colour;
}

void screen_pane::
set_border_colour( char new_colour )
{
   _border_colour = new_colour;
}
void screen_pane::
set_shadow_colour( char new_colour )
{
   _shadow_colour = new_colour;
}


