/*
     _______                     ___                          ________
    /       \         /\        |   |\             /\        |        \
   /         >       /  \       |   ||            /  \       |         \
  /   ______/ >     /    \      |   ||           /    \      |    __    \
 <   <_______/     /      \     |   ||          /      \     |   |\_\    \
  \        \      /   /\   \    |   ||         /   /\   \    |   ||  \    \
   \        \    |   /_L\   |   |   ||        |   /_L\   |   |   ||   >   |\
    \_____   \   |          |\  |   ||        |          |\  |   ||  /    /|
   __L____>   >  |          ||  |   |L____    |          ||  |   |L_/    / /
  /          / > |   ____   ||  |         |\  |   ____   ||  |          / /
 <          / /  |   |\_|   ||  |         ||  |   |\_|   ||  |         / /
  \________/ /   |___|| |___||  |_________||  |___|| |___||  |________/ /
   \________/     \___\  \___\   \_________\   \___\  \___\   \_______\/


                an Addon Package for Allegro by Sven Sandberg


This file contains some graphics routines.

*/
#ifndef s_gfx_c
#define s_gfx_c

#include "s_gfx.h"

#include "s_math.h"
#include "s_common.h"
#include "s_timer.h"
#include "s_rect.h"
#include "s_text.h"
#include "s_fade.h"
#include "s_alloc.h"
#include "s_global.h"
//For drawing_mode in before_draw().
#include <src\internal.h>

#include <allegro.h>

RGB
	black=      {0,0,0},    white=      {63,63,63},
	greyxdark=  {8,8,8},    greydark=   {16,16,16}, grey=       {32,32,32},
									greylight=  {48,48,48}, greyxlight= {56,56,56},
	browndark=  {21,11,0},  brown=      {39,23,12}, brownlight= {54,37,25},
	violetdark= {20,0,16},  violet=     {40,0,32},  violetlight={63,0,63},
	bluedark=   {0,0,20},   blue=       {0,0,63},   bluelight=  {32,32,63},
	cyandark=   {0,32,32},  cyan=       {0,63,63},  cyanlight=  {36,63,63},
	greendark=  {0,14,0},   green=      {0,30,0},   greenlight= {0,63,0},
	yellowdark= {48,38,0},  yellow=     {63,63,0},  yellowlight={63,63,36},
	orangedark= {45,23,0},  orange=     {63,32,0},  orangelight={63,45,28},
	reddark=    {31,0,0},   red=        {63,0,0},   redlight=   {63,45,21};


//Contains the gfx modes that functions shouldn't try to set.
ulonglong skip_gfx_modes=0;



int cmprgb(RGB c1, RGB c2)
{
return _cmprgb(c1,c2);
}
int is_fast_bitmap(BITMAP *bmp)
{
return _is_fast_bitmap(bmp);
}



/***********************
****                ****
**** blit_rearrange ****
****                ****
************************
Blits a bitmap onto another, rearranging the palette at the same time. How
the colors are rearranged is specified in the map parameter, that should
point to 256 uchars. If a pixel on the source bitmap has the color 3, the
palette index `map[3]' is used instead.
*/
void blit_rearrange(BITMAP *so,BITMAP *de,int x,int y,uchar *map)
{
//Store old map.
COLOR_MAP *global_color_map=color_map;
//Create a faked color map.
struct{
	uchar *(data[1]);
}fake_color_map;
*fake_color_map.data=map;
color_map=(COLOR_MAP*)&fake_color_map;
draw_lit_sprite(de,so,x,y,0);
color_map=global_color_map;
}



/*************************
****                  ****
**** relative drawing ****
****                  ****
**************************
The same as the standard Allegro functions, but with w/h coordinates relative
to x and y coordinates instead of x1/y1/x2/y2-coordinates.
*/
void rel_vline(BITMAP *bmp, int x, int y, int h, int color)
{
vline(bmp,x,y,y+h,color);
}
void rel_hline(BITMAP *bmp, int x, int y, int w, int color)
{
hline(bmp,x,y,x+w,color);
}
void rel_do_line(BITMAP *bmp, int x, int y, int w, int h, int d, void (*proc)())
{
do_line(bmp,x,y,x+w,y+h,d,proc);
}
void rel_line(BITMAP *bmp, int x, int y, int w, int h, int color)
{
line(bmp,x,y,x+w,y+h,color);
}
void rel_triangle(BITMAP *bmp, int x1, int y1, int x2, int y2, int x3, int y3, int color)
{
triangle(bmp,x1,y1, x1+x2,y1+y2, x1+x2+x3,y1+y2+y3, color);
}
void rel_polygon(BITMAP *bmp, int vertices, int *points, int color)
{
int i;
//Sets the array to absolute coordinates.
for(i=1;i<vertices;i+=2){
	points[i*2]+=points[i*2-2];
	points[i*2+1]+=points[i*2-1];
}
polygon(bmp,vertices,points,color);
//Restores the array to relative coordinates.
for(i=vertices-1; i>0; i-=2){
	points[i*2]-=points[i*2-2];
	points[i*2+1]-=points[i*2-1];
}
}
void rel_rect(BITMAP *bmp, int x, int y, int w, int h, int color)
{
rect(bmp,x,y,x+w-1,y+h-1,color);
}
void rel_rectfill(BITMAP *bmp, int x, int y, int w, int h, int color)
{
rectfill(bmp,x,y,x+w-1,y+h-1,color);
}

/****************************
****                     ****
**** Easier ways to blit ****
****                     ****
****************************/
void easy_blit(BITMAP *so,BITMAP *de)
{
blit(so,de,0,0,0,0,so->w,so->h);
}
void topcenter_blit(BITMAP *so,BITMAP *de)
{
blit(so,de,0,0,(de->w - so->w) / 2,0,so->w,so->h);
}
void center_blit(BITMAP *so,BITMAP *de)
{
blit(so,de,0,0,(de->w - so->w) / 2,(de->h - so->h) / 2,so->w,so->h);
}
void position_blit(BITMAP *so,BITMAP *de, int x, int y)
{
blit(so,de,0,0,x,y,so->w,so->h);
}

/****************************
****                     ****
**** drawing with TRECTs ****
****                     ****
*****************************
Instead of passing all those x1,x2,y1,y2 to the drawing functions, you can
with these functions pass only a TRECT struct.
*/
void trect_rectfill(BITMAP *bmp,TRECT r,int color)
{
rectfill(bmp,r.x,r.y,r.x+r.w-1,r.y+r.h-1,color);
}
void trect_rect(BITMAP *bmp,TRECT r,int color)
{
rect(bmp,r.x,r.y,r.x+r.w-1,r.y+r.h-1,color);
}
void trect_line(BITMAP *bmp,TRECT r,int color)
{
line(bmp,r.x,r.y,r.x+r.w-1,r.y+r.h-1,color);
}
void trect_do_line(BITMAP *bmp,TRECT r,int color,void (*proc)())
{
do_line(bmp,r.x,r.y,r.x+r.w-1,r.y+r.h-1,color,proc);
}








/**********************
****               ****
**** find_gfx_mode ****
****               ****
***********************
Finds the best screen mode (highest resolution) with pixels that are square
formed.
*/
int find_gfx_mode(void)
{
if((skip_gfx_modes&skip1600x1200)||(gfxmode(1600,1200)<0))
	if((skip_gfx_modes&skip1024x768)||(gfxmode(1024,768)<0))
		if((skip_gfx_modes&skip800x600)||(gfxmode(800,600)<0))
			if((skip_gfx_modes&skip640x480)||(gfxmode(640,480)<0))
				if((skip_gfx_modes&skip320x240)||(gfxmode(320,240)<0))
					return -1;
return 0;
}
/***************************
****                    ****
**** find_gfx_mode_size ****
****                    ****
****************************
Finds the smallest graphics mode that is bigger than or equal to w,h, or, if
it couldn't find any such mode, the biggest possible for the screen. Tests
only square-modes.
*/
int find_gfx_mode_size(int w, int h)
{
int success=-1;
int neww=0;
if((!(skip_gfx_modes&skip320x240))&&(w<=320)&&(h<=240))
	success=gfxmode(neww=320,240);
else if((!(skip_gfx_modes&skip640x480))&&(w<=640)&&(h<=480))
	success=gfxmode(neww=640,480);
else if((!(skip_gfx_modes&skip800x600))&&(w<=800)&&(h<=600))
	success=gfxmode(neww=800,600);
else if((!(skip_gfx_modes&skip1024x768))&&(w<=1024)&&(h<=768))
	success=gfxmode(neww=1024,768);
else if(!(skip_gfx_modes&skip1600x1200))
	success=gfxmode(neww=1600,1200);
if(success){
	switch(neww){
		case 1600:
			if(success)
				success=gfxmode(1024,768);
		case 1024:
			if(success)
				success=gfxmode(800,600);
		case 800:
			if(success)
				success=gfxmode(640,480);
		case 640:
			if(success)
				success=gfxmode(320,240);
		case 320:
			if(success)
				return -2;
		break;
	}
	return -1;
}
return 0;
}

/*******************
****            ****
**** bitmap_dup ****
****            ****
********************
Returns a newly allocated copy of a bitmap.
*/
BITMAP *bitmap_dup(BITMAP *bmp)
{
BITMAP *ret=create_bitmap(bmp->w,bmp->h);
_easyblit(bmp,ret);
return ret;
}



/*********************
****              ****
**** bmp_bw_color ****
****              ****
**********************
Converts a bitmap and its palette to a single-color picture using palette
indexes startindex..endindex, where startindex is black and endindex is white.
The palcolor defines the hue of the colors. If you for example pass a green
color as palcolor, the palette will fade from black, via green, to white.
All pixels will preserve their luminance. The bitmap will be returned.*/
BITMAP *bmp_bw_color(BITMAP *bmp,PALETTE pal,RGB palcolor,
 uchar startindex,uchar numcolors)
{
int x,y;
int i;
int denominator,roundterm;
uchar endindex=startindex+numcolors-1;
uchar *pos,*end;
//If only one color is used, all becomes gray.
if(numcolors<=1){
	clear_to_color(bmp,startindex);
	pal[startindex]=v2rgb(31,31,31);
	return bmp;
}
if(endindex>255){
	endindex=255;
	numcolors=endindex-startindex+1;
}
denominator=max_luminance/(numcolors-1);
roundterm=denominator/2;
//We can optimize it a lot if bitmap is a memory bitmap...
if(is_memory_bitmap(bmp)){
	pos=&bmp->line[0][0];
	end=pos+bmp->w*bmp->h;
	pos--;//To compensate for the ++ on next line the first loop.
	while((++pos)<end)
		*pos=(roundterm+luminancergb(pal[*pos]))/denominator+startindex;
//...but if is is only linear, we must make it a bit slower...
}else if(is_linear_bitmap(bmp)){
	for(x=0;x<bmp->w;x++)
		for(y=0;y<bmp->h;y++)
			_putpixel(bmp,x,y,(roundterm+luminancergb(pal[_getpixel(bmp,x,y)]))/denominator+startindex);
//...and if it isn't even linear, this will take years.
}else{
	for(x=0;x<bmp->w;x++)
		for(y=0;y<bmp->h;y++)
			putpixel(bmp,x,y,(roundterm+luminancergb(pal[getpixel(bmp,x,y)]))/denominator+startindex);
}
//Filling palette with colors. Could be optimized with Bressenham's line
//algorithm, but this is hardly the most time-critical part of the function.
for(i=startindex;i<=endindex;i++)
	pal[i]=luminate(palcolor,rounddiv((i-startindex)*max_luminance,numcolors));
return bmp;
}










/**********************
****               ****
**** rect_extended ****
****               ****
***********************
Draws an outline rectangle with custom thickness and individual colors on
the sides.
*/
void rect_extended(BITMAP *bmp,TRECT r,int thickness,
 int topc,int bottomc,int leftc,int rightc)
{
int i;
for(i=0;i<thickness;i++){
	hline(bmp,r.x,       r.y,        r.x+r.w-1,  topc);
	hline(bmp,r.x+1,     r.y+r.h-1,  r.x+r.w-2,  bottomc);
	vline(bmp,r.x,       r.y+1,      r.y+r.h-1,  leftc);
	vline(bmp,r.x+r.w-1, r.y+1,      r.y+r.h-1,  rightc);
	r.x++;
	r.w-=2;
	r.y++;
	r.h-=2;
}
}
void fillrect_extended(BITMAP *bmp,TRECT r,int thickness,
 int fillc,int topc,int bottomc,int leftc,int rightc)
{
rect_extended(bmp,r,thickness,topc,bottomc,leftc,rightc);
trect_rectfill(bmp,resizerect(r,-thickness),fillc);
}

/************************
****                 ****
**** bevel functions ****
****                 ****
*************************
A bevel is a rectangle that looks like a button in windows, i.e. it is 3d-
lighted.
*/
inline void bevel(BITMAP *bmp,TRECT r,int thickness,
 int light,int shadow)
{
rect_extended(bmp,r,thickness,light,shadow,light,shadow);
}
inline void fillbevel(BITMAP *bmp,TRECT r,int thickness,
 int fillc,int light,int shadow)
{
fillrect_extended(bmp,r,thickness,fillc,light,shadow,light,shadow);
}

void doublebevel(BITMAP *bmp,TRECT r,int thickness,int light,int shadow)
{
bevel(bmp,r,thickness,light,shadow);
bevel(bmp,resizerect(r,-thickness),thickness,light,shadow);
}


/*************************
****                  ****
**** button functions ****
****                  ****
**************************
Here are some functions for drawing buttons that look like those in windows
95, but a little better.
*/
//Draws the frame of a button, without filling it.
void draw_empty_button_frame(BITMAP *bmp,TRECT r,
 int face,int xlight,int light,int dark,int xdark)
{
hline(bmp,r.x+1,  r.y,           (r.w+=r.x)-2,  xlight);//w means now right.
hline(bmp,r.x+2,  r.y+1,         r.w-3,         light);
hline(bmp,r.x+2,  (r.h+=r.y)-2,  r.w-3,         dark);//h means now bottom.
hline(bmp,r.x+1,  r.h-1,         r.w-2,         xdark);
vline(bmp,r.x,    r.y,           r.h-2,         xlight);
vline(bmp,r.x+1,  r.y+1,         r.h-3,         light);
vline(bmp,r.w-2,  r.y+2,         r.h-2,         dark);
vline(bmp,r.w-1,  r.y+1,         r.h-1,         xdark);
putpixel(bmp,r.w-1,  r.y,     face);
putpixel(bmp,r.w-2,  r.y+1,   face);
putpixel(bmp,r.x,    r.h-1,   face);
putpixel(bmp,r.x+1,  r.h-2,   face);
}
//Draws a button and fills it.
void draw_empty_button(BITMAP *bmp,TRECT r,
 int face,int xlight,int light,int dark,int xdark)
{
draw_empty_button_frame(bmp,r,face,xlight,light,dark,xdark);
if((r.w>4)&&(r.h>4))
	rectfill(bmp, r.x+2, r.y+2, r.x+r.w-3, r.y+r.h-3, face);
}

//Draws a filled button with text on.
void draw_text_button(BITMAP *bmp,TRECT r, int face, int textcolor,
 int xlight,int light, int dark,int xdark,FONT *f, uchar *text)
{
TRECT clip=get_clip_asrect(bmp);

draw_empty_button(bmp,r,face,xlight,light,dark,xdark);
set_clip_fromrect(bmp,fitinboth(clip,resizerect(r,-3)));
text_mode(-1);
textout_au_centre(bmp,f,text,r.x+r.w/2,r.y+(r.h-text_height(f))/2,
 textcolor);

set_clip_fromrect(bmp,clip);
}

//Draws a filled button with text on that is pressed down.
void draw_text_button_down(BITMAP *bmp,TRECT r, int face, int textcolor,
 int xlight,int light, int dark,int xdark,FONT *f, uchar *text)
{
TRECT clip=get_clip_asrect(bmp);

draw_empty_button_down(bmp,r,face,xlight,light,dark,xdark);
set_clip_fromrect(bmp,fitinboth(clip,resizerect(r,-3)));
text_mode(-1);
textout_au_centre(bmp,f,text,r.x+r.w/2+2,
 r.y+(r.h-text_height(f))/2+2,
 textcolor);

set_clip_fromrect(bmp,clip);
}

//Draws a filled button with a picture on.
void draw_picture_button(BITMAP *bmp,TRECT r, int face,
 int xlight,int light, int dark,int xdark,BITMAP *sprite)
{
TRECT clip=get_clip_asrect(bmp);

draw_empty_button(bmp,r,face,xlight,light,dark,xdark);
set_clip_fromrect(bmp,fitinboth(clip,resizerect(r,-3)));
draw_sprite(bmp,sprite,r.x+r.w/2-sprite->w/2,r.y+r.h/2-sprite->h/2);

set_clip_fromrect(bmp,clip);
}
//Draws a filled button with a picture on, pressed down.
void draw_picture_button_down(BITMAP *bmp,TRECT r, int face,
 int xlight,int light, int dark,int xdark,BITMAP *sprite)
{
TRECT clip=get_clip_asrect(bmp);

draw_empty_button_down(bmp,r,face,xlight,light,dark,xdark);
set_clip_fromrect(bmp,fitinboth(clip,resizerect(r,-3)));
draw_sprite(bmp,sprite,r.x+r.w/2-sprite->w/2+2,r.y+r.h/2-sprite->h/2+2);

set_clip_fromrect(bmp,clip);
}


/***********************************
****                            ****
**** before_draw and after_draw ****
****                            ****
************************************
These functions automatically handle the showing and hiding of the mouse
pointer if you tell them where you want to draw. They automatically detect
if they need to hide the mouse pointer, if they just need to freeze it (which
is if the mouse pointer doesn't intersect with the area you want to draw to)
or if they don't need to hide it at all (which is if you want to draw to
another bitmap). Call first before_draw (which will perhaps hide the mouse
pointer), then do the drawing, and then call after_draw, which will restore
the mouse as it was before. Note that all before_draw-s _must_ have a
corresponding after_draw. You can do nested before_draws and after_draws.
(This is especially interesting when you make general graphics routines,
since it allows you to call before_draw independently of wether before_draw
already is called.) If you want to draw to different areas, you can call
change_drawing_area, which will detect if it need to change the state of the
mouse when changing to the new area (if you called after_draw and then
before_draw, we would risk the mouse to hide and then show, which could cause
flickering if you did it too many times).
The functions also store/restore the clipping of the bitmap and the text_mode
(background color of text), so it will be safe to change those even when you
don't know what the calling function wants.
*/


static struct{
	BITMAP *bmp;
	TRECT clip_before;
	int text_mode_before;
	/* We could also store drawing_mode, rgb_table, color_table, font,
		gui_font_baseline or gui_[f|m|b]g_color. The only one of those that
		ought to be interesting to reset should be drawing_mode, but the space
		required for that (16 bytes, since we need a bitmap and x_anchor and
		y_anchor) is out of all proportion to the usefulness of it. I need the
		text_mode to be restored in some of my functions, though, so I've added
		that. Functions that want both to draw and to change clipping/text_mode
		permanently will have to do the change after their `after_draw()'.
	*/
} *safe_draw_stack;

static BITMAP *safe_draw_mouse_screen;
static int mouse_hidden_stack_index=-1, mouse_frozen_stack_index=-1;
TRECT mouse_frozen_rect;
static int safe_draw_stack_size=0,malloced_safe_draw_stack_size=0;

//Allegro's `int mouse_<x|y>_focus' and `int mwidth,mheight' variables are
//static, so I can't read them.
TRECT get_mouse_rect_asrect(void)
{
return v2rect(mouse_x-32,mouse_y-32,64,64);
}
/*void get_mouse_rect(int *x,int *y,int *w,int *h)
{
*x=x-mouse_x_focus;
*y=mouse_y-mouse_y_focus;
*w=mwidth;
*h=mheight;
}
*/

static void free_draw_stack(void)
{
free(safe_draw_stack);
}

void before_draw(BITMAP *bmp,TRECT area)
{
if(_mouse_screen&&is_same_bitmap(bmp,_mouse_screen)){
	//Always freeze mouse.
	if(mouse_frozen_stack_index<0){
		mouse_frozen_stack_index=safe_draw_stack_size;
		freeze_mouse_flag=TRUE;
		mouse_frozen_rect=get_mouse_rect_asrect();
	}
	//Hide mouse only if it intersects.
	if(mouse_hidden_stack_index<0){
		/* If bmp!=_mouse_screen, that means at least one of them is a
			sub-bitmap, and that the other is either the parent bitmap of the
			first or a sub-bitmap of the first or another sub-bitmap to the same
			parent as the first or some-thing else. In either case it will be
			impossible to transfer the coordinates from one of the bitmaps to
			the other since we can't detect the parents or children of a bitmap.
		*/
		if((bmp!=_mouse_screen)||intersect(area,mouse_frozen_rect)){
			mouse_hidden_stack_index=safe_draw_stack_size;
			safe_draw_mouse_screen=_mouse_screen;
			show_mouse(NULL);
		}
	}
}
//Resize stack if it's full.
if(safe_draw_stack_size>=malloced_safe_draw_stack_size){
	//If stack has never been initialized, an atexit function must be installed
	//that frees the stack.
	if(safe_draw_stack_size==0){
		malloced_safe_draw_stack_size=16;
		safe_draw_stack=xxmalloc(
		 malloced_safe_draw_stack_size*sizeof(typeof(safe_draw_stack)));
		atexit(free_draw_stack);
	}else{
		malloced_safe_draw_stack_size+=16;
		safe_draw_stack=xxrealloc(safe_draw_stack,
		 malloced_safe_draw_stack_size*sizeof(typeof(safe_draw_stack)));
	}
}
safe_draw_stack[safe_draw_stack_size].bmp=bmp;
safe_draw_stack[safe_draw_stack_size].clip_before=get_clip_asrect(bmp);
safe_draw_stack[safe_draw_stack_size].text_mode_before=get_text_mode();
safe_draw_stack_size++;
}

void after_draw(void)
{
if(safe_draw_stack_size<=0)
	exitfail("Call to `after_draw()' without any corresponding `before_draw()'.");
safe_draw_stack_size--;
if(mouse_hidden_stack_index==safe_draw_stack_size){
	show_mouse(safe_draw_mouse_screen);
	mouse_hidden_stack_index=-1;
}
if(mouse_frozen_stack_index==safe_draw_stack_size){
	freeze_mouse_flag=FALSE;
	mouse_frozen_stack_index=-1;
}
set_clip_fromrect(safe_draw_stack[safe_draw_stack_size].bmp,
 safe_draw_stack[safe_draw_stack_size].clip_before);
text_mode(safe_draw_stack[safe_draw_stack_size].text_mode_before);
}




/********************
****             ****
**** draw_sphere ****
****             ****
*********************
The actual function for draw_sphere is in "s_sphere.c". This is because I
have implemented a kind of #define/#include trick that makes it possible to
have two versions of the function (one for linear bitmaps and one for screen
bitmaps) but only one code.
*/
#ifndef s_sphere_c

//Define the macros that the file needs.
#define draw_sphere_function_name            draw_sphere_linear
#define draw_sphere_putpixel(linear,x,y,c)   linear=c
#define draw_sphere_is_linear
//Include the file.
#include "s_sphere.c"
//Undefine the macros.
#undef draw_sphere_function_name
#undef draw_sphere_putpixel
#undef draw_sphere_is_linear
#undef s_sphere_c
//Define the macros again, but for screen.
#define draw_sphere_function_name            draw_sphere_screen
#define draw_sphere_putpixel(linear,x,y,c)   putpixel(bmp,sphere_x+x,sphere_y+y,c)
//Include second time.
#include "s_sphere.c"
//Undefine the macros.
#undef draw_sphere_function_name
#undef draw_sphere_putpixel

#endif//ifdef s_sphere_c

void draw_sphere(BITMAP *bmp,
 int sphere_x,int sphere_y,int sphere_radius,
 int light_x,int light_y,int light_z,
 int first_color_index,int last_color_index,int darkness_color)
{
if(is_fast_bitmap(bmp))
	draw_sphere_linear(bmp,sphere_x,sphere_y,sphere_radius,
	 light_x,light_y,light_z,
	 first_color_index, last_color_index, darkness_color);
else
	draw_sphere_screen(bmp,sphere_x,sphere_y,sphere_radius,
	 light_x,light_y,light_z,
	 first_color_index, last_color_index, darkness_color);
}




#endif
