/*
	Copyright (C) 2001 Achilleas Margaritis

	MX-Windows 2.0 window system for DJGPP + Allegro

	This library is free software; you can redistribute it and/or
	modify it under the terms of the GNU Library General Public
	License as published by the Free Software Foundation; either
	version 2 of the License, or (at your option) any later version.

	This library is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
	Library General Public License for more details.

	You should have received a copy of the GNU Library General Public
	License along with this library; if not, write to the Free
	Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

	Contact : axilmar@b-online.gr
*/


#ifndef MXWIN_H
#define MXWIN_H


#include "allegro.h"


#ifdef __cplusplus
   extern "C" {
#endif


/*************************************************************************
	CONSTANTS
 *************************************************************************/


/* events */
enum EVENTS {
	WE_NONE = 0,
	WE_LBUTTONDOWN,
    WE_RBUTTONDOWN,
    WE_MBUTTONDOWN,
	WE_LBUTTONUP,
    WE_RBUTTONUP,
    WE_MBUTTONUP,
    WE_MOUSEENTER,
    WE_MOUSEMOVE,
    WE_MOUSELEAVE,
    WE_KEYDOWN,
    WE_KEYUP,
    WE_TIMER,
    WE_PAINT,
	WE_USER
};


/* window flags */
#define WF_VISIBLE           0x0001
#define WF_ENABLED           0x0002
#define WF_CLIPCHILDREN      0x0004
#define WF_CLIPSIBLINGS      0x0008
#define WF_CLIPPED           0x0010
#define WF_USER              0x1000
#define WF_MISC              (WF_VISIBLE |\
                              WF_ENABLED |\
                              WF_CLIPCHILDREN |\
                              WF_CLIPSIBLINGS |\
                              WF_CLIPPED)


/*************************************************************************
	MACROS
 *************************************************************************/


/* sets the given rectangle to the given coordinates */
#define SET_RECT(R, LEFT, TOP, RIGHT, BOTTOM) {\
	(R).left   = LEFT  ;\
    (R).top    = TOP   ;\
    (R).right  = RIGHT ;\
    (R).bottom = BOTTOM;\
}


/* checks if two rectangles overlap */
#define RECTS_OVERLAP(A, B)\
	(!(((B).left   > (A).right ) ||\
	   ((B).top    > (A).bottom) ||\
       ((B).right  < (A).left  ) ||\
       ((B).bottom < (A).top   )))


/* calculates the intersection of two rectangles */
#define INTERSECT_RECTS(A, B, R) {\
	(R).left   = MAX((A).left,   (B).left  );\
    (R).top    = MAX((A).top,    (B).top   );\
    (R).right  = MIN((A).right,  (B).right );\
    (R).bottom = MIN((A).bottom, (B).bottom);\
}


/* calculates the union of two rectangles */
#define UNITE_RECTS(A, B, R) {\
	(R).left   = MIN((A).left,   (B).left  );\
    (R).top    = MIN((A).top,    (B).top   );\
    (R).right  = MAX((A).right,  (B).right );\
    (R).bottom = MAX((A).bottom, (B).bottom);\
}


/* checks if a point is within a rectangle */
#define POINT_IN_RECT(R, X, Y)\
	((X >= (R).left) && (X <= (R).right ) &&\
	 (Y >= (R).top ) && (Y <= (R).bottom))


/* begin paint macro */
#define BEGIN_PAINT(WND, RCT, OVER_CHILDREN)\
	while (_do_paint_loop(WND, (RECT *)RCT, OVER_CHILDREN)) {


/* end paint macro */
#define END_PAINT()          }


/*************************************************************************
	TYPES
 *************************************************************************/


/* numeric types */
typedef unsigned short USHORT;


/* boolean */
typedef int BOOL;


/* window */
typedef struct { int unused; } *WINDOW;


/* window procedure */
typedef int WINDOW_PROC(WINDOW wnd, void *wnd_data, int msg, void *msg_data);


/* cursor */
typedef struct { int unused; } *CURSOR;


/*************************************************************************
	STRUCTURES
 *************************************************************************/


/* rectangle */
typedef struct RECT {
	short left;
    short top;
    short right;
    short bottom;
} RECT;


/* mouse event */
typedef struct MOUSE_EVENT {
	int type;
    WINDOW wnd;
    WINDOW dst;
    short x;
    short y;
    short button;
    short clicks;
    short shifts;
} MOUSE_EVENT;


/* keyboard event */
typedef struct KEY_EVENT {
	int type;
    WINDOW wnd;
    WINDOW dst;
    short key;
    short ascii;
    short scanc;
    short shifts;
} KEY_EVENT;


/* timer event */
typedef struct TIMER_EVENT {
	int type;
    WINDOW wnd;
    WINDOW dst;
    int slot;
} TIMER_EVENT;


/* event */
typedef union EVENT {
	int type;
	MOUSE_EVENT mouse;
    KEY_EVENT key;
    TIMER_EVENT timer;
} EVENT;


/*************************************************************************
	FUNCTIONS
 *************************************************************************/


/* this must be public, but you should use the BEGIN_PAINT / END_PAINT
   macros */
BOOL _do_paint_loop(WINDOW wnd, RECT *rct, BOOL overdraw_children);


/* returns the parent window */
WINDOW GetParentWindow(WINDOW wnd);


/* returns the lower (in z-order) sibling window */
WINDOW GetLowerWindow(WINDOW wnd);


/* returns the higher (in z-order) sibling window */
WINDOW GetHigherWindow(WINDOW wnd);


/* returns the bottom (in z-order) child window */
WINDOW GetBottomWindow(WINDOW wnd);


/* returns the top (in z-order) child window */
WINDOW GetTopWindow(WINDOW wnd);


/* returns the number of children windows */
unsigned GetWindowCount(WINDOW wnd);


/* returns the x coordinate of the window */
short GetWindowX(WINDOW wnd);


/* returns the y coordinate of the window */
short GetWindowY(WINDOW wnd);


/* returns the width of the window */
USHORT GetWindowWidth(WINDOW wnd);


/* returns the height of the window */
USHORT GetWindowHeight(WINDOW wnd);


/* returns the viewport width of the window */
USHORT GetWindowViewportWidth(WINDOW wnd);


/* returns the viewport height of the window */
USHORT GetWindowViewportHeight(WINDOW wnd);


/* returns the z-order of a window */
int GetWindowZOrder(WINDOW wnd);


/* returns the frame of the window */
void GetWindowFrame(WINDOW wnd, RECT *frame);


/* returns the screen position of the window */
void GetWindowRect(WINDOW wnd, RECT *rct);


/* returns the window cursor */
CURSOR GetWindowCursor(WINDOW wnd);


/* returns the window proc */
WINDOW_PROC *GetWindowProc(WINDOW wnd);


/* returns the window data */
void *GetWindowData(WINDOW wnd);


/* returns the window flags */
unsigned GetWindowFlags(WINDOW wnd);


/* returns TRUE if window B is ancestor of window A */
BOOL IsAncestorWindow(WINDOW a, WINDOW b);


/* redraws part of a window */
BOOL RedrawWindowRect(WINDOW wnd, RECT *rect, BOOL redraw_children);


/* redraws the whole window */
BOOL RedrawWindow(WINDOW wnd, BOOL redraw_children);


/* calls a window's proc with the given message and message data */
int SendMessage(WINDOW wnd, int msg, void *msg_data);


/* creates a window */
WINDOW CreateWindow(WINDOW_PROC *proc, WINDOW parent,
	short x, short y, USHORT width, USHORT height,
    unsigned flags, void *data);


/* destroys a window and its children; returns FALSE if given window
   is NULL */
BOOL DestroyWindow(WINDOW wnd);


/* inserts a window in another window; returns NULL if window is
   already a child of another window */
BOOL MapWindow(WINDOW parent, WINDOW wnd, int z_order);


/* removes a window from its parent */
BOOL UnmapWindow(WINDOW wnd);


/* resizes a window; returns FALSE if given position and size
   is the same with the current one */
BOOL ResizeWindow(WINDOW wnd,
	short x, short y, USHORT width, USHORT height,
    BOOL repaint);


/* sets the frame of the given window */
BOOL SetWindowFrame(WINDOW wnd, RECT *frame);


/* sets the z-order of a window; returns FALSE if given z-order
   is the same as the current z-order */
BOOL SetWindowZOrder(WINDOW wnd, int z_order);


/* sets a window's cursor; if the window has a cursor, then
   the cursor is set automatically when the mouse pointer is
   over the window */
BOOL SetWindowCursor(WINDOW wnd, CURSOR cur);


/* sets a window's procedure; it does not redraw the window */
BOOL SetWindowProc(WINDOW wnd, WINDOW_PROC *proc);


/* sets a window's data pointer; it does not redraw the window  */
BOOL SetWindowData(WINDOW wnd, void *data);


/* shows a window; returns FALSE if window is already shown */
BOOL ShowWindow(WINDOW wnd);


/* hides a window; returns FALSE if window is already hidden */
BOOL HideWindow(WINDOW wnd);


/* enables a window in order to accept events; returns FALSE if window
   is already enabled */
BOOL EnableWindow(WINDOW wnd);


/* disables a window; returns FALSE if window is already disabled */
BOOL DisableWindow(WINDOW wnd);


/* sets the input focus to the given window; returns FALSE if
   the given window already has the focus */
BOOL SetFocus(WINDOW wnd);


/* removes the focus from the given window;
   returns FALSE if no window had the input focus */
BOOL RemoveFocus(WINDOW wnd);


/* the given window grabs mouse and keyboard events; all events
   are redirected to it until the window releases events; returns FALSE
   if another window has grabbed events */
BOOL GrabEvents(WINDOW wnd);


/* releases events from the given window; if no window
   had grabbed events, then it returns FALSE */
BOOL ReleaseEvents(WINDOW wnd);


/* returns the root window */
WINDOW GetRootWindow();


/* returns the window with the focus */
WINDOW GetFocusWindow();


/* returns the window which has captured events */
WINDOW GetEventWindow();


/* returns window under the given x, y coordinates starting
   from the given window; coordinates are relative to the screen */
WINDOW WindowFromPoint(WINDOW wnd, short x, short y);


/* returns the next event from the event queue */
BOOL GetEvent(EVENT *event);


/* dispatches an event to the destination window of the event,
   only if the destination window is enabled */
int DispatchEvent(EVENT *event);


/* sets a timer in the given window; returns timer slot or 0
   if no timer is available */
unsigned SetTimer(WINDOW wnd, unsigned timeout);


/* releases a timer */
BOOL RemoveTimer(WINDOW wnd, unsigned timer_slot);


/* returns the click timeout */
unsigned GetClickTimeout();


/* sets the click timeout */
void SetClickTimeout(unsigned milisecs);


/* resets the number of clicks */
void ResetClicks();


/* returns the mouse button values */
void GetMouseButtons(int *left, int *right, int *middle);


/* sets the mouse buttons */
void SetMouseButtons(int left, int right, int middle);


/* creates a cursor from the given bitmap */
CURSOR CreateCursor(BITMAP *bmp, int x, int y);


/* destroy a cursor */
BOOL DestroyCursor(CURSOR cur);


/* returns the current cursor */
CURSOR GetCursor();


/* sets the current cursor; pass NULL to hide the mouse */
BOOL SetCursor(CURSOR cur);


#ifdef __cplusplus
   }
#endif


#endif


