//----------------------------------------------------------------------------
//
// C++ Objects for Allegro's gui
//
// Douglas Eleveld (D.J.Eleveld@anest.azg.nl)
//
// A huge lot of this code is strongly based on or copied from Allegro's gui.c
//
//----------------------------------------------------------------------------
#include "degui.h"
#include "internal.h"
#include <ctype.h>

/* Checking for double clicks is complicated. The user could release the
* mouse button at almost any point, and I might miss it if I am doing some
* other processing at the same time (eg. sending the single-click message).
* To get around this I install a timer routine to do the checking for me,
* so it will notice double clicks whenever they happen.
*/

static volatile int dclick_status, dclick_time;
static int dclick_install_count = 0;

#define DCLICK_START      0
#define DCLICK_RELEASE    1
#define DCLICK_AGAIN      2
#define DCLICK_NOT        3


// dclick_check:
// Double click checking user timer routine.
//

static void dclick_check (...)
   {
   // first click...
	if (dclick_status==DCLICK_START)
	   {
		if (!mouse_b)
		   {
         // aah! released first
			dclick_status = DCLICK_RELEASE;
			dclick_time = 0;
			return;
		   }
	   }
   // wait for second click
	else if (dclick_status==DCLICK_RELEASE)
	   {
		if (mouse_b)
		   {
         // yes! the second click
			dclick_status = DCLICK_AGAIN;
			dclick_time = 0;
			return;
		   }
	   }
	else
		return;

   // timeout?
	if (dclick_time++ > 10) dclick_status = DCLICK_NOT;
   }


static END_OF_FUNCTION(dclick_check);

/* message:
*  Sends a message to a DIALOG at a specific offset from the base DIALOG*.
*  The current result is asjusted and if the object returns anything other
*  than D_O_K, then obj will be the index of that object.
*/
int dialog_do::message (const int i, const int msg, const int c, int* res, int* obj)
   {
   int r;

   /* Make sure that there is a proper dialog procedure there */
	if(((_dialog+i)->proc==NULL)||(i<0)) return FALSE; /* the message couldn't be sent */

   /* Actually send the message */
	r = (_dialog+i)->proc(msg,(_dialog+i),c);
	if(r!=D_O_K)
	   {
		*res |= r;
		*obj = i;
	   }
   /* The message worked */
	return TRUE;
   }

/* dialog_message:
*  Sends a message to all the objects in a dialog. If any of the objects
*  return values other than D_O_K, returns the value and sets obj to the
*  object which produced it.
*/
int dialog_do::dialog_message (int msg, int c, int *obj)
   {
	int count;
	int res;
	int r;

	if (msg == MSG_DRAW) show_mouse(NULL);

	res = D_O_K;
	for (count=0; _dialog[count].proc; count++)
	   {
		if (!(_dialog[count].flags & D_HIDDEN))
		   {
			r = send_message(count, msg, c);
			if (r != D_O_K)
			   {
				res |= r;
				*obj = count;
			   }
		   }
	   }
	if (msg == MSG_DRAW) show_mouse(screen);

	return res;
   }


/* find_mouse_object:
*  Finds which object the mouse is on top of.
*/
int dialog_do::find_mouse_object (void)
   {
   /* finds which object the mouse is on top of */
	int mouse_object = -1;
	int c;

	for (c=0; _dialog[c].proc; c++)
		if ((mouse_x >= _dialog[c].x) && (mouse_y >= _dialog[c].y) &&
			(mouse_x < _dialog[c].x + _dialog[c].w) && (mouse_y < _dialog[c].y + _dialog[c].h) &&
	      (!(_dialog[c].flags & (D_HIDDEN | D_DISABLED))))
	      mouse_object = c;

	return mouse_object;
   }


/* offer_focus:
*  Offers the input focus to a particular object.
*/
int dialog_do::offer_focus(int obj, int force)
   {
	int res = D_O_K;

	if((obj == focus_obj)||
	   ((obj >= 0)&&(_dialog[obj].flags&(D_HIDDEN|D_DISABLED))))
	   return D_O_K;

   /* check if object wants the focus */
	if(obj >= 0)
	   {
      if((_dialog+obj)->proc!=NULL)
         {
         res = (_dialog+obj)->proc(MSG_WANTFOCUS, _dialog+obj, 0);

		   if(res & D_WANTFOCUS) res ^= D_WANTFOCUS;
   		else obj = -1;
         }
      else obj = -1;
	   }

	if((obj >= 0) || (force))
	   {
      /* take focus away from old object */
		if(focus_obj >= 0)
		   {
			res |= send_message(focus_obj, MSG_LOSTFOCUS, 0);
			if(res & D_WANTFOCUS)
			   {
				if(obj < 0)
					return D_O_K;
				else
					res &= ~D_WANTFOCUS;
			   }
			_dialog[focus_obj].flags &= ~D_GOTFOCUS;
			show_mouse(NULL);
			res |= send_message(focus_obj, MSG_DRAW, 0);
			show_mouse(screen);
		   }

		focus_obj = obj;

      /* give focus to new object */
		if (obj >= 0)
		   {
			show_mouse(NULL);
			_dialog[obj].flags |= D_GOTFOCUS;
			res |= send_message(obj, MSG_GOTFOCUS, 0);
			res |= send_message(obj, MSG_DRAW, 0);
			show_mouse(screen);
		   }
	   }

	return res;
   }


#define MAX_OBJECTS     512

typedef struct OBJ_LIST
   {
	int index;
	int diff;
   } OBJ_LIST;


/* obj_list_cmp:
*  Callback function for qsort().
*/
static int obj_list_cmp(const void *e1, const void *e2)
   {
	return (((OBJ_LIST *)e1)->diff - ((OBJ_LIST *)e2)->diff);
   }


/* cmp_right:
*  Comparison function for right arrow key movement.
*/
static int cmp_right(DIALOG *d1, DIALOG *d2)
   {
	int ret = (d2->x - d1->x) + ABS(d1->y - d2->y) * 8;

	if (d1->x >= d2->x)
		ret += 0x10000;

	return ret;
   }


/* cmp_left:
*  Comparison function for left arrow key movement.
*/
static int cmp_left(DIALOG *d1, DIALOG *d2)
   {
	int ret = (d1->x - d2->x) + ABS(d1->y - d2->y) * 8;

	if (d1->x <= d2->x)
		ret += 0x10000;

	return ret;
   }


/* cmp_down:
*  Comparison function for down arrow key movement.
*/
static int cmp_down(DIALOG *d1, DIALOG *d2)
   {
	int ret = (d2->y - d1->y) + ABS(d1->x - d2->x) * 8;

	if (d1->y >= d2->y)
		ret += 0x10000;

	return ret;
   }


/* cmp_up:
*  Comparison function for up arrow key movement.
*/
static int cmp_up(DIALOG *d1, DIALOG *d2)
   {
	int ret = (d1->y - d2->y) + ABS(d1->x - d2->x) * 8;

	if (d1->y <= d2->y)
		ret += 0x10000;

	return ret;
   }


/* move_focus:
*  Handles arrow key and tab movement through a dialog, deciding which
*  object should be given the input focus.
*/
int dialog_do::move_focus(long ch)
   {
	int (*cmp)(DIALOG *d1, DIALOG *d2);
	OBJ_LIST obj[MAX_OBJECTS];
	int obj_count = 0;
	int fobj, c;
	int res = D_O_K;

   /* choose a comparison function */
	switch (ch >> 8)
	   {
      // Cycle through the objects
		case KEY_TAB:
         {
         /* find a forward object from the focused object to give the focus to */
         fobj = focus_obj;
	      for(c = fobj+1; _dialog[c].proc; c++)
	         {
		      res |= offer_focus(c, FALSE);
            if(res&D_WANTFOCUS) return res;
            }
         /* find a forward object from the beginning to give the focus to */
	      for(c = 0; c<focus_obj; c++)
	         {
		      res |= offer_focus(c, FALSE);
            if(res&D_WANTFOCUS) return res;
            }
         return D_O_K;
         }
		case KEY_RIGHT:   cmp = cmp_right;  break;
		case KEY_LEFT:    cmp = cmp_left;   break;
		case KEY_DOWN:    cmp = cmp_down;   break;
		case KEY_UP:      cmp = cmp_up;     break;
		default:          return D_O_K;
	   }

   /* fill temporary table */
	for(c=0; _dialog[c].proc; c++)
	   {
		if((focus_obj < 0) || (c != focus_obj))
		   {
			obj[obj_count].index = c;
			if (focus_obj >= 0)
				obj[obj_count].diff = cmp(_dialog+focus_obj, _dialog+c);
			else
				obj[obj_count].diff = c;
			obj_count++;
			if (obj_count >= MAX_OBJECTS)
				break;
		   }
	   }

   /* sort table */
	qsort(obj, obj_count, sizeof(OBJ_LIST), obj_list_cmp);

   /* find an object to give the focus to */
	fobj = focus_obj;
	for (c=0; c<obj_count; c++)
	   {
		res |= offer_focus(obj[c].index, FALSE);
		if (fobj != focus_obj)
			break;
	   }

	return res;
   }

//----------------------------------------------------------------------------
// Allegro dialog managing base class simulating Allegro's do_dialog function
//----------------------------------------------------------------------------
// Constructor
dialog_do::dialog_do (DIALOG* dia)
   :number(-1),
   max_number(MAXINT),
	_close(false),
   _redraw(false),
   _escape_exits(true),
   _cant_exit(false)
   {
   // Set up the array with the user info
   _dialog = dia;
   }
//----------------------------------------------------------------------------
// Constructor
dialog_do::dialog_do (const int num)
   :number(0),
   max_number(num+2),
	_close(false),
   _redraw(false),
   _escape_exits(true),
   _cant_exit(false)
   {
   // Set up the arrays with enough space
   _dialog = new DIALOG[num+2];
   if(_dialog==NULL) degui_no_memory();
   }
//----------------------------------------------------------------------------
// Destructor
dialog_do::~dialog_do (void)
   {
   if(number>=0) delete _dialog;
   }
//----------------------------------------------------------------------------
// Add a procedure object to the dialog
void dialog_do::add (int(*proc)(int,DIALOG*,int), const int x,  const int y, const int w, const int h, const int fg, const int bg, const int key, const int flags, const int d1, const int d2, void *dp)
	{
   // We aren't allowed to add to a user defined DIALOG array
   // Make sure that we have the room
   if((number<0)||(number>=max_number)) return;

	// Make the new object
	_dialog[number].proc = proc;
	_dialog[number].x = x;
	_dialog[number].y = y;
	_dialog[number].w = w;
	_dialog[number].h = h;
	_dialog[number].fg = fg;
	_dialog[number].bg = bg;
	_dialog[number].key = key;
	_dialog[number].flags = flags;
	_dialog[number].d1 = d1;
	_dialog[number].d2 = d2;
   _dialog[number].dp = dp;

   _dialog[number+1].proc = NULL;
   _dialog[number+1].x = 0;
   _dialog[number+1].y = 0;
   _dialog[number+1].w = 0;
   _dialog[number+1].h = 0;
   _dialog[number+1].fg = 0;
   _dialog[number+1].bg = 0;
   _dialog[number+1].key = 0;
   _dialog[number+1].flags = 0;
   _dialog[number+1].d1 = 0;
   _dialog[number+1].d2 = 0;
   _dialog[number+1].dp = NULL;
   number ++;
	}
//----------------------------------------------------------------------------
// After the dialog is setup a bit we can execute it
int dialog_do::execute (const int fobj)
   {
	int res = D_REDRAW;
	int obj;
	int c;
	long ch = 0;
	int ox, oy;

   focus_obj = fobj;

   // set up dclick checking code
	if (dclick_install_count <= 0)
	   {
		LOCK_VARIABLE(dclick_status);
		LOCK_VARIABLE(dclick_time);
		LOCK_FUNCTION(dclick_check);
		install_int(dclick_check, 20);
		dclick_install_count = 1;
	   }
	else
		dclick_install_count++;

   // initialise the dialog, this
	set_clip(screen, 0, 0, SCREEN_W-1, SCREEN_H-1);
	res |= dialog_message(MSG_START, 0, &obj);

	mouse_obj = find_mouse_object();
	if (mouse_obj >= 0)
		_dialog[mouse_obj].flags |= D_GOTMOUSE;

	for (c=0; _dialog[c].proc; c++)
	   {
		if (c == focus_obj)
			_dialog[c].flags |= D_GOTFOCUS;
		else
			_dialog[c].flags &= ~D_GOTFOCUS;
	   }

   /* while dialog is active */
	while (((!(res & D_CLOSE))&&(_close==false))||(_cant_exit==true))
	   {
		res &= ~D_USED_CHAR;

      /* need to draw it? */
		if((res & D_REDRAW)||(_redraw==true))
		   {
         _redraw = false;
			res ^= D_REDRAW;
			res |= dialog_message(MSG_DRAW, 0, &obj);
		   }

      /* need to give the input focus to someone? */
		if (res & D_WANTFOCUS)
		   {
			res ^= D_WANTFOCUS;
			res |= offer_focus(obj, FALSE);
		   }

      /* has mouse object changed? */
		c = find_mouse_object();
		if (c != mouse_obj)
		   {
			if (mouse_obj >= 0)
			   {
				_dialog[mouse_obj].flags &= ~D_GOTMOUSE;

				message(mouse_obj,MSG_LOSTMOUSE,0,&res,&obj);
			   }
			if (c >= 0)
			   {
				_dialog[c].flags |= D_GOTMOUSE;
				if(message(c,MSG_GOTMOUSE,0,&res,&obj)==FALSE) c = -1;
			   }
			mouse_obj = c;

         /* move the input focus as well? */
			if ((gui_mouse_focus) && (mouse_obj != focus_obj) && (mouse_obj>0))
				res |= offer_focus(mouse_obj, TRUE);
		   }

      /* deal with mouse button clicks */
		if (mouse_b)
		   {
			res |= offer_focus(mouse_obj, FALSE);

			if (mouse_obj >= 0)
			   {
				dclick_time = 0;
				dclick_status = DCLICK_START;
				ox = mouse_x;
				oy = mouse_y;

            /* send click message */
				if(message(mouse_obj,MSG_CLICK,0,&res,&obj)==FALSE)
               {
               mouse_obj = -1;
               continue;
               }
				if (res==D_O_K)
				   {
					do
					   {
						if ((ABS(ox-mouse_x) > 8) || (ABS(oy-mouse_y) > 8))
						   {
							dclick_status = DCLICK_NOT;
							break;
						   }
					   } while ((dclick_status != DCLICK_AGAIN) && (dclick_status != DCLICK_NOT));

               // double click!
					if ((dclick_status==DCLICK_AGAIN) &&
						(mouse_x >= _dialog[mouse_obj].x) &&
	               (mouse_y >= _dialog[mouse_obj].y) &&
					   (mouse_x <= _dialog[mouse_obj].x + _dialog[mouse_obj].w) &&
					   (mouse_y <= _dialog[mouse_obj].y + _dialog[mouse_obj].h))
					   {
						message(mouse_obj,MSG_DCLICK,0,&res,&obj);
					   }
				   }
			   }
			continue;
		   }

      /* deal with keyboard input */
		if (keypressed())
		   {
   		ch = readkey();

         /* let object deal with the key? */
			if (focus_obj >= 0)
			   {
            // Try to send a key to the focus object
				if(message(focus_obj,MSG_CHAR,ch,&res,&obj)==FALSE)
				   focus_obj = -1;
				if (res & D_USED_CHAR)
					continue;
			   }

         /* keyboard shortcut? */
			if (ch & 0xff)
			   {
				for (c=0; _dialog[c].proc; c++)
				   {
					if ((tolower(_dialog[c].key) == tolower((ch & 0xff))) &&
						(!(_dialog[c].flags & (D_HIDDEN | D_DISABLED))))
					   {
						message(c,MSG_KEY,ch,&res,&obj);
						ch = 0;
						break;
					   }
				   }
				if (!ch)
					continue;
			   }

         /* broadcast in case any other objects want it */
			for (c=0; _dialog[c].proc; c++)
			   {
				if (!(_dialog[c].flags & (D_HIDDEN | D_DISABLED)))
				   {
					message(c,MSG_XCHAR,ch,&res,&obj);
					if (res & D_USED_CHAR)
						continue;
				   }
			   }

         /* pass <CR> or <SPACE> to selected object? */
			if ((((ch & 0xff) == 10) || ((ch & 0xff) == 13) ||
				((ch & 0xff) == 32)) && (focus_obj >= 0))
			   {
				if(message(focus_obj,MSG_KEY,0,&res,&obj)==FALSE)
               focus_obj = -1;
				continue;
			   }

         /* ESC closes _dialog? */
			if(((ch & 0xff) == 27)&&(_escape_exits==true))
			   {
				res |= D_CLOSE;
				obj = -1;
				continue;
			   }

         /* move focus around the _dialog? */
			res |= move_focus(ch);
		   }

      /* send idle messages */
		res |= dialog_message(MSG_IDLE, 0, &obj);

      // Stop any exits if necessary
      if(_cant_exit==true)
         {
         _close = false;
         res &= ~D_CLOSE;
         }
	   }

   /* send the finish messages */
	dialog_message(MSG_END, 0, &obj);

   // remove the double click handler
	dclick_install_count--;
	if (dclick_install_count <= 0)
		remove_int(dclick_check);

	if (mouse_obj >= 0)
		_dialog[mouse_obj].flags &= ~D_GOTMOUSE;

   // return the one that caused the exit
   return obj;
   }
//----------------------------------------------------------------------------
/* popup_dialog:
 *  Like do_dialog(), but it stores the data on the screen before drawing
 *  the dialog and restores it when the dialog is closed. The screen area
 *  to be stored is calculated from the dimensions of the first object in
 *  the dialog, so all the other objects should lie within this one.
 */
int dialog_do::popup (int fobj)
   {
   BITMAP *bmp;
   int ret;

   int min_x = MAXINT;
   int min_y = MAXINT;
   int max_x = -MAXINT;
   int max_y = -MAXINT;
   int c;

   /* find the extents of the dialog */ 
   for (c=0; _dialog[c].proc; c++)
      {
      // Adjust only objects with size
      if((_dialog[c].w>0)&&(_dialog[c].h>0))
         {
         if (_dialog[c].x < min_x) min_x = _dialog[c].x;
         if (_dialog[c].y < min_y) min_y = _dialog[c].y;
         if (_dialog[c].x + _dialog[c].w > max_x) max_x = _dialog[c].x + _dialog[c].w;
         if (_dialog[c].y + _dialog[c].h > max_y) max_y = _dialog[c].y + _dialog[c].h;
         }
      }
   bmp = create_bitmap(max_x-min_x, max_y-min_y);

   if(bmp)
      {
      show_mouse(NULL);
      blit(screen, bmp, min_x, min_y, 0, 0, max_x-min_x+1, max_y-min_y+1);
      show_mouse(screen);
      }

   // Do the execute
   ret = execute(fobj);

   if(bmp)
      {
      show_mouse(NULL);
      blit(bmp, screen, min_x, min_y, 0, 0, max_x-min_x+1, max_y-min_y+1);
      destroy_bitmap(bmp);
      show_mouse(screen);
      }
   return ret;
   }

//----------------------------------------------------------------------------
// Redraw the dialog immediately or just request a redraw
void dialog_do::redraw (const bool immediate)
	{
   // Just request a redraw to be done later
   if(immediate==false)
      {
      _redraw = true;
      }
   // Force a compete redraw of all objects right now
   else
      {
      // Send a redraw to all the objects if we are running
      show_mouse(NULL);
      for(int i=0;i<number;i++)
         _dialog[i].proc(MSG_DRAW,&_dialog[i],0);
      show_mouse(screen);
      _redraw = false;
      }
	}
//----------------------------------------------------------------------------
/* centre_dialog:
 *  Moves all the objects in a dialog so that the dialog is centered in
 *  the screen.
 */
void dialog_do::center (const int i)
   {
   int min_x = MAXINT;
   int min_y = MAXINT;
   int max_x = -MAXINT;
   int max_y = -MAXINT;
   int xc, yc;
   int c;

   /* find the extents of the dialog */ 
   for (c=i; _dialog[c].proc; c++)
      {
      // Adjust only objects with size
      if((_dialog[c].w>0)&&(_dialog[c].h>0))
         {
         if (_dialog[c].x < min_x) min_x = _dialog[c].x;

         if (_dialog[c].y < min_y) min_y = _dialog[c].y;

         if (_dialog[c].x + _dialog[c].w > max_x) max_x = _dialog[c].x + _dialog[c].w;

         if (_dialog[c].y + _dialog[c].h > max_y) max_y = _dialog[c].y + _dialog[c].h;
         }
      }

   /* how much to move by? */
   xc = (SCREEN_W - (max_x - min_x)) / 2 - min_x;
   yc = (SCREEN_H - (max_y - min_y)) / 2 - min_y;

   /* move it */
   for (c=i; _dialog[c].proc; c++)
      {
      _dialog[c].x += xc;
      _dialog[c].y += yc;
      }
   }
//---------------------------------------------------------------------------

