//----------------------------------------------------------------------------
//
// C++ Objects for Allegro's gui
//
// Douglas Eleveld (D.J.Eleveld@anest.azg.nl)
//
//----------------------------------------------------------------------------

//----------------------------------------------------------------------------
#include <stdio.h>
#include <string.h>
#include "degui.h"
#include "internal.h"
#include "build_di.h"
#include "build_mu.h"
#include "manipul.h"

//----------------------------------------------------------------------------
// Helper stuff for the builder dialog object managing class
//----------------------------------------------------------------------------
// The header for the dialog files
static char *dialog_file_header = "Allegro gui builder dialog file\n";

//----------------------------------------------------------------------------
// Allegro dialog managing class
//----------------------------------------------------------------------------
// The main dialog
builder_dialog *gui_builder;

//----------------------------------------------------------------------------
builder_dialog::builder_dialog (const int num)
   :dialog(num+5),
   move_key(0),
   resize_key(0),
   pushback_key(0),
   pullfront_key(0),
   delete_key(0),
   granularity(5)
   {
   // Setup the menu stuff
   file_menu.add("&Load",load_dialog);
   file_menu.add("&Append",append_dialog);
   file_menu.add("&Save",save_dialog);
   file_menu.add("Ex&port",dialog_export);
   file_menu.add();
   file_menu.add("E&xit",main_dialog_close);

   panel_menu.add("&Box",::add_box_object);
   panel_menu.add("&Shadow Box",::add_shadow_box_object);
   panel_menu.add("&Raised",::add_panel_raised_object);
   panel_menu.add("S&unken",::add_panel_sunken_object);
   panel_menu.add("R&idge",::add_panel_ridge_object);
   panel_menu.add("&Groove",::add_panel_groove_object);
   panel_menu.add();
   panel_menu.add("&Window",::add_window_object);

   text_menu.add("&Text",::add_text_object);
   text_menu.add("&Centerd text",::add_centerd_text_object);
   text_menu.add("&Editable text",::add_edittext_object);
   text_menu.add("T&ext Box",::add_textbox_object);

   selectable_menu.add("&Button",::add_button_object);
   selectable_menu.add("&Checkbox",::add_checkbox_object);
   selectable_menu.add("&Radio button",::add_radio_button_object);

   list_menu.add("&List Box",::add_list_object);
   list_menu.add("&Hidden list Box",::add_hidden_list_object);

   info_menu.add("P&rogress bar",::add_progress_bar_object);

   object_menu.add("&Panel",panel_menu);
   object_menu.add("&Bitmap",::add_bitmap_object);
   object_menu.add("&Text",text_menu);
   object_menu.add("&Selectable",selectable_menu);
   object_menu.add("&List",list_menu);
   object_menu.add("&Info",info_menu);
   object_menu.add("&User",::add_user_object);

   options_menu.add("&Palette",::set_builder_palette);
   options_menu.add("");
   options_menu.add("&Center",::center_dialog);
   options_menu.add("&Granularity",::placement_granularity);
   options_menu.add("&Resolution",::choose_resolution);

   help_menu.add("&Help",::help_procedure);
   help_menu.add("");
   help_menu.add("&About",::about_procedure);

   allegro_menu.add("&File",file_menu);
   allegro_menu.add("&Object",object_menu);
   allegro_menu.add("O&ptions",options_menu);
   allegro_menu.add("&Help",help_menu);

   // Allocate the necessary control dialog objects
   grid = new grid_object(granularity,granularity);
   if(grid==NULL) degui_no_memory();
   main_menu = new menu_object(allegro_menu);
   if(main_menu==NULL) degui_no_memory();

   // Add the necessary control objects
   // Object 0 is added in the dialog constructor
   dialog::add(builder_dialog_proc,0,0,0,0,0,0,0,0,0,0,this); // Object 0
   dialog::add(*grid);                                        // Object 1
   dialog::add(*main_menu);                                   // Object 2

   // Don't let an escape stop the dialog ar anything else
   _escape_exits = false;
   _cant_exit = true;
   }
//----------------------------------------------------------------------------
builder_dialog::~builder_dialog (void)
   {
   // Delete the all the C++ objects
   for(int i=3;i<number;i++)
      {
      // Get the object
      dialog_object *object = (dialog_object*)_dialog[i].dp;
      if(object!=NULL)
         {
         // Get the description
         object_description *descript = (object_description*)object->dataptr;

         // Delete the description
         delete descript;

         // Delete the object
         delete object;
         _dialog[i].dp = NULL;
         _dialog[i].proc = NULL;
         }
      }
   // delete the internal stuff
   delete grid;
   _dialog[1].dp = NULL;
   delete main_menu;
   _dialog[2].dp = NULL;
   }
//----------------------------------------------------------------------------
// Add an object to the dialog
void builder_dialog::add (dialog_object *obj)
	{
   // Add the object to the base dialog
   dialog::add(*obj);

   // The description has to have been added before we set the index
   if(obj->dataptr!=NULL)
      {
      // Get the description and do some setup
      object_description *desc = (object_description*)obj->dataptr;
      desc->set_index(number-3);
      }
	}
//----------------------------------------------------------------------------
// Set some info about the dialog working
void builder_dialog::set_move_key (const int key)
	{
	move_key = key;
	}
void builder_dialog::set_multimove_key (const int key)
	{
	multimove_key = key;
	}
void builder_dialog::set_resize_key (const int key)
	{
	resize_key = key;
	}
void builder_dialog::set_pushback_key (const int key)
	{
	pushback_key = key;
	}
void builder_dialog::set_pullfront_key (const int key)
	{
	pullfront_key = key;
	}
void builder_dialog::set_delete_key (const int key)
	{
	delete_key = key;
	}
void builder_dialog::set_granularity (const int gran)
	{
	granularity = gran;
   grid->set_granularity(gran,gran);
	}
//----------------------------------------------------------------------------
// An Allegro object to get basic messages for the builder
int builder_dialog::builder_dialog_proc (int msg, DIALOG *d, int c)
    {
    int ret = D_O_K;

    // Get the dialog object
    builder_dialog *dia = (builder_dialog*)d->dp;

    // Look at the message that we were sent
    switch(msg)
       {
       case(MSG_START):
          {
          // Show the about dialog at startup
          return about_procedure();
          }
       case(MSG_XCHAR):
          {
          // Check for an alt-x for an exit
          if(((c>>8)==KEY_X)&&(key_shifts&=KB_ALT_FLAG))
             {
             dia->_cant_exit = false;
             ret |= D_CLOSE;
             }
          // Check for the keycode for a delete
          if((c>>8)==dia->delete_key)
             {
             // Get the Allegro dialogs
             DIALOG *current = dia->_dialog;

             // Find the smallest object being pointed to
             int index = dia->find_mouse_object();
             if(index<0) return D_O_K;

             // Make sure it was really in there
             if(current[index].proc==NULL) return D_O_K;

             // Make sure it was in the range that we are allowed to delete
             if((index<3)||(index>=dia->number)) return D_O_K;

             // Get the objects description
             dialog_object *self = (dialog_object*)current[index].dp;
             object_description* desc = (object_description*)(self->dataptr);

             const char *descript = NULL;
             if(desc!=NULL) descript = desc->get_type();

             // Make sure the user wants to delete the object
             if(alert("Delete object?",(char*)descript,NULL,"&Ok","&Cancel",0,0)!=1) return D_O_K;

             dia->remove(*self);
             delete self;  // Self will be automatically disconnected

             // Redraw and say that we used the character
             ret = D_REDRAW | D_USED_CHAR;
             }
          // Check for an object pushback
          else if((c>>8)==dia->pushback_key)
             {
             // Get the Allegro dialogs
             DIALOG *current = dia->_dialog;

             // Find the object being pointed to
             int index = dia->find_mouse_object();

             // Make sure that the object isn't pushed as far as we want
             if(index<=3) return D_O_K;

             // Tell what we might do
             if(alert("Push object back?",NULL,NULL,"Yes","No",'y','n')==2) return D_O_K;

             // Get the two objects that will be swapped
             dialog_object *front = (dialog_object*)current[index-1].dp;
             dialog_object *back = (dialog_object*)current[index].dp;

             // Swap the dialogs and connect properly to kkep sizes a flags proper
             DIALOG temp = current[index-1];
             front->connect(&temp);

             current[index-1] = current[index];
             back->connect(&current[index-1]);

             current[index] = temp;
             front->connect(&current[index]);

             ret = D_REDRAW | D_USED_CHAR;
             }
          // Check for an object pullfront
          else if((c>>8)==dia->pullfront_key)
             {
             // Get the Allegro dialogs
             DIALOG *current = dia->_dialog;

             // Find the object being pointed to
             int index = dia->find_mouse_object();

             // Make sure that we pointed to an object
             if(index<0) return D_O_K;

             // Make sure it was really in there
             if(current[index].proc==NULL) return D_O_K;

             // Make sure it was not pulled as far as possible allready
             if(current[index+1].proc==NULL) return D_O_K;

             // Make sure that the object isn't pulled as far as we want
             if(index>=dia->number) return D_O_K;

             // Tell what we might do
             if(alert("Pull object to front?",NULL,NULL,"Yes","No",'y','n')==2) return D_O_K;

             // Get the two objects that will be swapped
             dialog_object *front = (dialog_object*)current[index+1].dp;
             dialog_object *back = (dialog_object*)current[index].dp;

             // Swap the dialogs and connect properly to kkep sizes a flags proper
             DIALOG temp = current[index+1];
             front->connect(&temp);

             current[index+1] = current[index];
             back->connect(&current[index+1]);

             current[index] = temp;
             front->connect(&current[index]);

             ret = D_REDRAW | D_USED_CHAR;
             }
          // Check for an object move
          else if((c>>8)==dia->move_key)
             {
             // Get the Allegro dialogs
             DIALOG *current = dia->_dialog;

             // Find the object being pointed to
             int index = dia->find_mouse_object();

             // Make sure that we pointed to an object
             if(index<0) return D_O_K;

             // Make sure it was really in there
             if(current[index].proc==NULL) return D_O_K;

             // Move the object where the user wants it
             do_object_move(&current[index],current,dia->granularity);

             ret = D_REDRAW | D_USED_CHAR;
             }
          // Check for an object multi move
          else if((c>>8)==dia->multimove_key)
             {
             // Get the Allegro dialogs
             DIALOG *current = dia->_dialog;

             // Find the object being pointed to
             int index = dia->find_mouse_object();

             // Make sure that we pointed to an object
             if(index<0) return D_O_K;

             // Make sure it was really in there
             if(current[index].proc==NULL) return D_O_K;

             // Move the object(s) where the user wants it
             do_object_multimove(&current[index],current,dia->granularity);

             ret = D_REDRAW | D_USED_CHAR;
             }
          // Check for an object resize
          else if((c>>8)==dia->resize_key)
             {
             // Get the Allegro dialogs
             DIALOG *current = dia->_dialog;

             // Find the object being pointed to
             int index = dia->find_mouse_object();

             // Make sure that we pointed to an object
             if(index<0) return D_O_K;

             // Make sure it was really in there
             if(current[index].proc==NULL) return D_O_K;

             // Move the object where the user wants it
             do_object_resize(&current[index],current,dia->granularity);

             ret = D_REDRAW | D_USED_CHAR;
             }
          else
             {
             // Recieved an unknown MSG_XCAR
             }
          break;
          }
       default:
          break;
       }
    return ret;
    }

//----------------------------------------------------------------------------
// Load the dialogs from disk
bool builder_dialog::load (const char *filename, const bool append)
	{
   // Try to open the file
   FILE *stream = fopen(filename,"rt");
   if(stream==NULL) return false;

   // Load the header and see if it's OK
   char temp[2000];
   fgets(temp,999,stream);
   if(strcmp(temp,dialog_file_header)==1)
      {
      fclose(stream);
      return false;
      }

   // Remove the existing objects if this is not an append
   if(append==false) number = 3;

   // Read in each line and get it decoded
   while(true)
      {
      // Read in a line for decoding
      fgets(temp,999,stream);
      if(strlen(temp)<3) break;

      // Decode the object
      object_description *desc = new object_description;
      dialog_object *object = desc->decode(temp);
      object->dataptr = desc;

      // Now get more info about placement
      int x, y, w, h;
      fgets(temp,999,stream);
      sscanf(temp,"%i,%i,%i,%i",&x,&y,&w,&h);

      // Modify the object stuff
      object->set_x(x);
      object->set_y(y);
      object->set_w(w);
      object->set_h(h);

      // Now add the object to our builder dialog
      add(object);
      }
   fclose(stream);
   return true;
	}
//----------------------------------------------------------------------------
// Save the dialogs to disk
void builder_dialog::save (const char *filename)
	{
   // Try to delete the file and open a new one
   FILE *stream = fopen(filename,"wt");
   if(stream==NULL) return;

   fprintf(stream,"%s",dialog_file_header);

   // Write the object declarations out
   char temp[2000];
   for(int i=3;i<number;i++)
      {
      // Find the description
      dialog_object *object = (dialog_object*)_dialog[i].dp;
      object_description *descript = (object_description*)object->dataptr;

      // Encode the objects
      descript->encode(temp);

      // Send the encoded stuff to file
      fprintf(stream,"%s\n",temp);

      // Send out the placements
      fprintf(stream,"%i,%i,%i,%i\n",_dialog[i].x,_dialog[i].y,_dialog[i].w,_dialog[i].h);
      }
   fprintf(stream,"\n\n");
   fclose(stream);
	}
//----------------------------------------------------------------------------
// Export a C++ object file
void builder_dialog::export (const char *filename)
   {
   // Try to open the file
   FILE *stream = fopen(filename,"wt");
   if(stream==NULL) return;

   fprintf(stream,"   // Object definitions - DEGUI %s\n",DEGUI_VERSION_STR);

   // Write the object declarations out
   for(int i=3;i<number;i++)
      {
      // Find the description
      dialog_object *object = (dialog_object*)_dialog[i].dp;
      object_description *descript = (object_description*)object->dataptr;

      // Check if there are any arguments
      const char *args = descript->get_arguments();

      // Output for no arguments
      if(strcmp(args,"NULL")==0)
         {
         fprintf(stream,"   %s %s;\n",descript->get_type(),descript->get_name());
         }
      // Output for special arguments
      else
         {
         fprintf(stream,"   %s %s(%s);\n",descript->get_type(),descript->get_name(),args);
         }
      }

   fprintf(stream,"\n   // Add Objects to the dialog\n");

   // Make a dialog and add it to them
   fprintf(stream,"   dialog the_dialog;\n");
   for(int i=3;i<number;i++)
      {
      // Find the description
      dialog_object *object = (dialog_object*)_dialog[i].dp;
      object_description *descript = (object_description*)object->dataptr;

      fprintf(stream,"   the_dialog.add(%s,",descript->get_name());
      fprintf(stream,"%i,%i,%i,%i);\n",object->x(),object->y(),object->w(),object->h());
      }
   // Possibly center the dialog
   if(alert("Should dialog be centered?",NULL,NULL,"&Yes","&No",'y','n')==1)
      {
      fprintf(stream,"   the_dialog.center();\n");
      }
   // Possibly popup or execute the dialog
   if(alert("Should dialog replace it's","background when done?",NULL,"&Yes","&No",'y','n')==1)
      {
      fprintf(stream,"   the_dialog.popup();\n");
      }
   else
      {
      fprintf(stream,"   the_dialog.execute();\n\n\n");
      }
   fclose(stream);
   }

//----------------------------------------------------------------------------
// Object adding procedures
//----------------------------------------------------------------------------
int builder_dialog::add_box_object (void)
   {
   // Create the object
   box_object *object = new box_object;
   object_description *desc = new object_description;
	if((object==NULL)||(desc==NULL)) degui_no_memory();

   // Describe the object
   desc->set_type(box_type);

   // Link the object and the description
   object->dataptr = desc;

   // Add the object to the dialog
   add(object);
   return D_REDRAW;
   }
//----------------------------------------------------------------------------
int builder_dialog::add_shadow_box_object (void)
   {
   // Create the object
   shadow_box_object *object = new shadow_box_object;
   object_description *desc = new object_description;
	if((object==NULL)||(desc==NULL)) degui_no_memory();

   // Describe the object
   desc->set_type(shadow_box_type);

   // Link the object and the description
   object->dataptr = desc;

   add(object);
   return D_REDRAW;
   }
//----------------------------------------------------------------------------
int builder_dialog::add_panel_raised_object (void)
   {
   // Create the object
   panel_raised_object *object = new panel_raised_object;
   object_description *desc = new object_description;
	if((object==NULL)||(desc==NULL)) degui_no_memory();

   // Describe the object
   desc->set_type(panel_raised_type);

   // Link the object and the description
   object->dataptr = desc;

   // Add the object to the dialog
   add(object);
   return D_REDRAW;
   }
//----------------------------------------------------------------------------
int builder_dialog::add_panel_sunken_object (void)
   {
   // Create the object
   panel_sunken_object *object = new panel_sunken_object;
   object_description *desc = new object_description;
	if((object==NULL)||(desc==NULL)) degui_no_memory();

   // Describe the object
   desc->set_type(panel_sunken_type);

   // Link the object and the description
   object->dataptr = desc;

   // Add the object to the dialog
   add(object);
   return D_REDRAW;
   }
//----------------------------------------------------------------------------
int builder_dialog::add_panel_ridge_object (void)
   {
   // Create the object
   panel_ridge_object *object = new panel_ridge_object;
   object_description *desc = new object_description;
	if((object==NULL)||(desc==NULL)) degui_no_memory();

   // Describe the object
   desc->set_type(panel_ridge_type);

   // Link the object and the description
   object->dataptr = desc;

   // Add the object to the dialog
   add(object);
   return D_REDRAW;
   }
//----------------------------------------------------------------------------
int builder_dialog::add_panel_groove_object (void)
   {
   // Create the object
   panel_groove_object *object = new panel_groove_object;
   object_description *desc = new object_description;
	if((object==NULL)||(desc==NULL)) degui_no_memory();

   // Describe the object
   desc->set_type(panel_groove_type);

   // Link the object and the description
   object->dataptr = desc;

   // Add the object to the dialog
   add(object);
   return D_REDRAW;
   }
//----------------------------------------------------------------------------
int builder_dialog::add_bitmap_object (void)
   {
   // Data definitions
   char *variable_name = "";
   char *bit_name = "bitmap";
   char filename[80];
   strcpy(filename,"bitmap.pcx");

   // Object definitions
   panel_raised_object object_0;
   button_object object_1("&Ok",0,true);
   button_object object_2("&Cancel",0,true);
   text_object object_4("Variable name:");
   edittext_object object_9(bit_name,49);
   edittext_object object_11(variable_name,49);
   text_object bitmap_name("Bitmap name:");
   text_object object_7("Create bitmap object");
   checkbox_object loadable("Load bitmap from file");
   checkbox_object use_pal("Use palette from file");

   // Add Objects to the dialog
   dialog the_dialog;
   the_dialog.add(object_0,165,160,305,160);
   the_dialog.add(object_1,215,285,80,20);
   the_dialog.add(object_2,350,285,80,20);
   the_dialog.add(object_4,176,256,110,10);
   the_dialog.add(object_9,272,200,192,13);
   the_dialog.add(object_11,288,256,176,20);
   the_dialog.add(bitmap_name,176,200,96,8);
   the_dialog.add(object_7,239,170,160,8);
   the_dialog.add(loadable,176,216,280,8);
   the_dialog.add(use_pal,176,232,268,8);
   the_dialog.center();

   // Execute the dialog and go further if it's OK'd
   if(the_dialog.popup()==&object_1)
      {
      // Check if we load the bitmap from a file
      if((loadable.selected()==true)
         &&(file_select("Load bitmap object from file",filename,"bmp;tga;lbm;pcx")!=0))
         {
         // Create the object
         helper_bitmap_object *object = new helper_bitmap_object(filename);
         object_description *desc = new object_description;
         if((object==NULL)||(desc==NULL)) degui_no_memory();

         // Set the variable name if the user entered one
         desc->set_type(bitmap_type);
         object->dataptr = desc;

         char temp[100];
         sprintf(temp,"\"%s\"",filename);

         // Exchange the slashes with backslashes or we will have problems
         // that the slashes look like escape sequences in the char* in
         // the constructor.  uuuggghhh... wierd...
         char *t;
         while((t=strchr(temp,'\\'))!=NULL)
            {
            t[0] = '/';
            t++;
            }
         desc->set_arguments(temp);

         if(strlen(object_11.get_text())>1) desc->set_name(object_11.get_text());

         add(object);

         // Possibly modify the system palette
         if(use_pal.selected()==true)
            {
            // Find the type of file and load it
            PALETTE pal;
            BITMAP *b = load_bitmap((char*)filename,pal);

            // Tell the system to use this palette
            set_palette(pal);
            reset_default_object_colors(false);

            destroy_bitmap(b);
            }
         }
      // Or just use an empty bitmap
      else
         {
         // Create the object
         helper_bitmap_object *object = new helper_bitmap_object(object_9.get_text());
         object_description *desc = new object_description;
         if((object==NULL)||(desc==NULL)) degui_no_memory();

         // Set the variable name if the user entered one
         desc->set_type(bitmap_type);
         object->dataptr = desc;

         desc->set_arguments(bit_name);

         if(strlen(object_11.get_text())>1) desc->set_name(object_11.get_text());

         add(object);
         }
      }
   return D_REDRAW;
   }
//----------------------------------------------------------------------------
int builder_dialog::add_text_object (void)
   {
   // Data definitions
   char *variable_name = "";
   char *initial_text = "Text";
   char *pointer_name = "pointer";

   // Object definitions
   panel_raised_object object_0;
   button_object object_1("&Ok",0,true);
   button_object object_2("&Cancel",0,true);
   text_object object_3("Create text object");
   text_object object_4("Variable name:");
   radio_button_object object_7("Object text:",0,false);
   radio_button_object object_8("Pointer text:",0,false);
   edittext_object object_9(initial_text,49);
   edittext_object object_10(pointer_name,49);
   edittext_object object_11(variable_name,49);

   object_7.select();  // Use the normal text as default

   // Add Objects to the dialog
   dialog the_dialog;
   the_dialog.add(object_0,165,160,305,160);
   the_dialog.add(object_1,215,285,80,20);
   the_dialog.add(object_2,350,285,80,20);
   the_dialog.add(object_3,240,170,140,10);
   the_dialog.add(object_4,175,255,110,10);
   the_dialog.add(object_7,175,205,100,8);
   the_dialog.add(object_8,175,230,100,8);
   the_dialog.add(object_9,290,205,155,10);
   the_dialog.add(object_10,295,230,150,10);
   the_dialog.add(object_11,290,255,175,10);
   the_dialog.center();

   // Get some data from the user
   helper_text_object *object;
   if(the_dialog.popup()==&object_1)
      {
      char temp[100];

      // Allocate the object description
      object_description *desc = new object_description;
      if(desc==NULL) return D_REDRAW;
      desc->set_type(text_type);

      // Create the object
      if(object_7.selected()==true)
         {
         object = new helper_text_object(object_9.get_text());
         sprintf(temp,"\"%s\"",object_9.get_text());
         }
      else
         {
         object = new helper_text_object(object_10.get_text());
         sprintf(temp,"%s",object_10.get_text());
         }
		if(object==NULL) return D_REDRAW;

      desc->set_arguments(temp);

      object->dataptr = desc;

      // Set the variable name if the user entered one
      if(strlen(object_11.get_text())>1) desc->set_name(object_11.get_text());

      // Add the object to the gui
      add(object);
      }
   return D_REDRAW;
   }
//----------------------------------------------------------------------------
int builder_dialog::add_centerd_text_object (void)
   {
   // Data definitions
   char *variable_name = "";
   char *initial_text = "cText";
   char *pointer_name = "ctextpointer";

   // Object definitions
   panel_raised_object object_0;
   button_object object_1("&Ok",0,true);
   button_object object_2("&Cancel",0,true);
   text_object object_3("Create centerd text object");
   text_object object_4("Variable name:");
   radio_button_object object_7("Object text:",0,false);
   radio_button_object object_8("Pointer text:",0,false);
   edittext_object object_9(initial_text,49);
   edittext_object object_10(pointer_name,49);
   edittext_object object_11(variable_name,49);

   object_7.select();  // Use the normal text as default

   // Add Objects to the dialog
   dialog the_dialog;
   the_dialog.add(object_0,165,160,305,160);
   the_dialog.add(object_1,215,285,80,20);
   the_dialog.add(object_2,350,285,80,20);
   the_dialog.add(object_3,220,170,140,10);
   the_dialog.add(object_4,175,255,110,10);
   the_dialog.add(object_7,175,205,100,8);
   the_dialog.add(object_8,175,230,100,8);
   the_dialog.add(object_9,290,205,155,10);
   the_dialog.add(object_10,295,230,150,10);
   the_dialog.add(object_11,290,255,175,10);
   the_dialog.center();

   // Get some data from the user
   helper_centerd_text_object *object;
   if(the_dialog.popup()==&object_1)
      {
      char temp[100];

      // Allocate the object description
      object_description *desc = new object_description;
      if(desc==NULL) return D_REDRAW;
      desc->set_type(centerd_text_type);

      // Create the object
      if(object_7.selected()==true)
         {
         object = new helper_centerd_text_object(object_9.get_text());
         sprintf(temp,"\"%s\"",object_9.get_text());
         }
      else
         {
         object = new helper_centerd_text_object(pointer_name);
         sprintf(temp,"%s",pointer_name);
         }
		if(object==NULL) return D_REDRAW;

      desc->set_arguments(temp);

      object->dataptr = desc;

      // Set the variable name if the user entered one
      if(strlen(variable_name)>1) desc->set_name(variable_name);

      // Add the object to the gui
      add(object);
      }
   return D_REDRAW;
   }
//----------------------------------------------------------------------------
int builder_dialog::add_edittext_object (void)
   {
   // Data definitions
   char *variable_name = "";
   char *pointer_name = "pointer";
   char *length_string = "10";

   // Object definitions
   panel_raised_object object_0;
   button_object object_1("&Ok",0,true);
   button_object object_2("&Cancel",0,true);
   text_object object_4("Variable name:");
   edittext_object object_9(pointer_name,14);
   edittext_object object_10(length_string,4);
   edittext_object object_11(variable_name,49);
   text_object object_7("Create editable text object");
   text_object object_8("Pointer name:");
   text_object object_6("Max length:");

   // Add Objects to the dialog
   dialog the_dialog;
   the_dialog.add(object_0,165,160,305,160);
   the_dialog.add(object_1,215,285,80,20);
   the_dialog.add(object_2,350,285,80,20);
   the_dialog.add(object_4,175,255,110,10);
   the_dialog.add(object_9,290,205,155,10);
   the_dialog.add(object_10,290,230,155,10);
   the_dialog.add(object_11,285,255,175,10);
   the_dialog.add(object_7,210,170,215,10);
   the_dialog.add(object_8,185,205,100,8);
   the_dialog.add(object_6,200,230,85,10);
   the_dialog.center();

   // Get some data from the user
   edittext_object *object;
   if(the_dialog.popup()==&object_1)
      {
      // Set a minimum maximum length
      int len = atoi(object_10.get_text());
      if(len<1) len = 1;

      // Allocate the object description
      object_description *desc = new object_description;
      if(desc==NULL) return D_REDRAW;

      // Create the object
      object = new edittext_object(NULL,len);
		if(object==NULL) return D_REDRAW;

      // Setup the description
      desc->set_type(edittext_type);

      char temp[100];
      sprintf(temp,"%s,%i",object_9.get_text(),len);
      desc->set_arguments(temp);

      object->dataptr = desc;

      // Set the variable name if the user entered one
      if(strlen(object_11.get_text())>1) desc->set_name(object_11.get_text());

      // Add the object to the gui
      add(object);
      }
   return D_REDRAW;
   }
//----------------------------------------------------------------------------
int builder_dialog::add_button_object (void)
   {
   char *name = "Button";

   // Object definitions
   panel_raised_object object_0;
   text_object object_1("Button Object Properties");
   text_object object_2("Button text:");
   edittext_object object_3(name,79);
   checkbox_object object_4("Close dialog on click");
   button_object object_5("&Ok",'o',true);
   button_object object_6("&Cancel",'c',true);

   // Add objects to the dialog
   dialog the_dialog;
   the_dialog.add(object_0,100,100,260,135);
   the_dialog.add(object_1,135,110,190,10);
   the_dialog.add(object_2,115,135,100,8);
   the_dialog.add(object_3,115,145,200,8);
   the_dialog.add(object_4,115,165,185,10);
   the_dialog.add(object_5,110,195,115,30);
   the_dialog.add(object_6,240,195,110,30);
   the_dialog.center();

   if(the_dialog.popup()==&object_5)
      {
      // Allocate the object description
      object_description *desc = new object_description;
      if(desc==NULL) return D_REDRAW;

      // Find what the user wants
      bool clos = false;
      if(object_4.selected()==true) clos = true;

      helper_button_object *object = new helper_button_object(object_3.get_text(),0,clos);
		if(object==NULL) return D_REDRAW;

      // Setup the description
      desc->set_type(button_type);

      char temp[100];
      sprintf(temp,"\"%s\",0,",object_3.get_text());
      if(clos==true) strcat(temp,"true");
      else strcat(temp,"false");

      desc->set_arguments(temp);

      object->dataptr = desc;

      add(object);
      }
   return D_REDRAW;
   }
//----------------------------------------------------------------------------
int builder_dialog::add_checkbox_object (void)
   {
   // Data definitions
   char *variable_name = "";
   char *initial_text = "Checkbox";

   // Object definitions
   panel_raised_object object_0;
   button_object object_1("&Ok",0,true);
   button_object object_2("&Cancel",0,true);
   text_object object_3("Create checkbox object");
   text_object object_4("Variable name:");
   text_object object_7("  Object text:");
   edittext_object object_9(initial_text,49);
   edittext_object object_11(variable_name,49);

   // Add Objects to the dialog
   dialog the_dialog;
   the_dialog.add(object_0,165,160,305,160);
   the_dialog.add(object_1,215,285,80,20);
   the_dialog.add(object_2,350,285,80,20);
   the_dialog.add(object_3,240,170,140,10);
   the_dialog.add(object_4,175,245,110,10);
   the_dialog.add(object_7,175,215,100,8);
   the_dialog.add(object_9,290,215,155,10);
   the_dialog.add(object_11,290,245,175,10);
   the_dialog.center();

   // Get some data from the user
   if(the_dialog.popup()==&object_1)
      {
      // Allocate the object description
      object_description *desc = new object_description;
      if(desc==NULL) return D_REDRAW;

      // Create the object
      helper_checkbox_object *object = new helper_checkbox_object(object_9.get_text());
		if(object==NULL) return D_REDRAW;

      // Setup the description
      desc->set_type(checkbox_type);

      char temp[100];
      sprintf(temp,"\"%s\"",object_9.get_text());
      desc->set_arguments(temp);

      object->dataptr = desc;

      // Set the variable name if the user entered one
      if(strlen(object_11.get_text())>1) desc->set_name(object_11.get_text());

      // Add the object to the gui
      add(object);
      }
   return D_REDRAW;
   }
//----------------------------------------------------------------------------
int builder_dialog::add_radio_button_object (void)
   {
   // Data definitions
   char *variable_name = "";
   char *initial_text = "Radio button";
   char *group_string = "0";

   // Object definitions
   panel_raised_object object_0;
   button_object object_1("&Ok",0,true);
   button_object object_2("&Cancel",0,true);
   text_object object_4("Variable name:");
   edittext_object object_9(initial_text,49);
   edittext_object object_10(group_string,4);
   edittext_object object_11(variable_name,49);
   text_object object_text("Object text:");
   text_object object_12("Create radio button object");
   text_object group("Group number:");
   checkbox_object squareness("Square");

   // Add Objects to the dialog
   dialog the_dialog;
   the_dialog.add(object_0,165,160,305,160);
   the_dialog.add(object_1,215,285,80,20);
   the_dialog.add(object_2,350,285,80,20);
   the_dialog.add(object_4,175,255,110,10);
   the_dialog.add(object_9,275,205,190,10);
   the_dialog.add(object_10,280,220,30,10);
   the_dialog.add(object_11,290,255,175,10);
   the_dialog.add(object_text,175,205,100,8);
   the_dialog.add(object_12,210,170,210,10);
   the_dialog.add(group,175,220,100,8);
   the_dialog.add(squareness,175,235,100,8);
   the_dialog.center();

   // Get some data from the user
   helper_radio_button_object *object;
   if(the_dialog.popup()==&object_1)
      {
      // Allocate the object description
      object_description *desc = new object_description;
      if(desc==NULL) return D_REDRAW;

      // Create the object
      object = new helper_radio_button_object(object_9.get_text(),atoi(object_10.get_text()),squareness.selected());
		if(object==NULL) return D_REDRAW;

      // Setup the description
      desc->set_type(radio_button_type);

      char temp[100];
      sprintf(temp,"\"%s\",%i,",object_9.get_text(),atoi(object_10.get_text()));
      if(squareness.selected()==true) strcat(temp,"true");
      else strcat(temp,"false");
      desc->set_arguments(temp);

      object->dataptr = desc;

      // Set the variable name if the user entered one
      if(strlen(object_11.get_text())>1) desc->set_name(object_11.get_text());

      // Add the object to the gui
      add(object);
      }
   return D_REDRAW;
   }
//----------------------------------------------------------------------------
int builder_dialog::add_list_object (void)
   {
   // Data definitions
   char *variable_name = "";
   char *procedure_name = "default";

   // Object definitions
   panel_raised_object object_0;
   button_object object_1("&Ok",0,true);
   button_object object_2("&Cancel",0,true);
   text_object object_4("Variable name:");
   edittext_object object_10(procedure_name,14);
   edittext_object object_11(variable_name,14);
   text_object object_13("List procedure must be of the type");
   text_object object_8("Procedure:");
   text_object object_9("char* (*proc)(int, int*);");
   text_object object_12("Create list object");

   // Add Objects to the dialog
   dialog the_dialog;
   the_dialog.add(object_0,165,160,305,160);
   the_dialog.add(object_1,215,285,80,20);
   the_dialog.add(object_2,350,285,80,20);
   the_dialog.add(object_4,175,255,110,10);
   the_dialog.add(object_10,255,230,210,15);
   the_dialog.add(object_11,290,255,175,10);
   the_dialog.add(object_13,175,200,100,8);
   the_dialog.add(object_8,175,230,80,10);
   the_dialog.add(object_9,175,210,100,8);
   the_dialog.add(object_12,240,170,100,8);
   the_dialog.center();

   // Check if the default procedure has been chosen
   char *proc_name = NULL;
   if(stricmp(object_10.get_text(),"default")!=0) proc_name = object_10.get_text();

   // Get some data from the user
   list_object *object;
   if(the_dialog.popup()==&object_1)
      {
      // Allocate the object description
      object_description *desc = new object_description;
      if(desc==NULL) return D_REDRAW;

      // Create the object
      object = new list_object;
		if(object==NULL) return D_REDRAW;

      // Setup the description
      desc->set_type(list_type);

      desc->set_arguments(proc_name);

      object->dataptr = desc;

      // Set the variable name if the user entered one
      if(strlen(object_11.get_text())>1) desc->set_name(object_11.get_text());

      // Add the object to the gui
      add(object);
      }
   return D_REDRAW;
   }
//----------------------------------------------------------------------------
int builder_dialog::add_textbox_object (void)
   {
   // Data definitions
   char *variable_name = "";
   char *pointer_name = "pointer";

   // Object definitions - DEGUI 1.0 beta
   panel_raised_object object_0;
   button_object object_1("&Ok",0,true);
   button_object object_2("&Cancel",0,true);
   text_object object_4("Variable name:");
   edittext_object object_11(variable_name,49);
   text_object title("Create textbox object");
   text_object object_6("Pointer:");
   edittext_object edit_pname(pointer_name,49);
   checkbox_object use_wrap("Word wrap");
   checkbox_object text_wrap("Text wrap");

   // Add Objects to the dialog
   dialog the_dialog;
   the_dialog.add(object_0,165,160,305,160);
   the_dialog.add(object_1,215,285,80,20);
   the_dialog.add(object_2,350,285,80,20);
   the_dialog.add(object_4,175,260,110,10);
   the_dialog.add(object_11,290,260,170,20);
   the_dialog.add(title,230,170,168,8);
   the_dialog.add(object_6,185,205,64,8);
   the_dialog.add(edit_pname,250,205,210,10);
   the_dialog.add(use_wrap,185,240,88,8);
   the_dialog.add(text_wrap,185,225,88,8);
   the_dialog.center();

   // Get some data from the user
   textbox_object *object;
   if(the_dialog.popup()==&object_1)
      {
      // Allocate the object description
      object_description *desc = new object_description;
      if(desc==NULL) return D_REDRAW;

      // Create the object
		object = new textbox_object(NULL,text_wrap.selected(),use_wrap.selected());

      // Setup the description
      desc->set_type(textbox_type);

      char temp[100];
      sprintf(temp,"%s,",edit_pname.get_text());
      if(text_wrap.selected()==true) strcat(temp,"true,");
      else strcat(temp,"false,");
      if(use_wrap.selected()==true) strcat(temp,"true");
      else strcat(temp,"false");
      desc->set_arguments(temp);

      object->dataptr = desc;

      // Set the variable name if the user entered one
      if(strlen(object_11.get_text())>1) desc->set_name(object_11.get_text());

      // Add the object to the gui
      add(object);
      }
   return D_REDRAW;
   }
//----------------------------------------------------------------------------
int builder_dialog::add_progress_bar_object (void)
   {
   // Create the object
   progress_bar_object *object = new progress_bar_object;
   object->progress(50);
   object_description *desc = new object_description;
	if((object==NULL)||(desc==NULL)) degui_no_memory();

   // Describe the object
   desc->set_type(progress_bar_type);

   // Link the object and the description
   object->dataptr = desc;

   add(object);
   return D_REDRAW;
   }
//----------------------------------------------------------------------------
int builder_dialog::add_window_object (void)
   {
   // Create the object
   window_object *object = new helper_window_object("Window",10);
   object_description *desc = new object_description;
	if((object==NULL)||(desc==NULL)) degui_no_memory();

   // Describe the object
   desc->set_type(window_type);

   // Link the object and the description
   object->dataptr = desc;

   add(object);
   return D_REDRAW;
   }
//----------------------------------------------------------------------------
int builder_dialog::add_user_object (void)
   {
   // Data definitions
   char *variable_name = "";
   char *type_name = "user_type";
   char *args = "";

   // Object definitions
   panel_raised_object object_0;
   button_object object_1("&Ok",0,true);
   button_object object_2("&Cancel",0,true);
   text_object object_4("Variable name:");
   edittext_object object_6(type_name,49);
   edittext_object object_11(variable_name,49);
   text_object object_8("Type:");
   text_object object_12("Arguments:");
   edittext_object object_14(args,99);
   text_object object_13("Create user object");

   // Add Objects to the dialog
   dialog the_dialog;
   the_dialog.add(object_0,165,160,305,160);
   the_dialog.add(object_1,215,285,80,20);
   the_dialog.add(object_2,350,285,80,20);
   the_dialog.add(object_4,175,260,110,10);
   the_dialog.add(object_6,215,200,240,15);
   the_dialog.add(object_11,290,260,175,20);
   the_dialog.add(object_8,175,200,40,10);
   the_dialog.add(object_12,175,230,100,8);
   the_dialog.add(object_14,175,240,290,15);
   the_dialog.add(object_13,245,170,100,8);
   the_dialog.center();

   // Get some data from the user
   helper_object *object;
   if(the_dialog.popup()==&object_1)
      {
      // Allocate the object description
      object_description *desc = new object_description;
      if(desc==NULL) return D_REDRAW;

      // Create the text for the user object
      char temp[100];
      sprintf(temp,"%s",object_6.get_text());

      if(strlen(object_14.get_text())>1)
         {
         strcat(temp,"(");
         strcat(temp,object_14.get_text());
         strcat(temp,")");
         }
      strcat(temp,";");

      // Create the object
      object = new helper_object(temp);
		if(object==NULL) return D_REDRAW;

      // Setup the description
      desc->set_type(object_6.get_text());

      desc->set_arguments(object_14.get_text());

      object->dataptr = desc;

      // Set the variable name if the user entered one
      if(strlen(object_11.get_text())>1) desc->set_name(object_11.get_text());

      // Add the object to the gui
      add(object);
      }
   return D_REDRAW;
   }
//----------------------------------------------------------------------------
int builder_dialog::add_hidden_list_object (void)
   {
   // Create the object
   hidden_list_object *object = new hidden_list_object;
   object_description *desc = new object_description;
	if((object==NULL)||(desc==NULL)) degui_no_memory();

   // Describe the object
   desc->set_type(hidden_list_type);

   // Link the object and the description
   object->dataptr = desc;

   add(object);
   return D_REDRAW;
   }
//----------------------------------------------------------------------------

