indexing
    description    : "Allegro con Eiffel: read only packed file objects"
    status         : "Initial development"
    author         : "Peter Monks (pmonks@iname.com)"
    allegro_author : "Shawn Hargreaves (shawn@talula.demon.co.uk)"
    names          : pack_file_read
    date_started   : "8th February, 1997"
    version        : "0.1 beta"
    platforms      : "MS-DOS"
    dependencies   : "Allegro v2.2, DJGPP v2.01"


class PACK_FILE_READ


inherit
    PACK_FILE
    end  -- inherit PACK_FILE


creation { ANY }
    connect_to,
    make


------------------------------------------------------ Creation features
feature { ANY }

    connect_to(new_path : STRING) is
    -- Connect this object to the given file
    do
        make
        input_stream := fopen(new_path, mode)

        if input_stream.is_not_void then
            path := new_path
        end
    end


    disconnect is
    -- Disconnect this object from the current file
    local
        err : INTEGER
    do
        err := fclose(input_stream)
        path := Void
    end


    make is
    -- Make a PACK_FILE_READ object without connecting it to anything
    do
        mode := "rp"
    end


------------------------------------------------------ Pack file read features
feature { ANY }

    -- Last integer read using `read_integer'.
    last_integer : INTEGER


    -- Last real read with `read_real'.
    last_real : REAL


    -- Last double read with `read_double'.
    last_double : DOUBLE


    -- Last character read with `read_character'.
    last_character : CHARACTER is
    do
        Result := last_character_memory
    end  -- feature last_character


    last_string : STRING is
    -- Last STRING read with `read_line', `read_word' or `newline'.
    --
    -- NOTE: it is alway the same STRING.
    once
        !!Result.make(256);
    end  -- feature last_string


    read_line is
    -- Read a complete line ended by '%N' or `end_of_input'.
    -- Make the result available in `last_string'.
    -- Character '%N' is not added in `last_string'.
    --
    -- NOTE: the result is available in `last_string' without any
    --       memory allocation.
    require
        not end_of_input
    do
        read_line_in(last_string)
    end  -- feature read_line


    read_line_in(str : STRING) is
    -- Same jobs as `read_line' but storage is directly done in `str'.
    --
    require
        not end_of_input
    do
        from
            str.clear
            read_character
        until
            end_of_input or else
            last_character = '%N'
        loop
            str.extend(last_character)
            read_character
        end  -- from/loop
    end  -- feature read_line_in


    read_word is
    -- Read a word using `is_separator' of class CHARACTER.
    -- Result is available in `last_string' (no allocation
    -- of memory).
    -- Heading separators are automatically skipped.
    -- Trailing separators are not skipped (`last_character' is
    -- left on the first one).
    -- If `end_of_input' is encountered, Result can be the
    -- empty string.
    require
        not end_of_input
    do
        skip_separators

        from
            last_string.clear
        until
            end_of_input or else
            last_character.is_separator
        loop
            last_string.extend(last_character)
            read_character
        end  -- from/loop
    end  -- feature read_word


    read_word_using(separators: STRING) is
    -- Same jobs as `read_word' using `separators'.
    require
        not end_of_input
        separators /= void
    do
        -- Implemented by : Lars Brueckner
        -- (larsbruk@rbg.informatik.th-darmstadt.de)
        skip_separators_using(separators)

        from
            last_string.clear
        until
            end_of_input or else
            separators.has(last_character)
        loop
            last_string.extend(last_character)
            read_character
        end  -- from/loop
    end  -- feature read_word_using


    read_integer is
    -- Read an integer according to the Eiffel syntax.
    -- Make result available in `last_integer'.
    -- Heading separators (`is_separator' of CHARACTER)
    -- are automatically skipped.
    -- Trailing sseparators are not skipped (`last_character'
    -- is after the last digit of the number).
    local
        state : INTEGER
        sign : BOOLEAN
        -- state = 0 : waiting sign or first digit.
        -- state = 1 : sign read, waiting first digit.
        -- state = 2 : in the number.
        -- state = 3 : end state.
        -- state = 4 : error state.
    do
        from
        until
            state > 2
        loop
            read_character

            inspect state
            when 0 then
                if last_character.is_separator then
                    -- Do nothing
                elseif last_character.is_digit then
                    last_integer := last_character.value
                    state := 2
                elseif last_character = '-' then
                    sign := true
                    state := 1
                elseif last_character = '+' then
                    state := 1
                else
                    state := 4
                end  -- if

            when 1 then
                if last_character.is_separator then
                    -- Do nothing
                elseif last_character.is_digit then
                    last_integer := last_character.value
                    state := 2
                else
                    state := 4
                end  -- if

            else -- state = 2
                if last_character.is_digit then
                    last_integer := (last_integer * 10) + last_character.value
                else
                    state := 3
                end  -- if
            end  -- inspect
        end  -- from/loop

        debug
            if state = 4 then
                std_error.put_string("Error in PACK_FILE_READ.read_integer.%N")
                crash
            end  -- if
        end  -- debug

        if sign then
            last_integer := - last_integer
        end  -- if
    end  -- feature read_integer


    read_real is
    -- Read a REAL and make the result available in `last_real'
    -- and in `last_double'.
    -- The integral part is available in `last_integer'.
    do
        read_double
        last_real := last_double.to_real
    end  -- feature read_real


    read_double is
    -- Read a DOUBLE and make the result available in
    -- `last_double'.
    local
        state : INTEGER
        sign : BOOLEAN
        ip, i : INTEGER
        -- state = 0 : waiting sign or first digit.
        -- state = 1 : sign read, waiting first digit.
        -- state = 2 : in the integral part.
        -- state = 3 : in the fractional part.
        -- state = 4 : end state.
        -- state = 5 : error state.
    do
        from
        until
            state >= 4
        loop
            read_character

            inspect state
            when 0 then
                if last_character.is_separator then
                    -- Do nothing
                elseif last_character.is_digit then
                    ip := last_character.value
                    state := 2
                elseif last_character = '-' then
                    sign := true
                    state := 1
                elseif last_character = '+' then
                    state := 1
                elseif last_character = '.' then
                    tmp_read_double.clear
                    state := 3
                else
                    state := 5
                end  -- if

            when 1 then
                if last_character.is_separator then
                    -- Do nothing
                elseif last_character.is_digit then
                    ip := last_character.value
                    state := 2
                else
                    state := 5
                end  -- if

            when 2 then
                if last_character.is_digit then
                    ip := (ip * 10) + last_character.value
                elseif last_character = '.' then
                    tmp_read_double.clear
                    state := 3
                else
                    state := 4
                end  -- if

            else -- state 3
                if last_character.is_digit then
                    tmp_read_double.extend(last_character)
                else
                    state := 4
                end  -- if
            end  -- inspect
        end

        debug
            if state = 5 then
                std_error.put_string("Error in PACK_FILE_READ.read_double.%N")
                crash
            end  -- if
        end  -- debug

        from
            last_double := 0
            i := tmp_read_double.count
        until
            i = 0
        loop
            last_double := (last_double + tmp_read_double.item(i).value) / 10
            i := i - 1
        end

        last_double := last_double + ip

        if sign then
            last_double := - last_double
        end  -- if
    end  -- feature read_double


    unread_character is
    -- Un read the last character read.
    local
        err : INTEGER
    do
        err := ungetc(last_character,input_stream)
    end  -- feature unread_character


    read_character is
    -- Read a character and assign it to `last_character'.
    require
        not end_of_input
    do
        last_character_memory := fgetc(input_stream)
    end  -- feature read_character


    read_tail_in(str: STRING) is
    -- Read all remaining character of the file in `str'.
    do
        from
            if not end_of_input then
                read_character
            end  -- if
        until
            end_of_input
        loop
            str.extend(last_character)
            read_character
        end
    ensure
        end_of_input
    end  -- feature read_tail


    newline is
    -- Consume input until newline is found.
    -- Corresponding STRING is stored in `last_string'.
    -- Then consume newline character.
    do
        from
            last_string.clear
        until
            end_of_input or else last_character = '%N'
        loop
            read_character
            last_string.extend(last_character)
        end  -- from/loop

        if not end_of_input then
            read_character
        end  -- if
    end  -- feature newline


    end_of_input : BOOLEAN is
    -- Has end-of-input been reached ?
    do
        Result := feof(input_stream)
    end  -- feature end_of_input


------------------------------------------------------ Skipping features
feature { ANY }

    skip_separators is
    -- Stop doing `read_character' as soon as `end_of_file' is reached
    -- or as soon as `last_character' is not `is_separator'.
    -- When first character is already not `is_separator' nothing
    -- is done.
    do
        from
        until
            end_of_input or else
            not last_character.is_separator
        loop
            read_character
        end  -- from/loop
    end  -- feature skip_separators


    skip_separators_using(separators : STRING) is
    -- Same job as `skip_separators' using `separators'.
    require
        separators /= void
    do
        -- Implemented by : Lars Brueckner
        -- (larsbruk@rbg.informatik.th-darmstadt.de)
        from
        until
            end_of_input or else
            not separators.has(last_character)
        loop
            read_character
        end  -- from/loop
    end  -- feature skip_separators_using


------------------------------------------------------ FILE_TOOLS features
feature { FILE_TOOLS }

    same_as(other: like Current): BOOLEAN is
    require
        is_connected
        other.is_connected
    local
        is1, is2 : like input_stream
    do
        from
            is1 := input_stream
            is2 := other.input_stream
            Result := true
        until
            not Result or else feof(is1)
        loop
            Result := fgetc(is1) = fgetc(is2)
        end

        disconnect
        other.disconnect
    ensure
        not is_connected
        not other.is_connected
    end  -- feature same_as


------------------------------------------------------ PACK_FILE_READ features
feature { PACK_FILE_READ }

   input_stream : POINTER

   last_character_memory : CHARACTER


------------------------------------------------------ Internal features
feature { NONE }

   tmp_read_double : STRING is
   once
       !!Result.make(12)
   end


end  -- class PACK_FILE_READ
