/*
 * PNP.CPP - Contains Plug and Play functions.
 * Copyright (C) 1998, 1999 Prashant TR
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * See the file COPYING.TR for more details.
*/

// ID for this file.
#define _PNP_CC_

#include "pnp.h"

FILE *fp;
char cmdline[20];
int errflag = 0;
unsigned char buf1[512], buf2[256];
unsigned char pnpname[256], flag;
unsigned short count, isaport = 0;
PNP *header;

unsigned _stklen = 0x2000;

// Write any string to the file and check for successfulness.
void writestring(const char *string)
{
	if (fprintf(fp, "%s", string) == EOF) {
		errflag = 2;
		checkerrors();
	}
}

PNP *get_pnp_entry_point()
{
	unsigned short segment, offset, f;
	char flag;

	// Search for $PNP.
	segment = 0xf000;
	offset = flag = 0;

	do {
		if (!strncmp((char *)MK_FP(segment, offset), "$PnP", 4))
		flag++;
		else offset += 0x10;
	} while ((offset) && (!flag));

	// Check if actually PnP.
	if (flag) {
		flag = 0;
		for(f = 0; f <= 0x20; f++)
		flag += *(char *)MK_FP(segment, offset + f);
		flag = (!flag) ? 1 : 0;
	}
	return ((offset) ? (PNP *)MK_FP(segment, offset) : NULL);
}

void writepnpinfo()
{
	char output[256];
	unsigned short f, count;
	int irq, dma, portid, mask, masknum, card = 1;

	// Get the starting specifier type.
	count = 12;
	flag = irq = dma = 0;
	flag = buf1[count];	// The specifier.

	// Process PnP Information.
	while (((unsigned char)flag >> 3) != 0xf)
	{
		flag = buf1[count];	// The specifier.
		if ((flag & 0x78) == 0x10) {
			sprintf(output, "\n\tDevice Identification         : "
				"%c%c%c%02X%02X\n",
				((*(unsigned short *)&buf1[count + 1] >> 2) & 0x1f)
					+ 0x40,
				(char)((unsigned long)(((((unsigned long)
					*(unsigned short *)&buf1[count + 1]) >> 13)
					& 7)
					| ((buf1[count + 1] & 3) << 3))
					+ 0x40),
				(buf1[count + 2] & 0x1f)
					+ 0x40,
				buf1[count + 3],
				buf1[count + 4]);
			writestring(output);
			count += 6;
			continue;
		}

		switch (((unsigned char)flag >> 3) - 2) {
			case 0:
			case 1:
			case 4:
			case 5:
			case 8:  // Probably Memory - 8.
				break;
			// IRQ Setting.
			case 2:
				mask = *(unsigned short *)
					&buf1[count + 1];
				masknum = 0;
				do {
					if ((mask >> masknum) & 1) {
						irq++;
						sprintf(output,
							"\tIRQ "
							"Setting"
							" (%03u)"
							"       "
							"      : "
							"%d\n",
							irq,
							masknum);
						writestring(output);
						break;
					}
					masknum++;
				} while (masknum <= 15);
				break;

			// DMA Channel.
			case 3:
				mask = buf1[count + 1];
				masknum = 0;
				do {
					if ((mask >> masknum) & 1) {
						dma++;
						sprintf(output,
							"\tDMA "
							"Channel"
							" (%03u)"
							"       "
							"      : "
							"%d\n",
							dma,
							masknum);
						writestring(output);
						break;
					}
					masknum++;
				} while (masknum <= 8);
				break;

			// I/O Ports.
			case 6:
				sprintf(output, "\tI/O Range       "
					"              : "
					"%04X - %04X\n",
					*(int *)&buf1[count + 4],
					*(int *)&buf1[count + 4] +
					buf1[count + 7] - 1);
				writestring(output);
				break;

			// I/O Ports.
			case 7:
				sprintf(output, "\tI/O Range       "
					"              : "
					"%04X - %04X\n",
					*(int *)&buf1[count + 1],
					*(int *)&buf1[count + 1] +
					buf1[count + 3] - 1);
				writestring(output);
				break;

			default:
				if (!(flag & 0x80)) break;

				// Check if memory type specifier.
				if ((flag & 0xf) == 6) {
					sprintf(output, "\tMemory Range    "
						"              : "
						"%08lX - %08lX\n",
						*(long *)&buf1[count + 4],
						*(unsigned long *)&
							buf1[count + 4] +
						*(unsigned long *)&
							buf1[count + 8]
						- 1);
					writestring(output);
				}

				// Check if Card name specifier.
				if ((flag & 0xf) == 2) {
					char x = buf1[count +
							*(unsigned short *)&
							buf1[count + 1] + 3];

					buf1[count +
						*(unsigned short *)&
						buf1[count + 1] + 3] = 0;

					sprintf(output, (card == 1) ?
						"\tPlug and Play Ca"
						"rd            : "
						"%s\n" :
						"\tPlug and Play De"
						"vice          : "
						"%s\n",
						&buf1[count + 3]);
					writestring(output);

					buf1[count +
						*(unsigned short *)&
						buf1[count + 1] + 3] = x;

					if (card) card = 0;

				}

				// Modify counter and type;
				count += *(unsigned short *)
					&buf1[count + 1] + 2;
				flag &= 0x80;
				break;
		}

		// Increment pointer.
		count += (flag & 7) + 1;
	}
}

void pnp_wait(unsigned char data)
{
	asm {
		MOV	CX, 0C8H
		MOV	AL, data
	}
pnp_wait_loop:
	asm {
		OUT	0EDH, AL
		LOOP	pnp_wait_loop
	}
}

void init_isa_pnp(unsigned char card)
{
	unsigned char data = 0x6a, newdata;
	int f;
	outportb(0x279, 0);
	pnp_wait(0);
	outportb(0x279, 0);
	pnp_wait(0);
	outportb(0x279, data);
	pnp_wait(data);

	for(f = 1; f < 0x20; f++)
	{
		newdata = data >> 1;
		newdata = data ^ newdata;
		newdata = (newdata << 7);
		data = newdata | (data >> 1);
		outportb(0x279, data);
		pnp_wait(data);
	}
	outportb(0x279, 3);
	outportb(0xa79, card);
}

void shut_isa_pnp()
{
	outportb(0x279, 2);
	outportb(0xa79, 2);
}

unsigned char readisapnp(unsigned char func)
{
	unsigned char result;
	do {
		pnp_wait(result);
		outportb(0x279, 5);
		pnp_wait(result);
		result = inportb(isaport);
	} while (!(result & 1));
	outportb(0x279, func);
	pnp_wait(result);
	result = inportb(isaport);
	return (result);
}

void storepnpinfo(int *index_value)
{
	int f;
	int index = *index_value;
	unsigned char flag;

	// Get the starting specifier type.
	flag = buf1[index - 1];		// The specifier.

	// Process PnP Information.
	if (flag & 0x80) {
		f = readisapnp(4) + 256 * readisapnp(4);
		buf1[index++] = f & 0xff;
		buf1[index++] = f >> 8;
		while (f--) buf1[index++] = readisapnp(4);
		*index_value = index;
		return;
	}

	// Read data.
	for (f = flag & 7; f > 0; f--)
	buf1[index++] = readisapnp(4);
	*index_value = index;
}

void get_isa_pnp()
{
	unsigned char result;
	unsigned index = 0, f;
	for(f = 0; f < 9; f++) buf1[index++] = readisapnp(4);
	index = 12;
	if ((buf1[index++] = readisapnp(4)) != 0xff)
		storepnpinfo((int *)&index);
	else return;
	while ((buf1[index++] = readisapnp(4)) != 0x79)
	storepnpinfo((int *)&index);

/*	// For PCI.
	index = 0;
	while (index <= 0x1f)
	{
		int g;
		shut_isa_pnp();
		init_isa_pnp(1);
		outportb(0x279, 7);
		outportb(0xa79, index);

		for(f = 0; f < 16; f++) readisapnp(0x60 + f);
		for(f = 0; f < 4; f++) readisapnp(0x70 + f);
		for(f = 0; f < 2; f++) readisapnp(0x74 + f);
		for(f = 0; f < 8; f++) readisapnp(0x40 + f);
		for(f = 0; f < 4; f++)
		{
			for(g = 0; g < 4; g++)
			readisapnp(0x76 + f * 4 + g);
		}
		index++;
	}*/
}

int sysinfo()
{
	char output[256], classname[80], deviceid[10];
	int card_no = 1;
	FILE *pfp;

	// Create output file.
	if ((fp = fopen("pnp.txt", "w")) == NULL) {
		errflag = 1;
		checkerrors();
	}

	writestring("\nPLUG AND PLAY INFORMATION (PNP):\n\n");

	// Get $PNP address.
	header = get_pnp_entry_point();

	sprintf(output, "\tPlug and Play BIOS Present    : %s\n",
			(header != NULL) ? "Yes" : "No");
	writestring(output);

	if (header == NULL) {
		fclose(fp);
		return 0;
	}

	// Get the ISA data read port.
	errflag = header -> pnp_entry_point(0x40,
					buf1,
					header -> bios_selector);
	if ((!errflag) && (buf1[1] != 0xff))
			isaport = *(unsigned short *)&buf1[2];

	// Write PnP Information.
	sprintf(output, "\tPnP Header Pointer            : %lp\n",
			header);
	writestring(output);
	sprintf(output, "\tPnP BIOS Version              : %X.%X\n",
			header -> version >> 4,
			header -> version & 0xf);
	writestring(output);
	sprintf(output, "\tReal Mode Entry Point         : %lph\n",
			header -> pnp_entry_point);
	writestring(output);
	sprintf(output, "\tProtected Mode Entry Point    : %08lXh\n",
			((unsigned long)FP_SEG(header -> pnp_entry_point)
				<< 4) +
				header -> protected_mode_entry_offset);
	writestring(output);
	if (isaport)
		sprintf(output, "\tISA Read Data Port            : %04Xh\n",
			isaport);//header -> isa_read_port);
	writestring(output);
	sprintf(output, "\tBIOS Data Selector            : %04Xh\n",
			header -> bios_selector);
	writestring(output);

	// Get the number of OnBoard PnP Devices.
	errflag = header -> pnp_entry_point(0,
					buf2,
					buf1,
					header -> bios_selector);
	if (errflag) checkerrors();

	count = *(int *)buf2;
	if (count == 0xffff) {
		sprintf(output, "\tNo Plug and Play Devices found\n");
		writestring(output);
		fclose(fp);
		return 0;
	}
	memset(buf1, 0, 100);
	memset(buf2, 0, 100);

	// Get Plug and Play Information.
	do {
		if (errflag = header -> pnp_entry_point(1, buf2, buf1,
						1,
						header -> bios_selector))
						checkerrors();

		// Analyze information.
		writestring("\nPNP DEVICE INFORMATION:\n\n");

		sprintf(output, "\tDevice Identification         : "
			"%c%c%c%02X%02X\n",
			((*(unsigned short *)&buf1[3] >> 2) & 0x1f)
				+ 0x40,
			(char)((unsigned long)(((((unsigned long)
				*(unsigned short *)&buf1[3]) >> 13)
				& 7)
				| ((buf1[3] & 3) << 3))
				+ 0x40),
			(buf1[4] & 0x1f)
				+ 0x40,
			buf1[5],
			buf1[6]);
		writestring(output);

		sprintf(deviceid, "%c%c%c%02X%02X",
				((*(unsigned short *)&buf1[3] >> 2) & 0x1f)
					+ 0x40,
				(char)((unsigned long)(((((unsigned long)
					*(unsigned short *)&buf1[3]) >> 13)
					& 7)
					| ((buf1[3] & 3) << 3))
					+ 0x40),
				(buf1[4] & 0x1f)
					+ 0x40,
				buf1[5],
				buf1[6]);

		pfp = pinseek("sysinfo.pin", "pnpids.txt");
		strcpy(output, "");
		if (pfp != NULL) {
			do {
				if (fgets(output, 80, pfp) == NULL) break;

				while ((output[strlen(output) - 1] == '\n') ||
					(output[strlen(output) - 1] == '\r'))
					output[strlen(output) - 1] = 0;

				if (!strcmp(output, "")) continue;
				if (output[0] == ';')
					strcpy(classname, &output[1]);
				else {
					strcpy(pnpname, output);
					if (!strncmp(deviceid, output, 7)) {
						// Write out the information.
						sprintf(output,
							"\tDevice "
							"Class         "
							"         : "
							"%s\n",
							classname);
						writestring(output);
						sprintf(output,
							"\tDevice "
							"Name          "
							"         : "
							"%s\n",
							&pnpname[8]);
						writestring(output);
						break;
					}
				}
			} while (1);
			fclose(pfp);
		}

		writepnpinfo();

	} while (buf2[0] != 0xff);

	if (!isaport) {
		fclose(fp);
		return 0;
	}

	while (card_no <= 0xff)
	{
		init_isa_pnp(card_no);
		get_isa_pnp();
		shut_isa_pnp();
		if (buf1[12] == 0xff);
//			writestring("\tUnable to get ISAPNP information.\n");
		else {
			writestring("\nISAPNP DEVICE INFORMATION:\n\n");
			sprintf(output, "\tPlug and Play Card Number     : "
				"Card %02Xh\n", card_no);
			writestring(output);

			sprintf(output, "\tPlug and Play Card ID         : "
				"%c%c%c%02X%02X\n",
				((*(unsigned short *)&buf1[0] >> 2) & 0x1f)
					+ 0x40,
				(char)((unsigned long)(((((unsigned long)
					*(unsigned short *)&buf1[0]) >> 13)
					& 7)
					| ((buf1[0] & 3) << 3))
					+ 0x40),
				(buf1[1] & 0x1f)
					+ 0x40,
				buf1[2],
				buf1[3]);
			writestring(output);
			writepnpinfo();
		}
		card_no++;
	}

	fclose(fp);
	return 0;
}

int pnp_version()
{
	PNP *header;

	// Create output file.
	if ((fp = fopen("temp.$$$", "w")) == NULL) {
		errflag = 1;
		checkerrors();
	}

	// Get $PNP address.
	header = get_pnp_entry_point();

	if (header == NULL) {
		fprintf(fp, "%c", 0, 0);
		fprintf(fp, "%lp", NULL);
	}
	else {
		fprintf(fp, "%c", header -> version);
		fprintf(fp, "%lp", header -> pnp_entry_point);
	}

	fclose(fp);
	return 0;
}

void open_stderr()
{
	fclose(stdout);
	fclose(&_streams[2]);
	if (fopen("nul", "wb") == NULL) exit(0x7f);
	if (fopen("nul", "wb") == NULL) exit(0x7f);
	if ((stderr = fopen("errors.$$$", "ab")) == NULL) exit(0x7f);
}

void get_cmdline()
{
	if ((fp = fopen("cmdline.$$$", "rb")) == NULL) exit (0x7f);

	if (fscanf(fp, "%s", cmdline) != 1) {
		fclose(fp);
		exit (0x7f);
	}

	fclose(fp);
	unlink("cmdline.$$$");
}

#pragma argsused

// The main function.
int main(int argc, char **argv)
{
	open_stderr();
	get_cmdline();

	if (!strcmp(cmdline, "sysinfo")) return (sysinfo());
	if (!strcmp(cmdline, "pnp_version")) return (pnp_version());

	return 0;
}
