/*    EPZip - the simple free distribution preparing tool
**    Copyright (C) 1997  Esa Peuha

**    This program is free software; you can redistribute it and/or modify
**    it under the terms of the GNU General Public License as published by
**    the Free Software Foundation; either version 2 of the License, or
**    (at your option) any later version.

**    This program is distributed in the hope that it will be useful,
**    but WITHOUT ANY WARRANTY; without even the implied warranty of
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**    GNU General Public License for more details.

**    You should have received a copy of the GNU General Public License
**    along with this program; if not, write to the Free Software
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <fcntl.h>
#include <dos.h>
#include <io.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "epzip.h"

extern int _fmode;

static inline void put_short(unsigned short, FILE *);
static inline void put_int(unsigned int, FILE *);

unsigned short files;
FILE *centralfile;

void begin_process(FILE *outfile)
{
  files = 0;

  if(outfile == NULL)
    {
      fprintf(stderr, "epzip: internal error\n");
      exit(1);
    }

  _fmode = O_BINARY;

  centralfile = tmpfile();
  if(centralfile == NULL)
    {
      perror("epzip: can't open temporary file");
      exit(1);
    }

  initialize_crc();
}

void process_file(FILE *outfile, char *filename)
{
  unsigned short method, name_length;
  unsigned int crc, size, length, offset;

  /* awful hack to make things simple! */
  union td { unsigned int i; struct ftime s; } time_date;

  FILE *infile;
  char *uncompressed, *compressed;

  ++files;

  offset = ftell(outfile);
  name_length = strlen(filename);

  infile = fopen(filename, "rb");
  if(infile == NULL)
    {
      fprintf(stderr, "epzip: can't open input file \"%s\"", filename);
      perror("");
      exit(1);
    }

  length = filelength(fileno(infile));
  getftime(fileno(infile), &time_date.s);

  uncompressed = malloc(length);
  if(uncompressed == NULL)
    {
      fprintf(stderr, "epzip: can't allocate memory\n");
      exit(1);
    }

  if(fread(uncompressed, 1, length, infile) != length)
    {
      fprintf(stderr, "epzip: error reading file \"%s\"", filename);
      perror("");
      exit(1);
    }
  fclose(infile);

  crc = get_crc(uncompressed, length);

  compressed = NULL;
  size = length;

  /* deflate to see whether to deflate or store */
  /* not elegant but works! */
  deflate(uncompressed, &compressed, length, &size);

  if(size < length)
    {
      /* deflate */
      free(uncompressed);
      method = 8;
    }
  else
    {
      /* store */
      if(compressed != NULL)
	free(compressed);
      compressed = uncompressed;
      size = length;
      method = 0;
    }

  /* write central header */

  /* signature */
  fputs("PK\01\02", centralfile);

  /* made by version 2.0 on MS-DOS */
  put_short(20, centralfile);

  /* version needed to extract */
  put_short(method ? 20 : 10, centralfile);

  /* flags */
  put_short(method ? 2 : 0, centralfile);

  /* compresion method */
  put_short(method, centralfile);

  /* time & date */
  put_int(time_date.i, centralfile);

  /* crc */
  put_int(crc, centralfile);

  /* size */
  put_int(size, centralfile);

  /* length */
  put_int(length, centralfile);

  /* filename length */
  put_short(name_length, centralfile);

  /* extra field length */
  put_short(0, centralfile);

  /* file comment length */
  put_short(0, centralfile);

  /* number of disk */
  put_short(0, centralfile);

  /* internal attributes */
  /* treat all files as binary! */
  put_short(0, centralfile);

  /* external attributes */
  /* no reason to read the real attributes from file system! */
  put_int(0, centralfile);

  /* offset of local header */
  put_int(offset, centralfile);

  /* filename */
  fwrite(filename, 1, name_length, centralfile);

  /* write local header */

  /* signature */
  fputs("PK\03\04", outfile);

  /* version needed to extract */
  put_short(method ? 20 : 10, outfile);

  /* flags */
  put_short(method ? 2 : 0, outfile);

  /* compresion method */
  put_short(method, outfile);

  /* time & date */
  put_int(time_date.i, outfile);

  /* crc */
  put_int(crc, outfile);

  /* size */
  put_int(size, outfile);

  /* length */
  put_int(length, outfile);

  /* filename length */
  put_short(name_length, outfile);

  /* extra field length */
  put_short(0, outfile);

  /* filename */
  fwrite(filename, 1, name_length, outfile);

  /* write compressed file */
  fwrite(compressed, 1, size, outfile);

  free(compressed);
}

void end_process(FILE *outfile)
{
  char *buffer;

  unsigned int offset = ftell(outfile), size = ftell(centralfile);

  /* copy central directory from temporary to output */

  buffer = malloc(size);
  if(buffer == NULL)
    {
      fprintf(stderr, "epzip: can't allocate memory\n");
      exit(1);
    }
  if(fseek(centralfile, 0, SEEK_SET))
    {
      perror("epzip: error seeking temporary file");
      exit(1);
    }
  if(fread(buffer, 1, size, centralfile) != size)
    {
      perror("epzip: can't read from temporary file");
      exit(1);
    }
  if(fwrite(buffer, 1, size, outfile) != size)
    {
      perror("epzip: can't write to output file");
      exit(1);
    }

  free(buffer);
  fclose(centralfile);

  /* write end header */

  /* signature */
  fputs("PK\05\06", outfile);

  /* number of this disk */
  put_short(0, outfile);

  /* number of starting disk */
  put_short(0, outfile);

  /* files on this disk */
  put_short(files, outfile);

  /* files total */
  put_short(files, outfile);

  /* size of central directory */
  put_int(size, outfile);

  /* offset of central directory */
  put_int(offset, outfile);

  /* size of zip file comment */
  put_short(0, outfile);


  /* free CRC buffer */
  terminate_crc();
}

static inline void put_short(unsigned short s, FILE *f)
{
  fputc(s & 0xff, f);
  fputc(s >> 8, f);
}

static inline void put_int(unsigned int i, FILE *f)
{
  put_short(i & 0xffff, f);
  put_short(i >> 16, f);
}
