/*    EPZip - the simple free distribution preparing tool
**    Copyright (C) 1997  Esa Peuha

**    This program is free software; you can redistribute it and/or modify
**    it under the terms of the GNU General Public License as published by
**    the Free Software Foundation; either version 2 of the License, or
**    (at your option) any later version.

**    This program is distributed in the hope that it will be useful,
**    but WITHOUT ANY WARRANTY; without even the implied warranty of
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**    GNU General Public License for more details.

**    You should have received a copy of the GNU General Public License
**    along with this program; if not, write to the Free Software
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdio.h>
#include <stdlib.h>
#include "epzip.h"

static unsigned short init_blocks(struct file_el *, unsigned int **);
static void end_blocks(unsigned int **);
static void block_begin(unsigned short, unsigned short);
static void block_end();
static void block_put_literal(unsigned short);
static void block_put_dist_literal(unsigned short);
static void block_put_length(unsigned short);
static void block_put_distance(unsigned short);
static unsigned short reverse_bits(unsigned short, unsigned char);

FILE *codify(struct file_el *substfile)
{
  FILE *codefile = tmpfile();
  unsigned int *block_table;
  unsigned short block, blocks;

  blocks = init_blocks(substfile, &block_table);

  fseek(substfile->fp, 0, SEEK_SET);

  bwrite_begin(codefile);
  bread_begin(substfile->fp);
  for(block = 0; block < blocks; block++)
    {
      unsigned short n;

      block_begin(block, blocks);
      for(n = 0; n < block_table[block]; n++)
	{
	  if(bread(1))
	    {
	      block_put_length(bread(8));
	      block_put_distance(bread(15));
	    }
	  else
	    block_put_literal(bread(8));
	}
      block_end();
    }
  bread_end();
  bwrite_end();
  end_blocks(&block_table);

  fclose(substfile->fp);
  return codefile;
}



/* these functions should really scan the input stream and make a dynamical
** Huffman code for each block, but this shall do for now */

struct huffman_code
{
  unsigned short bits, code;
};

struct huffman_code *lit_table;
struct huffman_code *dist_table;

static unsigned short init_blocks(struct file_el *file, unsigned int **table)
{
  unsigned short value, i;
  unsigned int element = 0, j;

  j = ftell(file->fp);
  if((signed) j == -1)
    {
      perror("epzip: substfile error");
      exit(1);
    }
  value = (j - 1) / 0x8000 + 1;
  *table = malloc(value * sizeof(unsigned int));
  if(*table == NULL)
    {
      fprintf(stderr, "epzip: can't allocate block table\n");
      exit(1);
    }

  fseek(file->fp, 0, SEEK_SET);

  bread_begin(file->fp);
  for(i = 0; i < value; i++)
    {
      unsigned int count = 0;
      while(ftell(file->fp) < (i + 1) * 0x8000 && element < file->elements)
	{
	  element++;
	  count++;
	  if(bread(1))
	    bread(23);
	  else
	    bread(8);
	}
      (*table)[i] = count;
    }
  bread_end();

  lit_table = malloc(288 * sizeof(struct huffman_code));
  if(lit_table == NULL)
    {
      fprintf(stderr, "epzip: can't allocate Huffman code table\n");
      exit(1);
    }

  dist_table = malloc(32 * sizeof(struct huffman_code));
  if(dist_table == NULL)
    {
      fprintf(stderr, "epzip: can't allocate Huffman code table\n");
      exit(1);
    }

  for(i = 0; i < 144; i++)
    lit_table[i].bits = 8;
  for(i = 144; i < 256; i++)
    lit_table[i].bits = 9;
  for(i = 256; i < 280; i++)
    lit_table[i].bits = 7;
  for(i = 280; i < 288; i++)
    lit_table[i].bits = 8;
  for(i = 0; i < 32; i++)
    dist_table[i].bits = 5;

  return value;
}

static void end_blocks(unsigned int **table)
{
  free(lit_table);
  free(dist_table);
  free(*table);
}

static void block_begin(unsigned short block, unsigned short blocks)
{
  unsigned short i, j;
  unsigned short bit_length_count[16];
  unsigned short next_code[16];

  if(block + 1 < blocks)
    bwrite(0, 1);
  else
    bwrite(1, 1);
  bwrite(1, 2);

  for(i = 0; i < 16; i++)
    bit_length_count[i] = 0;
  for(i = 0; i < 288; i++)
    bit_length_count[lit_table[i].bits]++;
  j = 0;
  bit_length_count[0] = 0;
  for(i = 1; i < 16; i++)
    {
      j = (j + bit_length_count[i - 1]) << 1;
      next_code[i] = j;
    }
  for(i = 0; i < 288; i++)
    {
      j = lit_table[i].bits;
      if(j)
	lit_table[i].code = reverse_bits(next_code[j]++, j);
    }

  for(i = 0; i < 16; i++)
    bit_length_count[i] = 0;
  for(i = 0; i < 32; i++)
    bit_length_count[dist_table[i].bits]++;
  j = 0;
  bit_length_count[0] = 0;
  for(i = 1; i < 16; i++)
    {
      j = (j + bit_length_count[i - 1]) << 1;
      next_code[i] = j;
    }
  for(i = 0; i < 32; i++)
    {
      j = dist_table[i].bits;
      if(j)
	dist_table[i].code = reverse_bits(next_code[j]++, j);
    }
}

static void block_end()
{
  block_put_literal(256);
}

static void block_put_literal(unsigned short value)
{
  if(value > 285)
    {
      fprintf(stderr, "epzip: internal error:illegal literal\n");
      exit(1);
    }
  bwrite(lit_table[value].code, lit_table[value].bits);
}

static void block_put_dist_literal(unsigned short value)
{
  if(value > 29)
    {
      fprintf(stderr, "epzip: internal error:illegal literal\n");
      exit(1);
    }
  bwrite(dist_table[value].code, dist_table[value].bits);
}

static void block_put_length(unsigned short value)
{
  unsigned char i;
  if(value == 255)
    {
      block_put_literal(285);
      return;
    }
  for(i = 7; i > 2; i--)
    if(value & 1 << i)
      {
	block_put_literal(i * 4 + ((value >> (i - 2)) & 3) + 253);
	bwrite(value & ((1 << i) - 1), i - 2);
	return;
      }
  block_put_literal((value & 7) + 257);
}

static void block_put_distance(unsigned short value)
{
  unsigned char i;
  if(value & 0x8000)
    {
      fprintf(stderr, "epzip: internal error:illegal distance\n");
      exit(1);
    }
  for(i = 14; i > 1; i--)
    if(value & 1 << i)
      {
	block_put_dist_literal(i * 2 + ((value >> (i - 1)) & 1));
	bwrite(value & ((1 << i) - 1), i - 1);
	return;
      }
  block_put_dist_literal(value & 3);
}

static unsigned short reverse_bits(unsigned short in, unsigned char bits)
{
  unsigned short out = 0;
  for(;bits;bits--)
    {
      out <<= 1;
      out |= in & 1;
      in >>= 1;
    }
  return out;
}
