## Copyright (C) 1996-2015 Piotr Held
##
## This file is part of Octave.
##
## Octave is free software; you can redistribute it and/or
## modify it under the terms of the GNU General Public
## License as published by the Free Software Foundation;
## either version 3 of the License, or (at your option) any
## later version.
##
## Octave is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied
## warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
## PURPOSE.  See the GNU General Public License for more
## details.
##
## You should have received a copy of the GNU General Public
## License along with Octave; see the file COPYING.  If not,
## see <http://www.gnu.org/licenses/>.

## -*- texinfo -*-
## @deftypefn{Function File} {[@var{surro_data}, @var{pars}] =} surrogates (@var{S})
## @deftypefnx{Function File} {[@var{surro_data}, @var{pars}] =} surrogates (@var{S}, @var{paramName}, @var{paramValue}, @dots{})
##
## Generates multivariate surrogate data (implements the iterative Fourier scheme).
##
## @strong{Input}
##
## @table @var
## @item S
## This function always assumes that each time series is along the longer 
## dimension of matrix @var{S}. It also assumes that every dimension 
## (counting along the shorter dimension) of @var{S} is considered a 
## component of the time series. It's length must be factorizable by only
## 2, 3 and 5. If not the largest submatrix that fulfills this requirement will be
## used. The function @code{endtoend} can be used to determine what is the best
## submatrix for the data and then sending only that submatrix to this program.
## Padding with zeros is @strong{not} and option.
## @end table
##
## @strong {Parameters}
##
## @table @var
## @item n
## Sets the number of surrogates to be calculated. Determines the form of
## the output (see Output section) [default = 1].
## @item i
## The maximum number of permutations. Value '0' yields random permutations or if
## switch @var{exact} is set an unrescaled FFT surrogate. Value '1' is a surrogate
## close to the result of the AAFT procedure, but not quite the same. Value '-1'
## means the program will perform iterations until there is no change between them
## [default = -1].
## @item seed
## Set the seed for the random generator [default = use default seed].
## @end table
##
## @strong {Switch}
##
## @table @var
## @item exact
## This switch makes the spectrum of the output exact rather than a distribution.
## @end table
##
## @strong{Outputs}
##
## @table @var
## @item surro_data
## If parameter @code{n == 1} then this is a matrix that holds the surrogate data.
## If parameter @code{n > 1} then it is @var{n} x 1 cell array of matrixes with
## the data. In both cases the matrixes themselves are alligned with the input.
## @item pars
## This is a matrix of size @var{n} x 2 (if the input components were column
## vectors, otherwise transposed). The first column contains the number of
## iteration it took to generate the @var{i}-th surrogate, whereas the second
## column is the relative discrepency for the @var{i}-th surrogate.
## @end table
##
## @seealso{demo surrogates, endtoend}
##
## @strong{Algorithms}
##
## The algorithms for this functions have been taken from the TISEAN package.
## @end deftypefn

## Author: Piotr Held <pjheld@gmail.com>.
## This function is based on randomize module of TISEAN 3.0.1 
## https://github.com/heggus/Tisean"

function output = randomize (S, varargin)

  if (nargin < 1)
    print_usage;
  endif

  if ((ismatrix (S) == false) || (isreal (S) == false) || ...
       (isreal(S) == false))
    error ('Octave:invalid-input-arg', "S must be a real matrix");
  endif

  # Default values -- currently only for randomize_auto_exp_random
  # Main function
  no_sur               = 1;
  improvement_b4_write = 0.9;
  seed                 = 0;
  # Cost function
  no_lags      = 50; #required
  average      = 0;
  # Cooling options -- for exp from TISEAN
  init_temp                     = 0.0; 
  cooling_factor                = 0.0;
  total_steps_before_cooling    = 20000;
  succ_steps_before_cooling     = 2000;
  min_succ_steps_before_cooling = 200;
  goal_of_cost                  = 0.0;
  # Permutation options
  exclude = [];

  #### Parse the input
  p = inputParser ();
  p.FunctionName = "randomize";

  isPositiveIntScalar   = @(x) isreal (x) && isscalar (x) ...
                               && (x > 0) && (x-round(x) == 0);
  isPositiveScalar      = @(x) isreal (x) && isscalar (x) && (x > 0);
  isNonNegativeScalar   = @(x) isreal (x) && isscalar (x) ...
                               && ((x > 0) || (x == 0));
  isValidIndexMatrix    = @(x) isreal (x) && all (x <= length (S) ...
                                                     && x >= 1) ...
                               || isempty (x);

  # Running function
  p.addParamValue ("n", no_sur, isPositiveIntScalar);
  p.addParamValue ("u", improvement_b4_write, isPositiveScalar);
  p.addParamValue ("seed", seed, isNonNegativeScalar);
  # Cost function
  p.addParamValue ("d", no_lags, isPositiveIntScalar);
  p.addParamValue ("w", average, @(x) isscalar (x) && any (x == [0 1 2 3]));
  # Cooling function
  p.addParamValue ("t", init_temp, isNonNegativeScalar);
  p.addParamValue ("a", cooling_factor, isNonNegativeScalar);
  p.addParamValue ("totalsteps", total_steps_before_cooling, ...
                   isPositiveIntScalar);
  p.addParamValue ("succsteps", succ_steps_before_cooling, ...
                   isPositiveIntScalar);
  p.addParamValue ("minsteps", min_succ_steps_before_cooling, ...
                   isPositiveIntScalar);
  p.addParamValue ("c", goal_of_cost, isNonNegativeScalar);
  # Permutation function
  p.addParamValue ("exclude", exclude,isValidIndexMatrix);

  p.parse (varargin{:});

  # Assign values
  # Running function
  no_sur               = p.Results.n;
  improvement_b4_write = p.Results.u;
  seed                 = p.Results.seed;
  # Cost function
  no_lags = p.Results.d;
  average = p.Results.w;
  # Cooling function
  init_temp                     = p.Results.t;
  cooling_factor                = p.Results.a;
  total_steps_before_cooling    = p.Results.totalsteps;
  succ_steps_before_cooling     = p.Results.succsteps;
  min_succ_steps_before_cooling = p.Results.minsteps;
  goal_of_cost                  = p.Results.c;
  # Permutation function
  exclude = p.Results.exclude;

  # Load all of the parameters up into a struct

  # Running parameters
  run_params = struct;
  run_params = setfield (run_params, "no_sur", no_sur);
  run_params = setfield (run_params, "improvement_b4_write",...
                             improvement_b4_write);
  run_params = setfield (run_params, "seed", seed);

  # Cost parameters
  cost_params = struct;
  cost_params = setfield (cost_params, "no_lags", no_lags);
  cost_params = setfield (cost_params, "average", average);

  # Cooling parameters
  cool_params = struct;
  cool_params = setfield (cool_params, "init_temp", init_temp);
  cool_params = setfield (cool_params, "cooling_factor",...
                              cooling_factor);
  cool_params = setfield (cool_params, "total",
                              total_steps_before_cooling);
  cool_params = setfield (cool_params, "succ",
                              succ_steps_before_cooling);
  cool_params = setfield (cool_params, "min",
                              min_succ_steps_before_cooling);
  cool_params = setfield (cool_params, "goal", goal_of_cost);

  # Permutation parameters
  permute_params = struct;
  permute_params = setfield (permute_params, "exclude", exclude);

  # Correct S to always have more rows than columns
  trnspsd = false;
  if (rows (S) < columns (S))
    S = S.';
    trnspsd = true;
  endif

  [output, save_cool_f, current_time, end_now, input_saved_state, c, cmin, cost] = ...
    __randomize__ (S, run_params, cost_params, cool_params,...
                   permute_params);
  while (end_now == false)
    [output, save_cool_f, current_time, end_now, input_saved_state, c, cmin, cost] = ...
      __randomize__ (S, run_params, cost_params, cool_params,...
                     permute_params, save_cool_f, current_time,...
                     input_saved_state, c, cmin, cost);

    current_time
    fflush (stdout);
  endwhile
endfunction

%!xtest
%! load randomize_res_spike.dat;
%! load spike.dat
%! res = randomize (spike, 'd',50, 't', 0.01, 'seed', 0.25, 'u', 0.95 ,'c', 0.001);
%%! assert (res, randomize_res_spike, -1e-6);
