/*
 * Copyright 2002-2021 The OpenSSL Project Authors. All Rights Reserved.
 * Copyright (c) 2002, Oracle and/or its affiliates. All rights reserved
 *
 * Licensed under the Apache License 2.0 (the "License").  You may not use
 * this file except in compliance with the License.  You can obtain a copy
 * in the file LICENSE in the source distribution or at
 * https://www.openssl.org/source/license.html
 */

/*
 * ECDSA low level APIs are deprecated for public use, but still ok for
 * internal use.
 */
#include "internal/deprecated.h"

#include <string.h>
#include "ec_local.h"
#include <openssl/err.h>
#include <openssl/obj_mac.h>
#include <openssl/objects.h>
#include <openssl/opensslconf.h>
#include "internal/nelem.h"

typedef struct {
    int field_type, /* either NID_X9_62_prime_field or
                     * NID_X9_62_characteristic_two_field */
        seed_len, param_len;
    unsigned int cofactor; /* promoted to BN_ULONG */
} EC_CURVE_DATA;

/* the nist prime curves */
static const struct {
    EC_CURVE_DATA h;
    unsigned char data[20 + 28 * 6];
} _EC_NIST_PRIME_224 = {
    { NID_X9_62_prime_field, 20, 28, 1 },
    { /* seed */
        0xBD, 0x71, 0x34, 0x47, 0x99, 0xD5, 0xC7, 0xFC, 0xDC, 0x45, 0xB5, 0x9F,
        0xA3, 0xB9, 0xAB, 0x8F, 0x6A, 0x94, 0x8B, 0xC5,
        /* p */
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x01,
        /* a */
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0xFF, 0xFF, 0xFF, 0xFE, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0xFF, 0xFF, 0xFF, 0xFE,
        /* b */
        0xB4, 0x05, 0x0A, 0x85, 0x0C, 0x04, 0xB3, 0xAB, 0xF5, 0x41, 0x32, 0x56,
        0x50, 0x44, 0xB0, 0xB7, 0xD7, 0xBF, 0xD8, 0xBA, 0x27, 0x0B, 0x39, 0x43,
        0x23, 0x55, 0xFF, 0xB4,
        /* x */
        0xB7, 0x0E, 0x0C, 0xBD, 0x6B, 0xB4, 0xBF, 0x7F, 0x32, 0x13, 0x90, 0xB9,
        0x4A, 0x03, 0xC1, 0xD3, 0x56, 0xC2, 0x11, 0x22, 0x34, 0x32, 0x80, 0xD6,
        0x11, 0x5C, 0x1D, 0x21,
        /* y */
        0xbd, 0x37, 0x63, 0x88, 0xb5, 0xf7, 0x23, 0xfb, 0x4c, 0x22, 0xdf, 0xe6,
        0xcd, 0x43, 0x75, 0xa0, 0x5a, 0x07, 0x47, 0x64, 0x44, 0xd5, 0x81, 0x99,
        0x85, 0x00, 0x7e, 0x34,
        /* order */
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0xFF, 0xFF, 0x16, 0xA2, 0xE0, 0xB8, 0xF0, 0x3E, 0x13, 0xDD, 0x29, 0x45,
        0x5C, 0x5C, 0x2A, 0x3D }
};

static const struct {
    EC_CURVE_DATA h;
    unsigned char data[20 + 48 * 6];
} _EC_NIST_PRIME_384 = {
    { NID_X9_62_prime_field, 20, 48, 1 },
    { /* seed */
        0xA3, 0x35, 0x92, 0x6A, 0xA3, 0x19, 0xA2, 0x7A, 0x1D, 0x00, 0x89, 0x6A,
        0x67, 0x73, 0xA4, 0x82, 0x7A, 0xCD, 0xAC, 0x73,
        /* p */
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFE, 0xFF, 0xFF, 0xFF, 0xFF,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF,
        /* a */
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFE, 0xFF, 0xFF, 0xFF, 0xFF,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFC,
        /* b */
        0xB3, 0x31, 0x2F, 0xA7, 0xE2, 0x3E, 0xE7, 0xE4, 0x98, 0x8E, 0x05, 0x6B,
        0xE3, 0xF8, 0x2D, 0x19, 0x18, 0x1D, 0x9C, 0x6E, 0xFE, 0x81, 0x41, 0x12,
        0x03, 0x14, 0x08, 0x8F, 0x50, 0x13, 0x87, 0x5A, 0xC6, 0x56, 0x39, 0x8D,
        0x8A, 0x2E, 0xD1, 0x9D, 0x2A, 0x85, 0xC8, 0xED, 0xD3, 0xEC, 0x2A, 0xEF,
        /* x */
        0xAA, 0x87, 0xCA, 0x22, 0xBE, 0x8B, 0x05, 0x37, 0x8E, 0xB1, 0xC7, 0x1E,
        0xF3, 0x20, 0xAD, 0x74, 0x6E, 0x1D, 0x3B, 0x62, 0x8B, 0xA7, 0x9B, 0x98,
        0x59, 0xF7, 0x41, 0xE0, 0x82, 0x54, 0x2A, 0x38, 0x55, 0x02, 0xF2, 0x5D,
        0xBF, 0x55, 0x29, 0x6C, 0x3A, 0x54, 0x5E, 0x38, 0x72, 0x76, 0x0A, 0xB7,
        /* y */
        0x36, 0x17, 0xde, 0x4a, 0x96, 0x26, 0x2c, 0x6f, 0x5d, 0x9e, 0x98, 0xbf,
        0x92, 0x92, 0xdc, 0x29, 0xf8, 0xf4, 0x1d, 0xbd, 0x28, 0x9a, 0x14, 0x7c,
        0xe9, 0xda, 0x31, 0x13, 0xb5, 0xf0, 0xb8, 0xc0, 0x0a, 0x60, 0xb1, 0xce,
        0x1d, 0x7e, 0x81, 0x9d, 0x7a, 0x43, 0x1d, 0x7c, 0x90, 0xea, 0x0e, 0x5f,
        /* order */
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0xC7, 0x63, 0x4D, 0x81, 0xF4, 0x37, 0x2D, 0xDF, 0x58, 0x1A, 0x0D, 0xB2,
        0x48, 0xB0, 0xA7, 0x7A, 0xEC, 0xEC, 0x19, 0x6A, 0xCC, 0xC5, 0x29, 0x73 }
};

static const struct {
    EC_CURVE_DATA h;
    unsigned char data[20 + 66 * 6];
} _EC_NIST_PRIME_521 = {
    { NID_X9_62_prime_field, 20, 66, 1 },
    { /* seed */
        0xD0, 0x9E, 0x88, 0x00, 0x29, 0x1C, 0xB8, 0x53, 0x96, 0xCC, 0x67, 0x17,
        0x39, 0x32, 0x84, 0xAA, 0xA0, 0xDA, 0x64, 0xBA,
        /* p */
        0x01, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        /* a */
        0x01, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFC,
        /* b */
        0x00, 0x51, 0x95, 0x3E, 0xB9, 0x61, 0x8E, 0x1C, 0x9A, 0x1F, 0x92, 0x9A,
        0x21, 0xA0, 0xB6, 0x85, 0x40, 0xEE, 0xA2, 0xDA, 0x72, 0x5B, 0x99, 0xB3,
        0x15, 0xF3, 0xB8, 0xB4, 0x89, 0x91, 0x8E, 0xF1, 0x09, 0xE1, 0x56, 0x19,
        0x39, 0x51, 0xEC, 0x7E, 0x93, 0x7B, 0x16, 0x52, 0xC0, 0xBD, 0x3B, 0xB1,
        0xBF, 0x07, 0x35, 0x73, 0xDF, 0x88, 0x3D, 0x2C, 0x34, 0xF1, 0xEF, 0x45,
        0x1F, 0xD4, 0x6B, 0x50, 0x3F, 0x00,
        /* x */
        0x00, 0xC6, 0x85, 0x8E, 0x06, 0xB7, 0x04, 0x04, 0xE9, 0xCD, 0x9E, 0x3E,
        0xCB, 0x66, 0x23, 0x95, 0xB4, 0x42, 0x9C, 0x64, 0x81, 0x39, 0x05, 0x3F,
        0xB5, 0x21, 0xF8, 0x28, 0xAF, 0x60, 0x6B, 0x4D, 0x3D, 0xBA, 0xA1, 0x4B,
        0x5E, 0x77, 0xEF, 0xE7, 0x59, 0x28, 0xFE, 0x1D, 0xC1, 0x27, 0xA2, 0xFF,
        0xA8, 0xDE, 0x33, 0x48, 0xB3, 0xC1, 0x85, 0x6A, 0x42, 0x9B, 0xF9, 0x7E,
        0x7E, 0x31, 0xC2, 0xE5, 0xBD, 0x66,
        /* y */
        0x01, 0x18, 0x39, 0x29, 0x6a, 0x78, 0x9a, 0x3b, 0xc0, 0x04, 0x5c, 0x8a,
        0x5f, 0xb4, 0x2c, 0x7d, 0x1b, 0xd9, 0x98, 0xf5, 0x44, 0x49, 0x57, 0x9b,
        0x44, 0x68, 0x17, 0xaf, 0xbd, 0x17, 0x27, 0x3e, 0x66, 0x2c, 0x97, 0xee,
        0x72, 0x99, 0x5e, 0xf4, 0x26, 0x40, 0xc5, 0x50, 0xb9, 0x01, 0x3f, 0xad,
        0x07, 0x61, 0x35, 0x3c, 0x70, 0x86, 0xa2, 0x72, 0xc2, 0x40, 0x88, 0xbe,
        0x94, 0x76, 0x9f, 0xd1, 0x66, 0x50,
        /* order */
        0x01, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFA, 0x51, 0x86,
        0x87, 0x83, 0xBF, 0x2F, 0x96, 0x6B, 0x7F, 0xCC, 0x01, 0x48, 0xF7, 0x09,
        0xA5, 0xD0, 0x3B, 0xB5, 0xC9, 0xB8, 0x89, 0x9C, 0x47, 0xAE, 0xBB, 0x6F,
        0xB7, 0x1E, 0x91, 0x38, 0x64, 0x09 }
};

static const struct {
    EC_CURVE_DATA h;
    unsigned char data[20 + 32 * 6];
} _EC_X9_62_PRIME_256V1 = {
    { NID_X9_62_prime_field, 20, 32, 1 },
    { /* seed */
        0xC4, 0x9D, 0x36, 0x08, 0x86, 0xE7, 0x04, 0x93, 0x6A, 0x66, 0x78, 0xE1,
        0x13, 0x9D, 0x26, 0xB7, 0x81, 0x9F, 0x7E, 0x90,
        /* p */
        0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF,
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        /* a */
        0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF,
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFC,
        /* b */
        0x5A, 0xC6, 0x35, 0xD8, 0xAA, 0x3A, 0x93, 0xE7, 0xB3, 0xEB, 0xBD, 0x55,
        0x76, 0x98, 0x86, 0xBC, 0x65, 0x1D, 0x06, 0xB0, 0xCC, 0x53, 0xB0, 0xF6,
        0x3B, 0xCE, 0x3C, 0x3E, 0x27, 0xD2, 0x60, 0x4B,
        /* x */
        0x6B, 0x17, 0xD1, 0xF2, 0xE1, 0x2C, 0x42, 0x47, 0xF8, 0xBC, 0xE6, 0xE5,
        0x63, 0xA4, 0x40, 0xF2, 0x77, 0x03, 0x7D, 0x81, 0x2D, 0xEB, 0x33, 0xA0,
        0xF4, 0xA1, 0x39, 0x45, 0xD8, 0x98, 0xC2, 0x96,
        /* y */
        0x4f, 0xe3, 0x42, 0xe2, 0xfe, 0x1a, 0x7f, 0x9b, 0x8e, 0xe7, 0xeb, 0x4a,
        0x7c, 0x0f, 0x9e, 0x16, 0x2b, 0xce, 0x33, 0x57, 0x6b, 0x31, 0x5e, 0xce,
        0xcb, 0xb6, 0x40, 0x68, 0x37, 0xbf, 0x51, 0xf5,
        /* order */
        0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF,
        0xFF, 0xFF, 0xFF, 0xFF, 0xBC, 0xE6, 0xFA, 0xAD, 0xA7, 0x17, 0x9E, 0x84,
        0xF3, 0xB9, 0xCA, 0xC2, 0xFC, 0x63, 0x25, 0x51 }
};

#ifndef FIPS_MODULE
/* the secg prime curves (minus the nist and x9.62 prime curves) */
static const struct {
    EC_CURVE_DATA h;
    unsigned char data[0 + 32 * 6];
} _EC_SECG_PRIME_256K1 = {
    { NID_X9_62_prime_field, 0, 32, 1 },
    { /* no seed */
        /* p */
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0xFF, 0xFF, 0xFF, 0xFE, 0xFF, 0xFF, 0xFC, 0x2F,
        /* a */
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        /* b */
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07,
        /* x */
        0x79, 0xBE, 0x66, 0x7E, 0xF9, 0xDC, 0xBB, 0xAC, 0x55, 0xA0, 0x62, 0x95,
        0xCE, 0x87, 0x0B, 0x07, 0x02, 0x9B, 0xFC, 0xDB, 0x2D, 0xCE, 0x28, 0xD9,
        0x59, 0xF2, 0x81, 0x5B, 0x16, 0xF8, 0x17, 0x98,
        /* y */
        0x48, 0x3a, 0xda, 0x77, 0x26, 0xa3, 0xc4, 0x65, 0x5d, 0xa4, 0xfb, 0xfc,
        0x0e, 0x11, 0x08, 0xa8, 0xfd, 0x17, 0xb4, 0x48, 0xa6, 0x85, 0x54, 0x19,
        0x9c, 0x47, 0xd0, 0x8f, 0xfb, 0x10, 0xd4, 0xb8,
        /* order */
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0xFF, 0xFF, 0xFF, 0xFE, 0xBA, 0xAE, 0xDC, 0xE6, 0xAF, 0x48, 0xA0, 0x3B,
        0xBF, 0xD2, 0x5E, 0x8C, 0xD0, 0x36, 0x41, 0x41 }
};

static const struct {
    EC_CURVE_DATA h;
    unsigned char data[0 + 32 * 6];
} _EC_brainpoolP256r1 = {
    { NID_X9_62_prime_field, 0, 32, 1 },
    { /* no seed */
        /* p */
        0xA9, 0xFB, 0x57, 0xDB, 0xA1, 0xEE, 0xA9, 0xBC, 0x3E, 0x66, 0x0A, 0x90,
        0x9D, 0x83, 0x8D, 0x72, 0x6E, 0x3B, 0xF6, 0x23, 0xD5, 0x26, 0x20, 0x28,
        0x20, 0x13, 0x48, 0x1D, 0x1F, 0x6E, 0x53, 0x77,
        /* a */
        0x7D, 0x5A, 0x09, 0x75, 0xFC, 0x2C, 0x30, 0x57, 0xEE, 0xF6, 0x75, 0x30,
        0x41, 0x7A, 0xFF, 0xE7, 0xFB, 0x80, 0x55, 0xC1, 0x26, 0xDC, 0x5C, 0x6C,
        0xE9, 0x4A, 0x4B, 0x44, 0xF3, 0x30, 0xB5, 0xD9,
        /* b */
        0x26, 0xDC, 0x5C, 0x6C, 0xE9, 0x4A, 0x4B, 0x44, 0xF3, 0x30, 0xB5, 0xD9,
        0xBB, 0xD7, 0x7C, 0xBF, 0x95, 0x84, 0x16, 0x29, 0x5C, 0xF7, 0xE1, 0xCE,
        0x6B, 0xCC, 0xDC, 0x18, 0xFF, 0x8C, 0x07, 0xB6,
        /* x */
        0x8B, 0xD2, 0xAE, 0xB9, 0xCB, 0x7E, 0x57, 0xCB, 0x2C, 0x4B, 0x48, 0x2F,
        0xFC, 0x81, 0xB7, 0xAF, 0xB9, 0xDE, 0x27, 0xE1, 0xE3, 0xBD, 0x23, 0xC2,
        0x3A, 0x44, 0x53, 0xBD, 0x9A, 0xCE, 0x32, 0x62,
        /* y */
        0x54, 0x7E, 0xF8, 0x35, 0xC3, 0xDA, 0xC4, 0xFD, 0x97, 0xF8, 0x46, 0x1A,
        0x14, 0x61, 0x1D, 0xC9, 0xC2, 0x77, 0x45, 0x13, 0x2D, 0xED, 0x8E, 0x54,
        0x5C, 0x1D, 0x54, 0xC7, 0x2F, 0x04, 0x69, 0x97,
        /* order */
        0xA9, 0xFB, 0x57, 0xDB, 0xA1, 0xEE, 0xA9, 0xBC, 0x3E, 0x66, 0x0A, 0x90,
        0x9D, 0x83, 0x8D, 0x71, 0x8C, 0x39, 0x7A, 0xA3, 0xB5, 0x61, 0xA6, 0xF7,
        0x90, 0x1E, 0x0E, 0x82, 0x97, 0x48, 0x56, 0xA7 }
};

static const struct {
    EC_CURVE_DATA h;
    unsigned char data[0 + 32 * 6];
} _EC_brainpoolP256t1 = {
    { NID_X9_62_prime_field, 0, 32, 1 },
    { /* no seed */
        /* p */
        0xA9, 0xFB, 0x57, 0xDB, 0xA1, 0xEE, 0xA9, 0xBC, 0x3E, 0x66, 0x0A, 0x90,
        0x9D, 0x83, 0x8D, 0x72, 0x6E, 0x3B, 0xF6, 0x23, 0xD5, 0x26, 0x20, 0x28,
        0x20, 0x13, 0x48, 0x1D, 0x1F, 0x6E, 0x53, 0x77,
        /* a */
        0xA9, 0xFB, 0x57, 0xDB, 0xA1, 0xEE, 0xA9, 0xBC, 0x3E, 0x66, 0x0A, 0x90,
        0x9D, 0x83, 0x8D, 0x72, 0x6E, 0x3B, 0xF6, 0x23, 0xD5, 0x26, 0x20, 0x28,
        0x20, 0x13, 0x48, 0x1D, 0x1F, 0x6E, 0x53, 0x74,
        /* b */
        0x66, 0x2C, 0x61, 0xC4, 0x30, 0xD8, 0x4E, 0xA4, 0xFE, 0x66, 0xA7, 0x73,
        0x3D, 0x0B, 0x76, 0xB7, 0xBF, 0x93, 0xEB, 0xC4, 0xAF, 0x2F, 0x49, 0x25,
        0x6A, 0xE5, 0x81, 0x01, 0xFE, 0xE9, 0x2B, 0x04,
        /* x */
        0xA3, 0xE8, 0xEB, 0x3C, 0xC1, 0xCF, 0xE7, 0xB7, 0x73, 0x22, 0x13, 0xB2,
        0x3A, 0x65, 0x61, 0x49, 0xAF, 0xA1, 0x42, 0xC4, 0x7A, 0xAF, 0xBC, 0x2B,
        0x79, 0xA1, 0x91, 0x56, 0x2E, 0x13, 0x05, 0xF4,
        /* y */
        0x2D, 0x99, 0x6C, 0x82, 0x34, 0x39, 0xC5, 0x6D, 0x7F, 0x7B, 0x22, 0xE1,
        0x46, 0x44, 0x41, 0x7E, 0x69, 0xBC, 0xB6, 0xDE, 0x39, 0xD0, 0x27, 0x00,
        0x1D, 0xAB, 0xE8, 0xF3, 0x5B, 0x25, 0xC9, 0xBE,
        /* order */
        0xA9, 0xFB, 0x57, 0xDB, 0xA1, 0xEE, 0xA9, 0xBC, 0x3E, 0x66, 0x0A, 0x90,
        0x9D, 0x83, 0x8D, 0x71, 0x8C, 0x39, 0x7A, 0xA3, 0xB5, 0x61, 0xA6, 0xF7,
        0x90, 0x1E, 0x0E, 0x82, 0x97, 0x48, 0x56, 0xA7 }
};

static const struct {
    EC_CURVE_DATA h;
    unsigned char data[0 + 40 * 6];
} _EC_brainpoolP320r1 = {
    { NID_X9_62_prime_field, 0, 40, 1 },
    { /* no seed */
        /* p */
        0xD3, 0x5E, 0x47, 0x20, 0x36, 0xBC, 0x4F, 0xB7, 0xE1, 0x3C, 0x78, 0x5E,
        0xD2, 0x01, 0xE0, 0x65, 0xF9, 0x8F, 0xCF, 0xA6, 0xF6, 0xF4, 0x0D, 0xEF,
        0x4F, 0x92, 0xB9, 0xEC, 0x78, 0x93, 0xEC, 0x28, 0xFC, 0xD4, 0x12, 0xB1,
        0xF1, 0xB3, 0x2E, 0x27,
        /* a */
        0x3E, 0xE3, 0x0B, 0x56, 0x8F, 0xBA, 0xB0, 0xF8, 0x83, 0xCC, 0xEB, 0xD4,
        0x6D, 0x3F, 0x3B, 0xB8, 0xA2, 0xA7, 0x35, 0x13, 0xF5, 0xEB, 0x79, 0xDA,
        0x66, 0x19, 0x0E, 0xB0, 0x85, 0xFF, 0xA9, 0xF4, 0x92, 0xF3, 0x75, 0xA9,
        0x7D, 0x86, 0x0E, 0xB4,
        /* b */
        0x52, 0x08, 0x83, 0x94, 0x9D, 0xFD, 0xBC, 0x42, 0xD3, 0xAD, 0x19, 0x86,
        0x40, 0x68, 0x8A, 0x6F, 0xE1, 0x3F, 0x41, 0x34, 0x95, 0x54, 0xB4, 0x9A,
        0xCC, 0x31, 0xDC, 0xCD, 0x88, 0x45, 0x39, 0x81, 0x6F, 0x5E, 0xB4, 0xAC,
        0x8F, 0xB1, 0xF1, 0xA6,
        /* x */
        0x43, 0xBD, 0x7E, 0x9A, 0xFB, 0x53, 0xD8, 0xB8, 0x52, 0x89, 0xBC, 0xC4,
        0x8E, 0xE5, 0xBF, 0xE6, 0xF2, 0x01, 0x37, 0xD1, 0x0A, 0x08, 0x7E, 0xB6,
        0xE7, 0x87, 0x1E, 0x2A, 0x10, 0xA5, 0x99, 0xC7, 0x10, 0xAF, 0x8D, 0x0D,
        0x39, 0xE2, 0x06, 0x11,
        /* y */
        0x14, 0xFD, 0xD0, 0x55, 0x45, 0xEC, 0x1C, 0xC8, 0xAB, 0x40, 0x93, 0x24,
        0x7F, 0x77, 0x27, 0x5E, 0x07, 0x43, 0xFF, 0xED, 0x11, 0x71, 0x82, 0xEA,
        0xA9, 0xC7, 0x78, 0x77, 0xAA, 0xAC, 0x6A, 0xC7, 0xD3, 0x52, 0x45, 0xD1,
        0x69, 0x2E, 0x8E, 0xE1,
        /* order */
        0xD3, 0x5E, 0x47, 0x20, 0x36, 0xBC, 0x4F, 0xB7, 0xE1, 0x3C, 0x78, 0x5E,
        0xD2, 0x01, 0xE0, 0x65, 0xF9, 0x8F, 0xCF, 0xA5, 0xB6, 0x8F, 0x12, 0xA3,
        0x2D, 0x48, 0x2E, 0xC7, 0xEE, 0x86, 0x58, 0xE9, 0x86, 0x91, 0x55, 0x5B,
        0x44, 0xC5, 0x93, 0x11 }
};

static const struct {
    EC_CURVE_DATA h;
    unsigned char data[0 + 40 * 6];
} _EC_brainpoolP320t1 = {
    { NID_X9_62_prime_field, 0, 40, 1 },
    { /* no seed */
        /* p */
        0xD3, 0x5E, 0x47, 0x20, 0x36, 0xBC, 0x4F, 0xB7, 0xE1, 0x3C, 0x78, 0x5E,
        0xD2, 0x01, 0xE0, 0x65, 0xF9, 0x8F, 0xCF, 0xA6, 0xF6, 0xF4, 0x0D, 0xEF,
        0x4F, 0x92, 0xB9, 0xEC, 0x78, 0x93, 0xEC, 0x28, 0xFC, 0xD4, 0x12, 0xB1,
        0xF1, 0xB3, 0x2E, 0x27,
        /* a */
        0xD3, 0x5E, 0x47, 0x20, 0x36, 0xBC, 0x4F, 0xB7, 0xE1, 0x3C, 0x78, 0x5E,
        0xD2, 0x01, 0xE0, 0x65, 0xF9, 0x8F, 0xCF, 0xA6, 0xF6, 0xF4, 0x0D, 0xEF,
        0x4F, 0x92, 0xB9, 0xEC, 0x78, 0x93, 0xEC, 0x28, 0xFC, 0xD4, 0x12, 0xB1,
        0xF1, 0xB3, 0x2E, 0x24,
        /* b */
        0xA7, 0xF5, 0x61, 0xE0, 0x38, 0xEB, 0x1E, 0xD5, 0x60, 0xB3, 0xD1, 0x47,
        0xDB, 0x78, 0x20, 0x13, 0x06, 0x4C, 0x19, 0xF2, 0x7E, 0xD2, 0x7C, 0x67,
        0x80, 0xAA, 0xF7, 0x7F, 0xB8, 0xA5, 0x47, 0xCE, 0xB5, 0xB4, 0xFE, 0xF4,
        0x22, 0x34, 0x03, 0x53,
        /* x */
        0x92, 0x5B, 0xE9, 0xFB, 0x01, 0xAF, 0xC6, 0xFB, 0x4D, 0x3E, 0x7D, 0x49,
        0x90, 0x01, 0x0F, 0x81, 0x34, 0x08, 0xAB, 0x10, 0x6C, 0x4F, 0x09, 0xCB,
        0x7E, 0xE0, 0x78, 0x68, 0xCC, 0x13, 0x6F, 0xFF, 0x33, 0x57, 0xF6, 0x24,
        0xA2, 0x1B, 0xED, 0x52,
        /* y */
        0x63, 0xBA, 0x3A, 0x7A, 0x27, 0x48, 0x3E, 0xBF, 0x66, 0x71, 0xDB, 0xEF,
        0x7A, 0xBB, 0x30, 0xEB, 0xEE, 0x08, 0x4E, 0x58, 0xA0, 0xB0, 0x77, 0xAD,
        0x42, 0xA5, 0xA0, 0x98, 0x9D, 0x1E, 0xE7, 0x1B, 0x1B, 0x9B, 0xC0, 0x45,
        0x5F, 0xB0, 0xD2, 0xC3,
        /* order */
        0xD3, 0x5E, 0x47, 0x20, 0x36, 0xBC, 0x4F, 0xB7, 0xE1, 0x3C, 0x78, 0x5E,
        0xD2, 0x01, 0xE0, 0x65, 0xF9, 0x8F, 0xCF, 0xA5, 0xB6, 0x8F, 0x12, 0xA3,
        0x2D, 0x48, 0x2E, 0xC7, 0xEE, 0x86, 0x58, 0xE9, 0x86, 0x91, 0x55, 0x5B,
        0x44, 0xC5, 0x93, 0x11 }
};

static const struct {
    EC_CURVE_DATA h;
    unsigned char data[0 + 48 * 6];
} _EC_brainpoolP384r1 = {
    { NID_X9_62_prime_field, 0, 48, 1 },
    { /* no seed */
        /* p */
        0x8C, 0xB9, 0x1E, 0x82, 0xA3, 0x38, 0x6D, 0x28, 0x0F, 0x5D, 0x6F, 0x7E,
        0x50, 0xE6, 0x41, 0xDF, 0x15, 0x2F, 0x71, 0x09, 0xED, 0x54, 0x56, 0xB4,
        0x12, 0xB1, 0xDA, 0x19, 0x7F, 0xB7, 0x11, 0x23, 0xAC, 0xD3, 0xA7, 0x29,
        0x90, 0x1D, 0x1A, 0x71, 0x87, 0x47, 0x00, 0x13, 0x31, 0x07, 0xEC, 0x53,
        /* a */
        0x7B, 0xC3, 0x82, 0xC6, 0x3D, 0x8C, 0x15, 0x0C, 0x3C, 0x72, 0x08, 0x0A,
        0xCE, 0x05, 0xAF, 0xA0, 0xC2, 0xBE, 0xA2, 0x8E, 0x4F, 0xB2, 0x27, 0x87,
        0x13, 0x91, 0x65, 0xEF, 0xBA, 0x91, 0xF9, 0x0F, 0x8A, 0xA5, 0x81, 0x4A,
        0x50, 0x3A, 0xD4, 0xEB, 0x04, 0xA8, 0xC7, 0xDD, 0x22, 0xCE, 0x28, 0x26,
        /* b */
        0x04, 0xA8, 0xC7, 0xDD, 0x22, 0xCE, 0x28, 0x26, 0x8B, 0x39, 0xB5, 0x54,
        0x16, 0xF0, 0x44, 0x7C, 0x2F, 0xB7, 0x7D, 0xE1, 0x07, 0xDC, 0xD2, 0xA6,
        0x2E, 0x88, 0x0E, 0xA5, 0x3E, 0xEB, 0x62, 0xD5, 0x7C, 0xB4, 0x39, 0x02,
        0x95, 0xDB, 0xC9, 0x94, 0x3A, 0xB7, 0x86, 0x96, 0xFA, 0x50, 0x4C, 0x11,
        /* x */
        0x1D, 0x1C, 0x64, 0xF0, 0x68, 0xCF, 0x45, 0xFF, 0xA2, 0xA6, 0x3A, 0x81,
        0xB7, 0xC1, 0x3F, 0x6B, 0x88, 0x47, 0xA3, 0xE7, 0x7E, 0xF1, 0x4F, 0xE3,
        0xDB, 0x7F, 0xCA, 0xFE, 0x0C, 0xBD, 0x10, 0xE8, 0xE8, 0x26, 0xE0, 0x34,
        0x36, 0xD6, 0x46, 0xAA, 0xEF, 0x87, 0xB2, 0xE2, 0x47, 0xD4, 0xAF, 0x1E,
        /* y */
        0x8A, 0xBE, 0x1D, 0x75, 0x20, 0xF9, 0xC2, 0xA4, 0x5C, 0xB1, 0xEB, 0x8E,
        0x95, 0xCF, 0xD5, 0x52, 0x62, 0xB7, 0x0B, 0x29, 0xFE, 0xEC, 0x58, 0x64,
        0xE1, 0x9C, 0x05, 0x4F, 0xF9, 0x91, 0x29, 0x28, 0x0E, 0x46, 0x46, 0x21,
        0x77, 0x91, 0x81, 0x11, 0x42, 0x82, 0x03, 0x41, 0x26, 0x3C, 0x53, 0x15,
        /* order */
        0x8C, 0xB9, 0x1E, 0x82, 0xA3, 0x38, 0x6D, 0x28, 0x0F, 0x5D, 0x6F, 0x7E,
        0x50, 0xE6, 0x41, 0xDF, 0x15, 0x2F, 0x71, 0x09, 0xED, 0x54, 0x56, 0xB3,
        0x1F, 0x16, 0x6E, 0x6C, 0xAC, 0x04, 0x25, 0xA7, 0xCF, 0x3A, 0xB6, 0xAF,
        0x6B, 0x7F, 0xC3, 0x10, 0x3B, 0x88, 0x32, 0x02, 0xE9, 0x04, 0x65, 0x65 }
};

static const struct {
    EC_CURVE_DATA h;
    unsigned char data[0 + 48 * 6];
} _EC_brainpoolP384t1 = {
    { NID_X9_62_prime_field, 0, 48, 1 },
    { /* no seed */
        /* p */
        0x8C, 0xB9, 0x1E, 0x82, 0xA3, 0x38, 0x6D, 0x28, 0x0F, 0x5D, 0x6F, 0x7E,
        0x50, 0xE6, 0x41, 0xDF, 0x15, 0x2F, 0x71, 0x09, 0xED, 0x54, 0x56, 0xB4,
        0x12, 0xB1, 0xDA, 0x19, 0x7F, 0xB7, 0x11, 0x23, 0xAC, 0xD3, 0xA7, 0x29,
        0x90, 0x1D, 0x1A, 0x71, 0x87, 0x47, 0x00, 0x13, 0x31, 0x07, 0xEC, 0x53,
        /* a */
        0x8C, 0xB9, 0x1E, 0x82, 0xA3, 0x38, 0x6D, 0x28, 0x0F, 0x5D, 0x6F, 0x7E,
        0x50, 0xE6, 0x41, 0xDF, 0x15, 0x2F, 0x71, 0x09, 0xED, 0x54, 0x56, 0xB4,
        0x12, 0xB1, 0xDA, 0x19, 0x7F, 0xB7, 0x11, 0x23, 0xAC, 0xD3, 0xA7, 0x29,
        0x90, 0x1D, 0x1A, 0x71, 0x87, 0x47, 0x00, 0x13, 0x31, 0x07, 0xEC, 0x50,
        /* b */
        0x7F, 0x51, 0x9E, 0xAD, 0xA7, 0xBD, 0xA8, 0x1B, 0xD8, 0x26, 0xDB, 0xA6,
        0x47, 0x91, 0x0F, 0x8C, 0x4B, 0x93, 0x46, 0xED, 0x8C, 0xCD, 0xC6, 0x4E,
        0x4B, 0x1A, 0xBD, 0x11, 0x75, 0x6D, 0xCE, 0x1D, 0x20, 0x74, 0xAA, 0x26,
        0x3B, 0x88, 0x80, 0x5C, 0xED, 0x70, 0x35, 0x5A, 0x33, 0xB4, 0x71, 0xEE,
        /* x */
        0x18, 0xDE, 0x98, 0xB0, 0x2D, 0xB9, 0xA3, 0x06, 0xF2, 0xAF, 0xCD, 0x72,
        0x35, 0xF7, 0x2A, 0x81, 0x9B, 0x80, 0xAB, 0x12, 0xEB, 0xD6, 0x53, 0x17,
        0x24, 0x76, 0xFE, 0xCD, 0x46, 0x2A, 0xAB, 0xFF, 0xC4, 0xFF, 0x19, 0x1B,
        0x94, 0x6A, 0x5F, 0x54, 0xD8, 0xD0, 0xAA, 0x2F, 0x41, 0x88, 0x08, 0xCC,
        /* y */
        0x25, 0xAB, 0x05, 0x69, 0x62, 0xD3, 0x06, 0x51, 0xA1, 0x14, 0xAF, 0xD2,
        0x75, 0x5A, 0xD3, 0x36, 0x74, 0x7F, 0x93, 0x47, 0x5B, 0x7A, 0x1F, 0xCA,
        0x3B, 0x88, 0xF2, 0xB6, 0xA2, 0x08, 0xCC, 0xFE, 0x46, 0x94, 0x08, 0x58,
        0x4D, 0xC2, 0xB2, 0x91, 0x26, 0x75, 0xBF, 0x5B, 0x9E, 0x58, 0x29, 0x28,
        /* order */
        0x8C, 0xB9, 0x1E, 0x82, 0xA3, 0x38, 0x6D, 0x28, 0x0F, 0x5D, 0x6F, 0x7E,
        0x50, 0xE6, 0x41, 0xDF, 0x15, 0x2F, 0x71, 0x09, 0xED, 0x54, 0x56, 0xB3,
        0x1F, 0x16, 0x6E, 0x6C, 0xAC, 0x04, 0x25, 0xA7, 0xCF, 0x3A, 0xB6, 0xAF,
        0x6B, 0x7F, 0xC3, 0x10, 0x3B, 0x88, 0x32, 0x02, 0xE9, 0x04, 0x65, 0x65 }
};

static const struct {
    EC_CURVE_DATA h;
    unsigned char data[0 + 64 * 6];
} _EC_brainpoolP512r1 = {
    { NID_X9_62_prime_field, 0, 64, 1 },
    { /* no seed */
        /* p */
        0xAA, 0xDD, 0x9D, 0xB8, 0xDB, 0xE9, 0xC4, 0x8B, 0x3F, 0xD4, 0xE6, 0xAE,
        0x33, 0xC9, 0xFC, 0x07, 0xCB, 0x30, 0x8D, 0xB3, 0xB3, 0xC9, 0xD2, 0x0E,
        0xD6, 0x63, 0x9C, 0xCA, 0x70, 0x33, 0x08, 0x71, 0x7D, 0x4D, 0x9B, 0x00,
        0x9B, 0xC6, 0x68, 0x42, 0xAE, 0xCD, 0xA1, 0x2A, 0xE6, 0xA3, 0x80, 0xE6,
        0x28, 0x81, 0xFF, 0x2F, 0x2D, 0x82, 0xC6, 0x85, 0x28, 0xAA, 0x60, 0x56,
        0x58, 0x3A, 0x48, 0xF3,
        /* a */
        0x78, 0x30, 0xA3, 0x31, 0x8B, 0x60, 0x3B, 0x89, 0xE2, 0x32, 0x71, 0x45,
        0xAC, 0x23, 0x4C, 0xC5, 0x94, 0xCB, 0xDD, 0x8D, 0x3D, 0xF9, 0x16, 0x10,
        0xA8, 0x34, 0x41, 0xCA, 0xEA, 0x98, 0x63, 0xBC, 0x2D, 0xED, 0x5D, 0x5A,
        0xA8, 0x25, 0x3A, 0xA1, 0x0A, 0x2E, 0xF1, 0xC9, 0x8B, 0x9A, 0xC8, 0xB5,
        0x7F, 0x11, 0x17, 0xA7, 0x2B, 0xF2, 0xC7, 0xB9, 0xE7, 0xC1, 0xAC, 0x4D,
        0x77, 0xFC, 0x94, 0xCA,
        /* b */
        0x3D, 0xF9, 0x16, 0x10, 0xA8, 0x34, 0x41, 0xCA, 0xEA, 0x98, 0x63, 0xBC,
        0x2D, 0xED, 0x5D, 0x5A, 0xA8, 0x25, 0x3A, 0xA1, 0x0A, 0x2E, 0xF1, 0xC9,
        0x8B, 0x9A, 0xC8, 0xB5, 0x7F, 0x11, 0x17, 0xA7, 0x2B, 0xF2, 0xC7, 0xB9,
        0xE7, 0xC1, 0xAC, 0x4D, 0x77, 0xFC, 0x94, 0xCA, 0xDC, 0x08, 0x3E, 0x67,
        0x98, 0x40, 0x50, 0xB7, 0x5E, 0xBA, 0xE5, 0xDD, 0x28, 0x09, 0xBD, 0x63,
        0x80, 0x16, 0xF7, 0x23,
        /* x */
        0x81, 0xAE, 0xE4, 0xBD, 0xD8, 0x2E, 0xD9, 0x64, 0x5A, 0x21, 0x32, 0x2E,
        0x9C, 0x4C, 0x6A, 0x93, 0x85, 0xED, 0x9F, 0x70, 0xB5, 0xD9, 0x16, 0xC1,
        0xB4, 0x3B, 0x62, 0xEE, 0xF4, 0xD0, 0x09, 0x8E, 0xFF, 0x3B, 0x1F, 0x78,
        0xE2, 0xD0, 0xD4, 0x8D, 0x50, 0xD1, 0x68, 0x7B, 0x93, 0xB9, 0x7D, 0x5F,
        0x7C, 0x6D, 0x50, 0x47, 0x40, 0x6A, 0x5E, 0x68, 0x8B, 0x35, 0x22, 0x09,
        0xBC, 0xB9, 0xF8, 0x22,
        /* y */
        0x7D, 0xDE, 0x38, 0x5D, 0x56, 0x63, 0x32, 0xEC, 0xC0, 0xEA, 0xBF, 0xA9,
        0xCF, 0x78, 0x22, 0xFD, 0xF2, 0x09, 0xF7, 0x00, 0x24, 0xA5, 0x7B, 0x1A,
        0xA0, 0x00, 0xC5, 0x5B, 0x88, 0x1F, 0x81, 0x11, 0xB2, 0xDC, 0xDE, 0x49,
        0x4A, 0x5F, 0x48, 0x5E, 0x5B, 0xCA, 0x4B, 0xD8, 0x8A, 0x27, 0x63, 0xAE,
        0xD1, 0xCA, 0x2B, 0x2F, 0xA8, 0xF0, 0x54, 0x06, 0x78, 0xCD, 0x1E, 0x0F,
        0x3A, 0xD8, 0x08, 0x92,
        /* order */
        0xAA, 0xDD, 0x9D, 0xB8, 0xDB, 0xE9, 0xC4, 0x8B, 0x3F, 0xD4, 0xE6, 0xAE,
        0x33, 0xC9, 0xFC, 0x07, 0xCB, 0x30, 0x8D, 0xB3, 0xB3, 0xC9, 0xD2, 0x0E,
        0xD6, 0x63, 0x9C, 0xCA, 0x70, 0x33, 0x08, 0x70, 0x55, 0x3E, 0x5C, 0x41,
        0x4C, 0xA9, 0x26, 0x19, 0x41, 0x86, 0x61, 0x19, 0x7F, 0xAC, 0x10, 0x47,
        0x1D, 0xB1, 0xD3, 0x81, 0x08, 0x5D, 0xDA, 0xDD, 0xB5, 0x87, 0x96, 0x82,
        0x9C, 0xA9, 0x00, 0x69 }
};

static const struct {
    EC_CURVE_DATA h;
    unsigned char data[0 + 64 * 6];
} _EC_brainpoolP512t1 = {
    { NID_X9_62_prime_field, 0, 64, 1 },
    { /* no seed */
        /* p */
        0xAA, 0xDD, 0x9D, 0xB8, 0xDB, 0xE9, 0xC4, 0x8B, 0x3F, 0xD4, 0xE6, 0xAE,
        0x33, 0xC9, 0xFC, 0x07, 0xCB, 0x30, 0x8D, 0xB3, 0xB3, 0xC9, 0xD2, 0x0E,
        0xD6, 0x63, 0x9C, 0xCA, 0x70, 0x33, 0x08, 0x71, 0x7D, 0x4D, 0x9B, 0x00,
        0x9B, 0xC6, 0x68, 0x42, 0xAE, 0xCD, 0xA1, 0x2A, 0xE6, 0xA3, 0x80, 0xE6,
        0x28, 0x81, 0xFF, 0x2F, 0x2D, 0x82, 0xC6, 0x85, 0x28, 0xAA, 0x60, 0x56,
        0x58, 0x3A, 0x48, 0xF3,
        /* a */
        0xAA, 0xDD, 0x9D, 0xB8, 0xDB, 0xE9, 0xC4, 0x8B, 0x3F, 0xD4, 0xE6, 0xAE,
        0x33, 0xC9, 0xFC, 0x07, 0xCB, 0x30, 0x8D, 0xB3, 0xB3, 0xC9, 0xD2, 0x0E,
        0xD6, 0x63, 0x9C, 0xCA, 0x70, 0x33, 0x08, 0x71, 0x7D, 0x4D, 0x9B, 0x00,
        0x9B, 0xC6, 0x68, 0x42, 0xAE, 0xCD, 0xA1, 0x2A, 0xE6, 0xA3, 0x80, 0xE6,
        0x28, 0x81, 0xFF, 0x2F, 0x2D, 0x82, 0xC6, 0x85, 0x28, 0xAA, 0x60, 0x56,
        0x58, 0x3A, 0x48, 0xF0,
        /* b */
        0x7C, 0xBB, 0xBC, 0xF9, 0x44, 0x1C, 0xFA, 0xB7, 0x6E, 0x18, 0x90, 0xE4,
        0x68, 0x84, 0xEA, 0xE3, 0x21, 0xF7, 0x0C, 0x0B, 0xCB, 0x49, 0x81, 0x52,
        0x78, 0x97, 0x50, 0x4B, 0xEC, 0x3E, 0x36, 0xA6, 0x2B, 0xCD, 0xFA, 0x23,
        0x04, 0x97, 0x65, 0x40, 0xF6, 0x45, 0x00, 0x85, 0xF2, 0xDA, 0xE1, 0x45,
        0xC2, 0x25, 0x53, 0xB4, 0x65, 0x76, 0x36, 0x89, 0x18, 0x0E, 0xA2, 0x57,
        0x18, 0x67, 0x42, 0x3E,
        /* x */
        0x64, 0x0E, 0xCE, 0x5C, 0x12, 0x78, 0x87, 0x17, 0xB9, 0xC1, 0xBA, 0x06,
        0xCB, 0xC2, 0xA6, 0xFE, 0xBA, 0x85, 0x84, 0x24, 0x58, 0xC5, 0x6D, 0xDE,
        0x9D, 0xB1, 0x75, 0x8D, 0x39, 0xC0, 0x31, 0x3D, 0x82, 0xBA, 0x51, 0x73,
        0x5C, 0xDB, 0x3E, 0xA4, 0x99, 0xAA, 0x77, 0xA7, 0xD6, 0x94, 0x3A, 0x64,
        0xF7, 0xA3, 0xF2, 0x5F, 0xE2, 0x6F, 0x06, 0xB5, 0x1B, 0xAA, 0x26, 0x96,
        0xFA, 0x90, 0x35, 0xDA,
        /* y */
        0x5B, 0x53, 0x4B, 0xD5, 0x95, 0xF5, 0xAF, 0x0F, 0xA2, 0xC8, 0x92, 0x37,
        0x6C, 0x84, 0xAC, 0xE1, 0xBB, 0x4E, 0x30, 0x19, 0xB7, 0x16, 0x34, 0xC0,
        0x11, 0x31, 0x15, 0x9C, 0xAE, 0x03, 0xCE, 0xE9, 0xD9, 0x93, 0x21, 0x84,
        0xBE, 0xEF, 0x21, 0x6B, 0xD7, 0x1D, 0xF2, 0xDA, 0xDF, 0x86, 0xA6, 0x27,
        0x30, 0x6E, 0xCF, 0xF9, 0x6D, 0xBB, 0x8B, 0xAC, 0xE1, 0x98, 0xB6, 0x1E,
        0x00, 0xF8, 0xB3, 0x32,
        /* order */
        0xAA, 0xDD, 0x9D, 0xB8, 0xDB, 0xE9, 0xC4, 0x8B, 0x3F, 0xD4, 0xE6, 0xAE,
        0x33, 0xC9, 0xFC, 0x07, 0xCB, 0x30, 0x8D, 0xB3, 0xB3, 0xC9, 0xD2, 0x0E,
        0xD6, 0x63, 0x9C, 0xCA, 0x70, 0x33, 0x08, 0x70, 0x55, 0x3E, 0x5C, 0x41,
        0x4C, 0xA9, 0x26, 0x19, 0x41, 0x86, 0x61, 0x19, 0x7F, 0xAC, 0x10, 0x47,
        0x1D, 0xB1, 0xD3, 0x81, 0x08, 0x5D, 0xDA, 0xDD, 0xB5, 0x87, 0x96, 0x82,
        0x9C, 0xA9, 0x00, 0x69 }
};
#endif /* FIPS_MODULE */

#if !defined(OPENSSL_NO_SM2) && !defined(FIPS_MODULE)
static const struct {
    EC_CURVE_DATA h;
    unsigned char data[0 + 32 * 6];
} _EC_sm2p256v1 = {
    { NID_X9_62_prime_field, 0, 32, 1 },
    {
        /* no seed */

        /* p */
        0xff,
        0xff,
        0xff,
        0xfe,
        0xff,
        0xff,
        0xff,
        0xff,
        0xff,
        0xff,
        0xff,
        0xff,
        0xff,
        0xff,
        0xff,
        0xff,
        0xff,
        0xff,
        0xff,
        0xff,
        0x00,
        0x00,
        0x00,
        0x00,
        0xff,
        0xff,
        0xff,
        0xff,
        0xff,
        0xff,
        0xff,
        0xff,
        /* a */
        0xff,
        0xff,
        0xff,
        0xfe,
        0xff,
        0xff,
        0xff,
        0xff,
        0xff,
        0xff,
        0xff,
        0xff,
        0xff,
        0xff,
        0xff,
        0xff,
        0xff,
        0xff,
        0xff,
        0xff,
        0x00,
        0x00,
        0x00,
        0x00,
        0xff,
        0xff,
        0xff,
        0xff,
        0xff,
        0xff,
        0xff,
        0xfc,
        /* b */
        0x28,
        0xe9,
        0xfa,
        0x9e,
        0x9d,
        0x9f,
        0x5e,
        0x34,
        0x4d,
        0x5a,
        0x9e,
        0x4b,
        0xcf,
        0x65,
        0x09,
        0xa7,
        0xf3,
        0x97,
        0x89,
        0xf5,
        0x15,
        0xab,
        0x8f,
        0x92,
        0xdd,
        0xbc,
        0xbd,
        0x41,
        0x4d,
        0x94,
        0x0e,
        0x93,
        /* x */
        0x32,
        0xc4,
        0xae,
        0x2c,
        0x1f,
        0x19,
        0x81,
        0x19,
        0x5f,
        0x99,
        0x04,
        0x46,
        0x6a,
        0x39,
        0xc9,
        0x94,
        0x8f,
        0xe3,
        0x0b,
        0xbf,
        0xf2,
        0x66,
        0x0b,
        0xe1,
        0x71,
        0x5a,
        0x45,
        0x89,
        0x33,
        0x4c,
        0x74,
        0xc7,
        /* y */
        0xbc,
        0x37,
        0x36,
        0xa2,
        0xf4,
        0xf6,
        0x77,
        0x9c,
        0x59,
        0xbd,
        0xce,
        0xe3,
        0x6b,
        0x69,
        0x21,
        0x53,
        0xd0,
        0xa9,
        0x87,
        0x7c,
        0xc6,
        0x2a,
        0x47,
        0x40,
        0x02,
        0xdf,
        0x32,
        0xe5,
        0x21,
        0x39,
        0xf0,
        0xa0,
        /* order */
        0xff,
        0xff,
        0xff,
        0xfe,
        0xff,
        0xff,
        0xff,
        0xff,
        0xff,
        0xff,
        0xff,
        0xff,
        0xff,
        0xff,
        0xff,
        0xff,
        0x72,
        0x03,
        0xdf,
        0x6b,
        0x21,
        0xc6,
        0x05,
        0x2b,
        0x53,
        0xbb,
        0xf4,
        0x09,
        0x39,
        0xd5,
        0x41,
        0x23,
    }
};
#endif /* OPENSSL_NO_SM2 */

typedef struct _ec_list_element_st {
    int nid;
    const EC_CURVE_DATA *data;
    const EC_METHOD *(*meth)(void);
    const char *comment;
} ec_list_element;

#ifdef FIPS_MODULE
static const ec_list_element curve_list[] = {
    /* prime field curves */
    /* secg curves */
    { NID_secp224r1, &_EC_NIST_PRIME_224.h,
#if !defined(OPENSSL_NO_EC_NISTP_64_GCC_128)
        EC_GFp_nistp224_method,
#else
        0,
#endif
        "NIST/SECG curve over a 224 bit prime field" },
    /* SECG secp256r1 is the same as X9.62 prime256v1 and hence omitted */
    { NID_secp384r1, &_EC_NIST_PRIME_384.h,
#if defined(S390X_EC_ASM)
        EC_GFp_s390x_nistp384_method,
#else
        0,
#endif
        "NIST/SECG curve over a 384 bit prime field" },

    { NID_secp521r1, &_EC_NIST_PRIME_521.h,
#if defined(S390X_EC_ASM)
        EC_GFp_s390x_nistp521_method,
#elif !defined(OPENSSL_NO_EC_NISTP_64_GCC_128)
        EC_GFp_nistp521_method,
#else
        0,
#endif
        "NIST/SECG curve over a 521 bit prime field" },

    /* X9.62 curves */
    { NID_X9_62_prime256v1, &_EC_X9_62_PRIME_256V1.h,
#if defined(ECP_NISTZ256_ASM)
        EC_GFp_nistz256_method,
#elif defined(S390X_EC_ASM)
        EC_GFp_s390x_nistp256_method,
#elif !defined(OPENSSL_NO_EC_NISTP_64_GCC_128)
        EC_GFp_nistp256_method,
#else
        0,
#endif
        "X9.62/SECG curve over a 256 bit prime field" },

#ifndef OPENSSL_NO_EC2M
    /* characteristic two field curves */
    /* NIST/SECG curves */
    { NID_sect163k1, &_EC_NIST_CHAR2_163K.h, 0,
        "NIST/SECG/WTLS curve over a 163 bit binary field" },
    { NID_sect163r2, &_EC_NIST_CHAR2_163B.h, 0,
        "NIST/SECG curve over a 163 bit binary field" },
    { NID_sect233k1, &_EC_NIST_CHAR2_233K.h, 0,
        "NIST/SECG/WTLS curve over a 233 bit binary field" },
    { NID_sect233r1, &_EC_NIST_CHAR2_233B.h, 0,
        "NIST/SECG/WTLS curve over a 233 bit binary field" },
    { NID_sect283k1, &_EC_NIST_CHAR2_283K.h, 0,
        "NIST/SECG curve over a 283 bit binary field" },
    { NID_sect283r1, &_EC_NIST_CHAR2_283B.h, 0,
        "NIST/SECG curve over a 283 bit binary field" },
    { NID_sect409k1, &_EC_NIST_CHAR2_409K.h, 0,
        "NIST/SECG curve over a 409 bit binary field" },
    { NID_sect409r1, &_EC_NIST_CHAR2_409B.h, 0,
        "NIST/SECG curve over a 409 bit binary field" },
    { NID_sect571k1, &_EC_NIST_CHAR2_571K.h, 0,
        "NIST/SECG curve over a 571 bit binary field" },
    { NID_sect571r1, &_EC_NIST_CHAR2_571B.h, 0,
        "NIST/SECG curve over a 571 bit binary field" },
#endif
};

#else

static const ec_list_element curve_list[] = {
    /* prime field curves */
    /* secg curves */
#ifndef OPENSSL_NO_EC_NISTP_64_GCC_128
    { NID_secp224r1, &_EC_NIST_PRIME_224.h, EC_GFp_nistp224_method,
        "NIST/SECG curve over a 224 bit prime field" },
#else
    { NID_secp224r1, &_EC_NIST_PRIME_224.h, 0,
        "NIST/SECG curve over a 224 bit prime field" },
#endif
    { NID_secp256k1, &_EC_SECG_PRIME_256K1.h, 0,
        "SECG curve over a 256 bit prime field" },
    /* SECG secp256r1 is the same as X9.62 prime256v1 and hence omitted */
    { NID_secp384r1, &_EC_NIST_PRIME_384.h,
#if defined(S390X_EC_ASM)
        EC_GFp_s390x_nistp384_method,
#else
        0,
#endif
        "NIST/SECG curve over a 384 bit prime field" },
    { NID_secp521r1, &_EC_NIST_PRIME_521.h,
#if defined(S390X_EC_ASM)
        EC_GFp_s390x_nistp521_method,
#elif !defined(OPENSSL_NO_EC_NISTP_64_GCC_128)
        EC_GFp_nistp521_method,
#else
        0,
#endif
        "NIST/SECG curve over a 521 bit prime field" },
    /* X9.62 curves */
    { NID_X9_62_prime256v1, &_EC_X9_62_PRIME_256V1.h,
#if defined(ECP_NISTZ256_ASM)
        EC_GFp_nistz256_method,
#elif defined(S390X_EC_ASM)
        EC_GFp_s390x_nistp256_method,
#elif !defined(OPENSSL_NO_EC_NISTP_64_GCC_128)
        EC_GFp_nistp256_method,
#else
        0,
#endif
        "X9.62/SECG curve over a 256 bit prime field" },
#ifndef OPENSSL_NO_EC2M
    /* characteristic two field curves */
    /* NIST/SECG curves */
    { NID_sect113r1, &_EC_SECG_CHAR2_113R1.h, 0,
        "SECG curve over a 113 bit binary field" },
    { NID_sect113r2, &_EC_SECG_CHAR2_113R2.h, 0,
        "SECG curve over a 113 bit binary field" },
    { NID_sect131r1, &_EC_SECG_CHAR2_131R1.h, 0,
        "SECG/WTLS curve over a 131 bit binary field" },
    { NID_sect131r2, &_EC_SECG_CHAR2_131R2.h, 0,
        "SECG curve over a 131 bit binary field" },
    { NID_sect163k1, &_EC_NIST_CHAR2_163K.h, 0,
        "NIST/SECG/WTLS curve over a 163 bit binary field" },
    { NID_sect163r1, &_EC_SECG_CHAR2_163R1.h, 0,
        "SECG curve over a 163 bit binary field" },
    { NID_sect163r2, &_EC_NIST_CHAR2_163B.h, 0,
        "NIST/SECG curve over a 163 bit binary field" },
    { NID_sect193r1, &_EC_SECG_CHAR2_193R1.h, 0,
        "SECG curve over a 193 bit binary field" },
    { NID_sect193r2, &_EC_SECG_CHAR2_193R2.h, 0,
        "SECG curve over a 193 bit binary field" },
    { NID_sect233k1, &_EC_NIST_CHAR2_233K.h, 0,
        "NIST/SECG/WTLS curve over a 233 bit binary field" },
    { NID_sect233r1, &_EC_NIST_CHAR2_233B.h, 0,
        "NIST/SECG/WTLS curve over a 233 bit binary field" },
    { NID_sect239k1, &_EC_SECG_CHAR2_239K1.h, 0,
        "SECG curve over a 239 bit binary field" },
    { NID_sect283k1, &_EC_NIST_CHAR2_283K.h, 0,
        "NIST/SECG curve over a 283 bit binary field" },
    { NID_sect283r1, &_EC_NIST_CHAR2_283B.h, 0,
        "NIST/SECG curve over a 283 bit binary field" },
    { NID_sect409k1, &_EC_NIST_CHAR2_409K.h, 0,
        "NIST/SECG curve over a 409 bit binary field" },
    { NID_sect409r1, &_EC_NIST_CHAR2_409B.h, 0,
        "NIST/SECG curve over a 409 bit binary field" },
    { NID_sect571k1, &_EC_NIST_CHAR2_571K.h, 0,
        "NIST/SECG curve over a 571 bit binary field" },
    { NID_sect571r1, &_EC_NIST_CHAR2_571B.h, 0,
        "NIST/SECG curve over a 571 bit binary field" },
    /* X9.62 curves */
    { NID_X9_62_c2pnb163v1, &_EC_X9_62_CHAR2_163V1.h, 0,
        "X9.62 curve over a 163 bit binary field" },
    { NID_X9_62_c2pnb163v2, &_EC_X9_62_CHAR2_163V2.h, 0,
        "X9.62 curve over a 163 bit binary field" },
    { NID_X9_62_c2pnb163v3, &_EC_X9_62_CHAR2_163V3.h, 0,
        "X9.62 curve over a 163 bit binary field" },
    { NID_X9_62_c2pnb176v1, &_EC_X9_62_CHAR2_176V1.h, 0,
        "X9.62 curve over a 176 bit binary field" },
    { NID_X9_62_c2tnb191v1, &_EC_X9_62_CHAR2_191V1.h, 0,
        "X9.62 curve over a 191 bit binary field" },
    { NID_X9_62_c2tnb191v2, &_EC_X9_62_CHAR2_191V2.h, 0,
        "X9.62 curve over a 191 bit binary field" },
    { NID_X9_62_c2tnb191v3, &_EC_X9_62_CHAR2_191V3.h, 0,
        "X9.62 curve over a 191 bit binary field" },
    { NID_X9_62_c2pnb208w1, &_EC_X9_62_CHAR2_208W1.h, 0,
        "X9.62 curve over a 208 bit binary field" },
    { NID_X9_62_c2tnb239v1, &_EC_X9_62_CHAR2_239V1.h, 0,
        "X9.62 curve over a 239 bit binary field" },
    { NID_X9_62_c2tnb239v2, &_EC_X9_62_CHAR2_239V2.h, 0,
        "X9.62 curve over a 239 bit binary field" },
    { NID_X9_62_c2tnb239v3, &_EC_X9_62_CHAR2_239V3.h, 0,
        "X9.62 curve over a 239 bit binary field" },
    { NID_X9_62_c2pnb272w1, &_EC_X9_62_CHAR2_272W1.h, 0,
        "X9.62 curve over a 272 bit binary field" },
    { NID_X9_62_c2pnb304w1, &_EC_X9_62_CHAR2_304W1.h, 0,
        "X9.62 curve over a 304 bit binary field" },
    { NID_X9_62_c2tnb359v1, &_EC_X9_62_CHAR2_359V1.h, 0,
        "X9.62 curve over a 359 bit binary field" },
    { NID_X9_62_c2pnb368w1, &_EC_X9_62_CHAR2_368W1.h, 0,
        "X9.62 curve over a 368 bit binary field" },
    { NID_X9_62_c2tnb431r1, &_EC_X9_62_CHAR2_431R1.h, 0,
        "X9.62 curve over a 431 bit binary field" },
    /*
     * the WAP/WTLS curves [unlike SECG, spec has its own OIDs for curves
     * from X9.62]
     */
    { NID_wap_wsg_idm_ecid_wtls1, &_EC_WTLS_1.h, 0,
        "WTLS curve over a 113 bit binary field" },
    { NID_wap_wsg_idm_ecid_wtls3, &_EC_NIST_CHAR2_163K.h, 0,
        "NIST/SECG/WTLS curve over a 163 bit binary field" },
    { NID_wap_wsg_idm_ecid_wtls4, &_EC_SECG_CHAR2_113R1.h, 0,
        "SECG curve over a 113 bit binary field" },
    { NID_wap_wsg_idm_ecid_wtls5, &_EC_X9_62_CHAR2_163V1.h, 0,
        "X9.62 curve over a 163 bit binary field" },
#endif
#ifndef OPENSSL_NO_EC2M
    { NID_wap_wsg_idm_ecid_wtls10, &_EC_NIST_CHAR2_233K.h, 0,
        "NIST/SECG/WTLS curve over a 233 bit binary field" },
    { NID_wap_wsg_idm_ecid_wtls11, &_EC_NIST_CHAR2_233B.h, 0,
        "NIST/SECG/WTLS curve over a 233 bit binary field" },
#endif
#ifndef OPENSSL_NO_EC2M
    /* IPSec curves */
    { NID_ipsec3, &_EC_IPSEC_155_ID3.h, 0,
        "\n\tIPSec/IKE/Oakley curve #3 over a 155 bit binary field.\n"
        "\tNot suitable for ECDSA.\n\tQuestionable extension field!" },
    { NID_ipsec4, &_EC_IPSEC_185_ID4.h, 0,
        "\n\tIPSec/IKE/Oakley curve #4 over a 185 bit binary field.\n"
        "\tNot suitable for ECDSA.\n\tQuestionable extension field!" },
#endif
    /* brainpool curves */
    { NID_brainpoolP256r1, &_EC_brainpoolP256r1.h, 0,
        "RFC 5639 curve over a 256 bit prime field" },
    { NID_brainpoolP256t1, &_EC_brainpoolP256t1.h, 0,
        "RFC 5639 curve over a 256 bit prime field" },
    { NID_brainpoolP320r1, &_EC_brainpoolP320r1.h, 0,
        "RFC 5639 curve over a 320 bit prime field" },
    { NID_brainpoolP320t1, &_EC_brainpoolP320t1.h, 0,
        "RFC 5639 curve over a 320 bit prime field" },
    { NID_brainpoolP384r1, &_EC_brainpoolP384r1.h, 0,
        "RFC 5639 curve over a 384 bit prime field" },
    { NID_brainpoolP384t1, &_EC_brainpoolP384t1.h, 0,
        "RFC 5639 curve over a 384 bit prime field" },
    { NID_brainpoolP512r1, &_EC_brainpoolP512r1.h, 0,
        "RFC 5639 curve over a 512 bit prime field" },
    { NID_brainpoolP512t1, &_EC_brainpoolP512t1.h, 0,
        "RFC 5639 curve over a 512 bit prime field" },
#ifndef OPENSSL_NO_SM2
    { NID_sm2, &_EC_sm2p256v1.h, 0,
        "SM2 curve over a 256 bit prime field" },
#endif
};
#endif /* FIPS_MODULE */

#define curve_list_length OSSL_NELEM(curve_list)

static const ec_list_element *ec_curve_nid2curve(int nid)
{
    size_t i;

    if (nid <= 0)
        return NULL;

    for (i = 0; i < curve_list_length; i++) {
        if (curve_list[i].nid == nid)
            return &curve_list[i];
    }
    return NULL;
}

static EC_GROUP *ec_group_new_from_data(OSSL_LIB_CTX *libctx,
    const char *propq,
    const ec_list_element curve)
{
    EC_GROUP *group = NULL;
    EC_POINT *P = NULL;
    BN_CTX *ctx = NULL;
    BIGNUM *p = NULL, *a = NULL, *b = NULL, *x = NULL, *y = NULL, *order = NULL;
    int ok = 0;
    int seed_len, param_len;
    const EC_METHOD *meth;
    const EC_CURVE_DATA *data;
    const unsigned char *params;

    /* If no curve data curve method must handle everything */
    if (curve.data == NULL)
        return ossl_ec_group_new_ex(libctx, propq,
            curve.meth != NULL ? curve.meth() : NULL);

    if ((ctx = BN_CTX_new_ex(libctx)) == NULL) {
        ERR_raise(ERR_LIB_EC, ERR_R_MALLOC_FAILURE);
        goto err;
    }

    data = curve.data;
    seed_len = data->seed_len;
    param_len = data->param_len;
    params = (const unsigned char *)(data + 1); /* skip header */
    params += seed_len; /* skip seed */

    if ((p = BN_bin2bn(params + 0 * param_len, param_len, NULL)) == NULL
        || (a = BN_bin2bn(params + 1 * param_len, param_len, NULL)) == NULL
        || (b = BN_bin2bn(params + 2 * param_len, param_len, NULL)) == NULL) {
        ERR_raise(ERR_LIB_EC, ERR_R_BN_LIB);
        goto err;
    }

    if (curve.meth != 0) {
        meth = curve.meth();
        if (((group = ossl_ec_group_new_ex(libctx, propq, meth)) == NULL) || (!(group->meth->group_set_curve(group, p, a, b, ctx)))) {
            ERR_raise(ERR_LIB_EC, ERR_R_EC_LIB);
            goto err;
        }
    } else if (data->field_type == NID_X9_62_prime_field) {
        if ((group = EC_GROUP_new_curve_GFp(p, a, b, ctx)) == NULL) {
            ERR_raise(ERR_LIB_EC, ERR_R_EC_LIB);
            goto err;
        }
    }
#ifndef OPENSSL_NO_EC2M
    else { /* field_type ==
            * NID_X9_62_characteristic_two_field */

        if ((group = EC_GROUP_new_curve_GF2m(p, a, b, ctx)) == NULL) {
            ERR_raise(ERR_LIB_EC, ERR_R_EC_LIB);
            goto err;
        }
    }
#endif

    EC_GROUP_set_curve_name(group, curve.nid);

    if ((P = EC_POINT_new(group)) == NULL) {
        ERR_raise(ERR_LIB_EC, ERR_R_EC_LIB);
        goto err;
    }

    if ((x = BN_bin2bn(params + 3 * param_len, param_len, NULL)) == NULL
        || (y = BN_bin2bn(params + 4 * param_len, param_len, NULL)) == NULL) {
        ERR_raise(ERR_LIB_EC, ERR_R_BN_LIB);
        goto err;
    }
    if (!EC_POINT_set_affine_coordinates(group, P, x, y, ctx)) {
        ERR_raise(ERR_LIB_EC, ERR_R_EC_LIB);
        goto err;
    }
    if ((order = BN_bin2bn(params + 5 * param_len, param_len, NULL)) == NULL
        || !BN_set_word(x, (BN_ULONG)data->cofactor)) {
        ERR_raise(ERR_LIB_EC, ERR_R_BN_LIB);
        goto err;
    }
    if (!EC_GROUP_set_generator(group, P, order, x)) {
        ERR_raise(ERR_LIB_EC, ERR_R_EC_LIB);
        goto err;
    }
    if (seed_len) {
        if (!EC_GROUP_set_seed(group, params - seed_len, seed_len)) {
            ERR_raise(ERR_LIB_EC, ERR_R_EC_LIB);
            goto err;
        }
    }

#ifndef FIPS_MODULE
    if (EC_GROUP_get_asn1_flag(group) == OPENSSL_EC_NAMED_CURVE) {
        /*
         * Some curves don't have an associated OID: for those we should not
         * default to `OPENSSL_EC_NAMED_CURVE` encoding of parameters and
         * instead set the ASN1 flag to `OPENSSL_EC_EXPLICIT_CURVE`.
         *
         * Note that `OPENSSL_EC_NAMED_CURVE` is set as the default ASN1 flag on
         * `EC_GROUP_new()`, when we don't have enough elements to determine if
         * an OID for the curve name actually exists.
         * We could implement this check on `EC_GROUP_set_curve_name()` but
         * overloading the simple setter with this lookup could have a negative
         * performance impact and unexpected consequences.
         */
        ASN1_OBJECT *asn1obj = OBJ_nid2obj(curve.nid);

        if (asn1obj == NULL) {
            ERR_raise(ERR_LIB_EC, ERR_R_OBJ_LIB);
            goto err;
        }
        if (OBJ_length(asn1obj) == 0)
            EC_GROUP_set_asn1_flag(group, OPENSSL_EC_EXPLICIT_CURVE);

        ASN1_OBJECT_free(asn1obj);
    }
#else
    /*
     * Inside the FIPS module we do not support explicit curves anyway
     * so the above check is not necessary.
     *
     * Skipping it is also necessary because `OBJ_length()` and
     * `ASN1_OBJECT_free()` are not available within the FIPS module
     * boundaries.
     */
#endif

    ok = 1;
err:
    if (!ok) {
        EC_GROUP_free(group);
        group = NULL;
    }
    EC_POINT_free(P);
    BN_CTX_free(ctx);
    BN_free(p);
    BN_free(a);
    BN_free(b);
    BN_free(order);
    BN_free(x);
    BN_free(y);
    return group;
}

EC_GROUP *EC_GROUP_new_by_curve_name_ex(OSSL_LIB_CTX *libctx, const char *propq,
    int nid)
{
    EC_GROUP *ret = NULL;
    const ec_list_element *curve;

    if ((curve = ec_curve_nid2curve(nid)) == NULL
        || (ret = ec_group_new_from_data(libctx, propq, *curve)) == NULL) {
#ifndef FIPS_MODULE
        ERR_raise_data(ERR_LIB_EC, EC_R_UNKNOWN_GROUP,
            "name=%s", OBJ_nid2sn(nid));
#else
        ERR_raise(ERR_LIB_EC, EC_R_UNKNOWN_GROUP);
#endif
        return NULL;
    }

    return ret;
}

#ifndef FIPS_MODULE
EC_GROUP *EC_GROUP_new_by_curve_name(int nid)
{
    return EC_GROUP_new_by_curve_name_ex(NULL, NULL, nid);
}
#endif

size_t EC_get_builtin_curves(EC_builtin_curve *r, size_t nitems)
{
    size_t i, min;

    if (r == NULL || nitems == 0)
        return curve_list_length;

    min = nitems < curve_list_length ? nitems : curve_list_length;

    for (i = 0; i < min; i++) {
        r[i].nid = curve_list[i].nid;
        r[i].comment = curve_list[i].comment;
    }

    return curve_list_length;
}

const char *EC_curve_nid2nist(int nid)
{
    return ossl_ec_curve_nid2nist_int(nid);
}

int EC_curve_nist2nid(const char *name)
{
    return ossl_ec_curve_nist2nid_int(name);
}

#define NUM_BN_FIELDS 6
/*
 * Validates EC domain parameter data for known named curves.
 * This can be used when a curve is loaded explicitly (without a curve
 * name) or to validate that domain parameters have not been modified.
 *
 * Returns: The nid associated with the found named curve, or NID_undef
 *          if not found. If there was an error it returns -1.
 */
int ossl_ec_curve_nid_from_params(const EC_GROUP *group, BN_CTX *ctx)
{
    int ret = -1, nid, len, field_type, param_len;
    size_t i, seed_len;
    const unsigned char *seed, *params_seed, *params;
    unsigned char *param_bytes = NULL;
    const EC_CURVE_DATA *data;
    const EC_POINT *generator = NULL;
    const BIGNUM *cofactor = NULL;
    /* An array of BIGNUMs for (p, a, b, x, y, order) */
    BIGNUM *bn[NUM_BN_FIELDS] = { NULL, NULL, NULL, NULL, NULL, NULL };

    /* Use the optional named curve nid as a search field */
    nid = EC_GROUP_get_curve_name(group);
    field_type = EC_GROUP_get_field_type(group);
    seed_len = EC_GROUP_get_seed_len(group);
    seed = EC_GROUP_get0_seed(group);
    cofactor = EC_GROUP_get0_cofactor(group);

    BN_CTX_start(ctx);

    /*
     * The built-in curves contains data fields (p, a, b, x, y, order) that are
     * all zero-padded to be the same size. The size of the padding is
     * determined by either the number of bytes in the field modulus (p) or the
     * EC group order, whichever is larger.
     */
    param_len = BN_num_bytes(group->order);
    len = BN_num_bytes(group->field);
    if (len > param_len)
        param_len = len;

    /* Allocate space to store the padded data for (p, a, b, x, y, order)  */
    param_bytes = OPENSSL_malloc(param_len * NUM_BN_FIELDS);
    if (param_bytes == NULL)
        goto end;

    /* Create the bignums */
    for (i = 0; i < NUM_BN_FIELDS; ++i) {
        if ((bn[i] = BN_CTX_get(ctx)) == NULL)
            goto end;
    }
    /*
     * Fill in the bn array with the same values as the internal curves
     * i.e. the values are p, a, b, x, y, order.
     */
    /* Get p, a & b */
    if (!(EC_GROUP_get_curve(group, bn[0], bn[1], bn[2], ctx)
            && ((generator = EC_GROUP_get0_generator(group)) != NULL)
            /* Get x & y */
            && EC_POINT_get_affine_coordinates(group, generator, bn[3], bn[4], ctx)
            /* Get order */
            && EC_GROUP_get_order(group, bn[5], ctx)))
        goto end;

    /*
     * Convert the bignum array to bytes that are joined together to form
     * a single buffer that contains data for all fields.
     * (p, a, b, x, y, order) are all zero padded to be the same size.
     */
    for (i = 0; i < NUM_BN_FIELDS; ++i) {
        if (BN_bn2binpad(bn[i], &param_bytes[i * param_len], param_len) <= 0)
            goto end;
    }

    for (i = 0; i < curve_list_length; i++) {
        const ec_list_element curve = curve_list[i];

        data = curve.data;
        /* Get the raw order byte data */
        params_seed = (const unsigned char *)(data + 1); /* skip header */
        params = params_seed + data->seed_len;

        /* Look for unique fields in the fixed curve data */
        if (data->field_type == field_type
            && param_len == data->param_len
            && (nid <= 0 || nid == curve.nid)
            /* check the optional cofactor (ignore if its zero) */
            && (cofactor == NULL || BN_is_zero(cofactor)
                || BN_is_word(cofactor, (const BN_ULONG)curve.data->cofactor))
            /* Check the optional seed (ignore if its not set) */
            && (data->seed_len == 0 || seed_len == 0
                || ((size_t)data->seed_len == seed_len
                    && memcmp(params_seed, seed, seed_len) == 0))
            /* Check that the groups params match the built-in curve params */
            && memcmp(param_bytes, params, param_len * NUM_BN_FIELDS)
                == 0) {
            ret = curve.nid;
            goto end;
        }
    }
    /* Gets here if the group was not found */
    ret = NID_undef;
end:
    OPENSSL_free(param_bytes);
    BN_CTX_end(ctx);
    return ret;
}
