#!/usr/bin/env python3
#
# Copyright (c) 2017 Jon Turney
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#

#
# mkgitoliteconf - creates a gitolite conf file fragment from cygwin-pkg-maint
#

import argparse
import sys

from . import common_constants
from . import maintainers


#
# transform username to charset acceptable to gitolite
#


def transform_username(name):
    name = name.replace('.', '')
    name = name.replace(' ', '_')
    return name


#
#
#

def do_main(pkglist, file=sys.stdout):
    # Get per-package list of maintainers
    pkgs = maintainers.pkg_list(pkglist)

    # header
    print("# automatically generated by mkgitoliteconf", file=file)

    # global configuration
    print('', file=file)
    print('@leads = %s' % ' '.join(map(transform_username, common_constants.TRUSTEDMAINT.split('/'))), file=file)
    print('', file=file)
    print('repo @allpackages', file=file)
    print('    RW = @leads', file=file)
    print('# anyone can create, push, rewind or delete the \'playground\' branch', file=file)
    print('    RW+ playground$ = @all', file=file)
    print('    R  = @all', file=file)
    print('    R  = gitweb daemon', file=file)
    print('    option hook.post-receive = pkgbuild', file=file)
    print('    config core.sharedrepository = all', file=file)
    print('    config uploadpack.allowReachableSHA1InWant = true', file=file)
    print('    config receive.advertisePushOptions = true', file=file)
    print('# this rejects binary files over the size limit, text files of any size are still permitted', file=file)
    print('    - VREF/MAX_NEWBIN_SIZE/1024 = @all', file=file)
    print('# this checks for trees which contain more than one .cygport file', file=file)
    print('    - VREF/HIGHLANDER/cygport = @all', file=file)
    print('', file=file)

    # for each package
    for p in sorted(pkgs):
        pkg = pkgs[p]

        users = ' '.join(map(transform_username, [m for m in pkg.maintainers() if m != 'ORPHANED']))

        owner = ''
        if pkg.maintainers():
            owner = pkg.maintainers()[0]  # first named maintainer

        if pkg.is_orphaned():
            owner = 'ORPHANED'

        print("repo git/cygwin-packages/%s" % (p), file=file)
        print("C  = %s @leads" % (users), file=file)
        if users:
            print("RW master$ = %s" % (users), file=file)
            print("RW main$ = %s" % (users), file=file)  # symref for master (or vice versa)
            print("RW refs/tags/ = %s" % (users), file=file)

        # these gitconfig keys also need to be listed in gitolite's
        # GIT_CONFIG_KEYS setting to be permitted

        # gitweb info
        if owner:
            print("owner = %s" % (owner), file=file)
        print("desc = cygwin packaging for %s" % (p), file=file)

        # cgit info (see 'man cgitrc')
        if owner:
            print("config repo.owner = %s" % (owner), file=file)
        print("config repo.desc = cygwin packaging for %s" % (p), file=file)
        print("", file=file)

    print('# accumulate list of all package repos into allpackages group', file=file)
    for p in sorted(pkgs):
        print("@allpackages = git/cygwin-packages/%s" % (p), file=file)


#
#
#

def main():
    pkglist_default = common_constants.PKGMAINT

    parser = argparse.ArgumentParser(description='gitolite rules config generator')
    parser.add_argument('--pkglist', action='store', metavar='FILE', help="package maintainer list (default: " + pkglist_default + ")", default=pkglist_default)
    (args) = parser.parse_args()

    do_main(args.pkglist)

    return 0


#
#
#

if __name__ == "__main__":
    sys.exit(main())
