/*
 * ScanSSH - simple SSH version scanner V1.2
 *
 * Copyright 2000 Niels Provos <provos@citi.umich.edu>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/queue.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <errno.h>
#include <err.h>
#include <fcntl.h>
#include <netdb.h>
#include <pcap.h>
#include <unistd.h>

#include "exclude.h"
#include "pcapu.h"

#ifndef howmany
#define howmany(x,y)	(((x) + ((y) - 1)) / (y))
#endif

ssize_t atomicio(ssize_t (*)(), int, void *, size_t);
int ipv4toa(char *, size_t, void *);

/* Globals */
int usepcap;
pcap_t *pd;
int rndexclude = 1;

#define MAX_PROCESSES	30

int commands[MAX_PROCESSES];
int results[MAX_PROCESSES];
int alarmed;

#define SSHMAPVERSION	"SSH-1.0-SSH_Version_Mapper\n"
#define MAXITER		10
#define LONGWAIT	30
#define SHORTWAIT	2
#define CONNECTWAIT	5
#define SYNWAIT		1
#define SYNRETRIES	5
#define MAXBUF		255
#define MAXSYNQUEUESZ	256
#define MAXBURST	64

TAILQ_HEAD (queue_list, argument) readyqueue;
TAILQ_HEAD (syn_list, argument) synqueue;

struct argument {
	TAILQ_ENTRY (argument) a_next;
	struct timeval a_tv;
	sa_family_t a_type;
	int a_retry;		/* what a waste of memory */
	u_int32_t a_seqnr;
	union {
		struct in_addr ipv4;
	} ip;

#define a_ipv4 ip.ipv4
};

#define synlist_hasspace()	(synqueuesz < MAXSYNQUEUESZ)
#define synlist_empty()		(synqueuesz == 0)

int synqueuesz;

void
synlist_init(void)
{
	TAILQ_INIT(&synqueue);
	synqueuesz = 0;
}

int
synlist_insert(struct argument *arg)
{
	struct timeval tmp;

	if (!synlist_hasspace())
		return (-1);

	timerclear(&tmp);
	tmp.tv_sec += SYNWAIT;
	if (arg->a_retry > SYNRETRIES/2)
		tmp.tv_sec += SYNWAIT;

	gettimeofday(&arg->a_tv, NULL);
	timeradd(&tmp, &arg->a_tv, &arg->a_tv);
	TAILQ_INSERT_TAIL(&synqueue, arg, a_next);
	synqueuesz++;

	return (0);
}

void
synlist_remove(struct argument *arg)
{
	TAILQ_REMOVE(&synqueue, arg, a_next);
	synqueuesz--;
}

struct argument *
synlist_dequeue(void)
{
	struct argument *arg;

	arg = TAILQ_FIRST(&synqueue);
	if (arg == NULL)
		return (NULL);

	synlist_remove(arg);
	return (arg);
}

int
synlist_expired(void)
{
	struct argument *arg;
	struct timeval tv;

	arg = TAILQ_FIRST(&synqueue);
	if (arg == NULL)
		return (0);

	gettimeofday(&tv, NULL);

	return (timercmp(&tv, &arg->a_tv, >));
}

int
synlist_probe(struct argument *arg)
{
	int res = 0; 

        if (arg->a_type != AF_INET ||
	    (res = synprobe_send(arg->a_ipv4, &arg->a_seqnr)) == -1 ||
	    synlist_insert(arg) == -1)
		TAILQ_INSERT_TAIL(&readyqueue, arg, a_next);
	    
	return (res);
}

void
sigchld_handler(int sig)
{
        int save_errno = errno;
	int status;
	wait(&status);
        signal(SIGCHLD, sigchld_handler);
        errno = save_errno;
}

void
sigalrm_handler(int sig)
{
        int save_errno = errno;
	alarmed = 1;
        errno = save_errno;
}

int
scanhost(struct argument *arg, char *buf, size_t size)
{
	int res, sock, j;
	struct addrinfo hints, *ai;
	char firstline[255];
	char ntop[NI_MAXHOST];
	
	switch (arg->a_type) {
	case AF_INET:
		ipv4toa(ntop, sizeof(ntop), &arg->a_ipv4);
		break;
	default:
		strlcpy(buf, "<unsuppfamily>", size);
		return (-1);
	}

	sock = socket(arg->a_type, SOCK_STREAM, 0);
	if (sock == -1) {
		strlcpy(buf, "<socketcreate>", size);
		return (-1);
	}

	memset(&hints, 0, sizeof(hints));
        hints.ai_family = AF_UNSPEC;
        hints.ai_socktype = SOCK_STREAM;
        if (getaddrinfo(ntop, "22", &hints, &ai) != 0) {
		strlcpy(buf, "<getaddrinfo>", size);
		return (-1);
	}

	alarm(CONNECTWAIT);
	res = connect(sock, ai->ai_addr, ai->ai_addrlen);
	alarm(0);
	if (res == -1) {
		close(sock);
		switch (errno) {
		case ETIMEDOUT:
		case EINTR:
#ifdef __linux__
		case EHOSTUNREACH:
#endif
			strlcpy(buf, "<timeout>", size);
			return (-1);
		case ECONNREFUSED:
			strlcpy(buf, "<refused>", size);
			return (-1);
		case ENETUNREACH:
			strlcpy(buf, "<unreachable>", size);
			return (-1);
		default:
			snprintf(buf, size, "<%s>", strerror(errno));
			return (-1);
		}
	}
	
	/* Read other side\'s version identification. */
        for (j = 0; j < MAXITER; j++) {
		int i;
                for (i = 0; i < size - 1; i++) {
                        int len = -1;
			if (!j && !i)
				alarm (LONGWAIT);
			else
				alarm (SHORTWAIT);
			while ((len = read(sock, &buf[i], 1)) == -1) {
				if (len == -1 && errno == EINTR)
					break;
			}
			alarm(0);
                        if (len < 0 && !j) {
				close(sock);
				strlcpy(buf, "<timeout>", size);
				return (-1);
			} else if (len != 1 && !j) {
				close(sock);
                                strlcpy(buf, "<closed>", size);
				return (-1);
			} else if (len != 1) {
				j = MAXITER;
				break;
			}
			
                        if (buf[i] == '\r') {
                                buf[i] = 0;
                                continue;
                        }
                        if (buf[i] == '\n') {
                                buf[i] = 0;
                                break;
                        }
                }
                buf[size - 1] = 0;
                if (strncmp(buf, "SSH-", 4) == 0)
                        break;
		if (j == 0)
			strlcpy(firstline, buf, sizeof(firstline));
        }
	if (j >= MAXITER)
		strlcpy(buf, firstline, sizeof(firstline));
	else
		atomicio(write, sock, SSHMAPVERSION, sizeof(SSHMAPVERSION));
	close(sock);
	return (0);
}

void
waitforcommands(int readfd, int writefd)
{
	struct argument arg;
	char buf[200];
	char result[200];
	struct sigaction sa;

        /* block signals, get terminal modes and turn off echo */
        memset(&sa, 0, sizeof(sa));
        sa.sa_handler = sigalrm_handler;
        (void) sigaction(SIGALRM, &sa, NULL);

	while(atomicio(read, readfd, &arg, sizeof(arg))) {
		/* exit command */
		if (arg.a_ipv4.s_addr == 0)
			return;

		scanhost(&arg, result, sizeof(result));
		ipv4toa(buf, sizeof(buf), &arg.a_ipv4);
		strlcat(buf, " ", sizeof(buf));
		strlcat(buf, result, sizeof(buf));

		atomicio(write, writefd, buf, strlen(buf) + 1);
	}
}

void
printres(struct argument *exp, char *result)
{
	char ntop[NI_MAXHOST];
	ipv4toa(ntop, sizeof(ntop), &exp->a_ipv4);
	fprintf(stdout, "%s %s\n", ntop, result);
}

void
scanips(struct argument *args, u_int32_t entries)
{
	fd_set *readset, *writeset;
	int maxfd = 0, fdsetsz, counts = 0;
	struct timeval tv, ptv;
	int res, i;
	char buf[MAXBUF], *p;

	for (i = 0; i < MAX_PROCESSES; i++) {
		if (commands[i] > maxfd)
			maxfd = commands[i];
		if (results[i] > maxfd)
			maxfd = results[i];
	}
	fdsetsz = howmany(maxfd + 1, NFDBITS) * sizeof(fd_mask);
	if ((readset = malloc(fdsetsz)) == NULL)
		err(1, "malloc for readset");
	if ((writeset = malloc(fdsetsz)) == NULL)
		err(1, "malloc for readset");

	TAILQ_INIT(&readyqueue);
	synlist_init();
	timerclear(&ptv);
	while (entries || counts || !synlist_empty()) {
		memset(readset, 0, fdsetsz);
		memset(writeset, 0, fdsetsz);
		
		for (i = 0; i < MAX_PROCESSES; i++) {
			if (commands[i] >= 0)
				FD_SET(commands[i], writeset);
			if (results[i] >= 0)
				FD_SET(results[i], readset);
		}
		timerclear(&tv);
		tv.tv_usec = 500000;
		/* Wait in select until there is a connection. */
		if ((res = select(maxfd + 1, readset, writeset, NULL,
				  &tv)) == -1) {
			if (errno != EINTR)
				err(1, "select");
			continue;
		}

		/* Read back commands */
		for (i = 0; i < MAX_PROCESSES; i++) {
			if (results[i] < 0 || !FD_ISSET(results[i], readset))
				continue;

			res = -1;
			while (res == -1) {
				res = read(results[i], buf, sizeof(buf));
				if (res == -1 &&
				    (errno != EINTR && errno != EAGAIN))
					break;
				if (res != -1 && buf[res - 1] == '\0')
					break;
			}
			if (res == 0) {
				/* Child terminated */
				close (results[i]);
				results[i] = -1;
				if (commands[i] != -1) {
					close(commands[i]);
					commands[i] = -1;
				}
				continue;
			} else if(res == -1)
				err(1, "read");
			p = buf;
			while (res > 0) {
				fprintf(stdout, "%s\n", p);
				fflush(stdout);
				res -= strlen(p) + 1;
				p += strlen(p) + 1;
				counts--;
			}
		}

		if (usepcap) {
			int res, nsent;
			struct in_addr address;
			struct argument *addrchk;
			struct timeval now;

			/* See if we get results from our syn probe */
			while (!synlist_empty() &&
			       (res = pcap_get_address(pd, &address)) != -1) {
				TAILQ_FOREACH(addrchk, &synqueue, a_next) {
					if (address.s_addr == addrchk->a_ipv4.s_addr)
						break;
				}
				if (addrchk) {
					synlist_remove(addrchk);
					if (res == 0)
						printres(addrchk, "<refused>");
					else
						TAILQ_INSERT_TAIL(&readyqueue,
								  addrchk,
								  a_next);
				}
			}

			gettimeofday(&now, NULL);
			nsent = 0;
			if (timercmp(&now, &ptv, >)) {
				/* Remove all expired entries */
				while (synlist_expired() && nsent < MAXBURST) {
					struct argument *exp;

					exp = synlist_dequeue();
					if (exp->a_retry < SYNRETRIES) {
						exp->a_retry++;
						if (synlist_probe(exp) == -1)
							break;
						nsent++;
					} else
						printres(exp, "<timeout>");
				}
				while (entries && nsent < MAXBURST &&
				       synlist_hasspace()) {
					entries--;
					args[entries].a_retry = 0;
					if (synlist_probe(&args[entries])== -1)
						break;
					nsent++;
				}
				if (nsent == MAXBURST) {
					timerclear(&ptv);
					ptv.tv_usec = 300000;
					timeradd(&now, &ptv, &ptv);
				}
			}
		} else {
			/* No advanced syn probing, place data direcly
			 * on ready queue.
			 */
			for (i = 0; i < MAX_PROCESSES && entries; i++) {
				entries--;
				TAILQ_INSERT_TAIL(&readyqueue, &args[entries],
						  a_next);
			}
		}

		for (i = 0; i < MAX_PROCESSES; i++) {
			if (commands[i] < 0 ||
			    !FD_ISSET(commands[i], writeset))
				continue;
			if (TAILQ_FIRST(&readyqueue)) {
				struct argument *arg;
				arg = TAILQ_FIRST(&readyqueue);
				TAILQ_REMOVE(&readyqueue, arg, a_next);
				atomicio(write, commands[i], arg,
					 sizeof(struct argument));
				counts++;
			} else if (synlist_empty()) {
				struct argument arg;
				memset(&arg, 0, sizeof(arg));
				atomicio(write, commands[i], &arg,
					 sizeof(arg));
				close (commands[i]);
				commands[i] = -1;
			}
		}
	}
}

void
setupchildren(void)
{
	struct rlimit rlimit;
	int cpipefds[2], rpipefds[2];
	int i;
	pid_t pid;

        /* Increase number of open files */
        if (getrlimit(RLIMIT_NOFILE, &rlimit) == -1)
                err(1, "router_socket: getrlimit");
        rlimit.rlim_cur = rlimit.rlim_max;
        if (setrlimit(RLIMIT_NOFILE, &rlimit) == -1)
                err(1, "router_socket: setrlimit");

	/* Arrange SIGCHLD to be caught. */
	signal(SIGCHLD, sigchld_handler);

	for (i = 0; i < MAX_PROCESSES; i++) {
		if (pipe(cpipefds) == -1)
			err(1, "pipe for commands fds failed");
		if (pipe(rpipefds) == -1)
			err(1, "pipe for results fds failed");
		fcntl(cpipefds[1], F_SETFL, O_NONBLOCK);
		fcntl(rpipefds[0], F_SETFL, O_NONBLOCK);
		if ((pid = fork()) == 0) {
			/* Child */
			close(cpipefds[1]);
			close(rpipefds[0]);
			waitforcommands(cpipefds[0], rpipefds[1]);
			exit (0);
		} else if(pid == -1)
			err(1, "fork failed");
		
		/* Parent */
		close(cpipefds[0]);
		close(rpipefds[1]);

		commands[i] = cpipefds[1];
		results[i] = rpipefds[0];
	}

}

void
usage(char *name)
{
	fprintf(stderr, 
		"%s: [-ERh] [-e excludefile] [-p ifaddr] <IP address|network>...\n\n"
		"\t-E          do not exit if exclude file is missing,\n"
		"\t-R          do not honor exclude file for random addresses,\n"
		"\t-e <file>   exlude the IP addresses and networks in <file>,\n"
		"\t-p <ifaddr> specifies the local interface address,\n"
		"\t-h          this message.\n\n",
		name);
}

#ifndef HAVE_SOCKADDR_STORAGE
struct sockaddr_storage {
	u_char iamasuckyoperatingsystem[2048];
};
#endif

int
ipv4toa(char *buf, size_t size, void *addr)
{
	struct sockaddr_storage from;
	struct sockaddr_in *sfrom = (struct sockaddr_in *)&from;
	socklen_t fromlen = sizeof(from);

	memset(&from, 0, fromlen);
#ifdef HAVE_SIN_LEN
	sfrom->sin_len = sizeof (struct sockaddr_in);
#endif
	sfrom->sin_family = AF_INET;
	memcpy(&sfrom->sin_addr.s_addr, addr, 4);

	if (getnameinfo ((struct sockaddr *)sfrom,
#ifdef HAVE_SIN_LEN
			 sfrom->sin_len,
#else
			 sizeof (struct sockaddr_in),
#endif
			 buf, size, NULL, 0, NI_NUMERICHOST) != 0) {
		fprintf(stderr, "ipsec_ipv4toa: getnameinfo() failed");
		return -1;
	}
	return 0;
}

/*
 * Given an IP prefix and mask create all addresses contained
 * exlcuding any addresses specified in the exlcude queues.
 */

int
populate(struct argument **args, u_int32_t *entries, sa_family_t type,
	 void *address, int bits, int dorandom)
{
	struct in_addr ipv4addr, ipv4next, ipv4end;
	struct in_addr ipv4mask;
	struct argument *work;
	u_int32_t n, oldn, i;

	switch (type) {
	case AF_INET:
		ipv4addr = *(struct in_addr *)address;
		ipv4mask.s_addr = 0xFFFFFFFF << (32 - bits);
		ipv4addr.s_addr = ntohl(ipv4addr.s_addr);
		ipv4addr.s_addr &= ipv4mask.s_addr;
		ipv4end.s_addr = ipv4addr.s_addr | (~ipv4mask.s_addr);
		break;
		/* IPv6 should be here */
	default:
		fprintf(stderr, "Unknown address family: %d\n", type);
		return -1;
	}

	if (!dorandom) {
		n = 0;
		for (ipv4next = ipv4addr;
		     ipv4next.s_addr <= ipv4end.s_addr &&
			     ipv4next.s_addr != INADDR_ANY;
		     ipv4next.s_addr++, n++) {
			ipv4next = exclude(ipv4next, &excludequeue);
			if (ipv4next.s_addr > ipv4end.s_addr)
				break;
		}
		if (n < (1 << (32 - bits)))
			fprintf (stderr, "Excluding %d addresses\n",
				 (1 << (32 - bits)) - n);

		n += *entries;
	} else {
		struct in_addr tmp;

		rndsboxinit();
		n = 0;
		for (i = 0; i < dorandom; i++) {
			ipv4next = rndgetaddr(&ipv4addr, bits, i);
			tmp = exclude(ipv4next, &rndexclqueue);
			if (ipv4next.s_addr != tmp.s_addr)
				continue;
			if (rndexclude) {
				tmp = exclude(ipv4next, &excludequeue);
				if (ipv4next.s_addr != tmp.s_addr)
					continue;
			}

			n++;
		}
		if (n != dorandom)
			fprintf(stderr, "Excluding %d addresses\n",
				dorandom - n);
		n += *entries;
	}
	
	if (*args != NULL)
		work = realloc(*args, n * sizeof(struct argument));
	else
		work = malloc(n * sizeof(struct argument));
	if (work == NULL) {
		warn("{Re,m}alloc failed: %d", n * sizeof(struct argument));
		return -1;
	}
	
	oldn = *entries;
	memset(work + oldn, 0, (n - oldn) * sizeof(struct argument));

	if (!dorandom) {
		i = 0;
		for (; ipv4addr.s_addr <= ipv4end.s_addr &&
			     ipv4addr.s_addr != INADDR_ANY;
		     ipv4addr.s_addr++, i++) {
			ipv4addr = exclude(ipv4addr, &excludequeue); 
			if (ipv4addr.s_addr > ipv4end.s_addr)
				break;
			work[i + oldn].a_type = type;
			work[i + oldn].a_ipv4.s_addr = htonl(ipv4addr.s_addr);
		}
	} else {
		int count = 0;
		struct in_addr tmp;
		for (i = 0; i < dorandom; i++) {
			ipv4next = rndgetaddr(&ipv4addr, bits, i);
			tmp = exclude(ipv4next, &rndexclqueue);
			if (ipv4next.s_addr != tmp.s_addr)
				continue;
			if (rndexclude) {
				tmp = exclude(ipv4next, &excludequeue);
				if (ipv4next.s_addr != tmp.s_addr)
					continue;
			}
			
			work[count + oldn].a_type = type;
			work[count + oldn].a_ipv4.s_addr = htonl(ipv4next.s_addr);
			count++;
		}
	}

	*args = work;
	*entries = n;

	return 0;
}

int
mix(struct argument **args, u_int32_t n)
{
	struct argument *work, *oldargs;
	u_int32_t i, tmp;

	if ((work = malloc(n * sizeof(struct argument))) == NULL) {
		warn("malloc failed");
		return (-1);
	}

	oldargs = *args;
	for (i = 0; i < n; i++) {
		tmp = arc4random() % (n - i);
		work[i] = oldargs[tmp];
		oldargs[tmp] = oldargs[n - i - 1];
	}

	*args = work;
	free (oldargs);

	return (0);
}

int
main(int argc, char **argv)
{
	char *name, *src = NULL;
	int ch;
	u_int32_t entries = 0;
	struct argument *args = NULL;
	struct in_addr address;
	int bits, dorandom;
	int failonexclude = 1;

	name = argv[0];
	while ((ch = getopt(argc, argv, "hp:e:ER")) != -1)
		switch(ch) {
		case 'p':
			src = optarg;
			break;
		case 'e':
			excludefile = optarg;
			break;
		case 'E':
			failonexclude=0;
			break;
		case 'R':
			rndexclude=0;
			break;
		case 'h':
		default:
			usage(name);
			exit(1);
		}

	argc -= optind;
	argv += optind;

	/* With probe optimization */
	if (src) {
		synprobe_init(src);
		pd = pcap_filter_init("(tcp[13] & 18 = 18 or tcp[13] & 4 = 4) "
				      "and src port 22");
		usepcap = 1;
	} else
		usepcap = 0;


	/* revoke privs */
        seteuid(getuid());
        setuid(getuid());

	setupchildren();

	if (setupexcludes() == -1 && failonexclude)
		exit(1);

	while (argc) {
		if (sscanf(argv[0], "random(%d)/", &dorandom) == 1) {
			strsep(&argv[0], "/");
		} else
			dorandom = 0;
		if (parseaddress(argv[0], &address, &bits) == -1) {
			fprintf(stderr, "Can not parse %s\n", argv[0]);
			exit (1);
		}
		if (populate(&args, &entries, AF_INET, &address, bits,
			     dorandom) == -1)
			errx(1, "populate failed");

		argv++;
		argc--;
	}

	if (entries == 0)
		errx(1, "nothing to scan");

	if (mix(&args, entries) == -1)
		exit (1);

	scanips(args, entries);

	return (1);
}
